#include "mbed.h"
#include "device_configuration.h"


/*
 * Debug option
 */
#if 0
//Enable debug
#include <cstdio>
#define INFO(x, ...) std::printf(""x"\r\n", ##__VA_ARGS__); 
#define DBG(x, ...) std::printf("[dev_conf: DBG]"x"\r\n", ##__VA_ARGS__); 
#define WARN(x, ...) std::printf("[dev_conf: WARN]"x"\r\n", ##__VA_ARGS__); 
#define ERR(x, ...) std::printf("[dev_conf: ERR]"x"\r\n", ##__VA_ARGS__); 
#else
//Disable debug
#define INFO(x, ...)
#define DBG(x, ...) 
#define WARN(x, ...)
#define ERR(x, ...) 
#endif


/*
 * Variables for network configuration
 */
// Device configuration
uint16_t u16IpAddr[4], u16IpSubnet[4], u16IpGateway[4], u16MacAddr[3]; // 16-bits variables to be compatible with eeprom functions
char strIpAddr[16], strIpSubnet[16], strIpGateway[16], strMacAddr[20]; // RPC variables, converted from 16-bits u16ip_xxx
uint16_t u16DeviceConfiguredFlag = 0;  // flag indicates whether device has been configured (0xA5A5) or not
// TCP server/UDP
uint16_t u16LocalTcpServerPort = DEFAULT_LOCAL_TCP_SERVER_PORT;
uint16_t u16LocalUdpServerPort = DEFAULT_LOCAL_UDP_SERVER_PORT;
// TCP client mode, set parameters of the remote TCP server this device connects to.
// When enabled, this device will send its status to the server every transmit_time_period.
uint16_t u16RemoteTcpServerIpAddr[4]; // 16-bit variable to be compatible with eeprom functions
char strRemoteTcpServerIpAddr[16]; // RPC variable, converted from 16-bits u16server_ip_addr
uint16_t u16RemoteTcpServerPort; // 16-bit variable to be compatible with eeprom functions
uint16_t u16AutoTransmitFlag = 0, u16TransmitPeriod = DEFAULT_TRANSMIT_PERIOD; // auto transmit status, time period = 1s
// UDP client mode
uint16_t u16RemoteUdpServerIpAddr[4]; // 16-bit variable to be compatible with eeprom functions
char strRemoteUdpServerIpAddr[16]; // RPC variable, converted from 16-bits u16server_ip_addr
uint16_t u16RemoteUdpServerPort; // 16-bit variable to be compatible with eeprom functions
// enable modes
uint16_t u16EnableTcpServer = DEFAULT_ENABLE_FLAG_VALUE, u16EnableTcpClient = DEFAULT_DISABLE_FLAG_VALUE; // flags for enabling TCP server/client and UDP (UDP is always on for configuration)
uint16_t u16EnableUdpServer = DEFAULT_DISABLE_FLAG_VALUE, u16EnableUdpClient = DEFAULT_DISABLE_FLAG_VALUE;
// time period between UART data package before sending through Ethernet interface
uint16_t u16InterDataPeriod;
// special character to signal sending data
uint16_t u16SpecialChar;
// uart
uint16_t u16UartBaudrate, u16UartBits, u16UartStopBits, u16UartParity, u16UartHandshake;
// dhcp vs static
uint16_t u16EnableDHCP;

// extra, not actually in EEPROM
uint8_t u8IpAddr[4]; // keep device ip address in 8-bits
uint8_t u8MacAddr[6]; // keep mac in 8-bits
uint8_t u8RemoteTcpServerIpAddr[4]; // remote TCP server ip address in 8-bits
uint8_t u8RemoteUdpServerIpAddr[4]; // remote UDP server ip address in 8-bits


// erase page
void erase_device_configuration() {
    // erase first_run flag
    enableEEPROMWriting();
    disableEEPROMWriting();
}


/*!
 * Function to write module network configuration
 * @param <char *buf> configuration buffer
 * @note 4x16-bit IP address
 * @note 4x16-bit subnet
 * @note 4x16-bit gateway
 * @note 3x16-bit MAC
 * @note 16-bit TCP local port
 * @note 16-bit UDP local port
 */
void write_device_configuration(uint16_t* ip, uint16_t* subnet, uint16_t* gateway, uint16_t* mac,
            uint16_t tcp_port, uint16_t udp_port,
            uint16_t* remote_tcp_ip, uint16_t remote_tcp_port, uint16_t auto_transmit, uint16_t transmit_period,
            uint16_t* remote_udp_ip, uint16_t remote_udp_port,
            uint16_t enable_tcp_server, uint16_t enable_tcp_client, uint16_t enable_udp_server, uint16_t enable_udp_client,
            uint16_t inter_data_period, uint16_t special_char,
            uint16_t uart_baudrate, uint16_t uart_bits, uint16_t uart_stopbits, uint16_t uart_parity, uint16_t uart_handshake,
            uint16_t enable_dhcp) {
    // Write network configuration
    // 4-byte IP address + 4-byte subnet + 4-byte gateway + 3-byte MAC
    // + local TCP server port + local UDP port
    
    INFO("Saving network configuration for this device...");
    
    enableEEPROMWriting();
    
    // erase first_run flag
    writeEEPROMHalfWord(DEVICE_CONFIGURED_FLAG_POS, DEFAULT_ENABLE_FLAG_VALUE);
    
    // IP address
    writeEEPROMHalfWord(IP_ADDRESS_POS+2*0, ip[0]);
    writeEEPROMHalfWord(IP_ADDRESS_POS+2*1, ip[1]);
    writeEEPROMHalfWord(IP_ADDRESS_POS+2*2, ip[2]);
    writeEEPROMHalfWord(IP_ADDRESS_POS+2*3, ip[3]);
    
    // IP subnet
    writeEEPROMHalfWord(IP_SUBNET_POS+2*0, subnet[0]);
    writeEEPROMHalfWord(IP_SUBNET_POS+2*1, subnet[1]);
    writeEEPROMHalfWord(IP_SUBNET_POS+2*2, subnet[2]);
    writeEEPROMHalfWord(IP_SUBNET_POS+2*3, subnet[3]);
    
    // IP gateway
    writeEEPROMHalfWord(IP_GATEWAY_POS+2*0, gateway[0]);
    writeEEPROMHalfWord(IP_GATEWAY_POS+2*1, gateway[1]);
    writeEEPROMHalfWord(IP_GATEWAY_POS+2*2, gateway[2]);
    writeEEPROMHalfWord(IP_GATEWAY_POS+2*3, gateway[3]);
    
    // MAC address
    writeEEPROMHalfWord(MAC_ADDRESS_POS+2*0, mac[0]);
    writeEEPROMHalfWord(MAC_ADDRESS_POS+2*1, mac[1]);
    writeEEPROMHalfWord(MAC_ADDRESS_POS+2*2, mac[2]);
    
    // Ports
    writeEEPROMHalfWord(LOCAL_TCP_SERVER_PORT_POS, tcp_port);
    writeEEPROMHalfWord(LOCAL_UDP_SERVER_PORT_POS, udp_port);
    
    // Remote TCP server
    writeEEPROMHalfWord(REMOTE_TCP_SERVER_IP_ADDR_POS+2*0, remote_tcp_ip[0]);
    writeEEPROMHalfWord(REMOTE_TCP_SERVER_IP_ADDR_POS+2*1, remote_tcp_ip[1]);
    writeEEPROMHalfWord(REMOTE_TCP_SERVER_IP_ADDR_POS+2*2, remote_tcp_ip[2]);
    writeEEPROMHalfWord(REMOTE_TCP_SERVER_IP_ADDR_POS+2*3, remote_tcp_ip[3]);
    
    // Remote TCP server port
    writeEEPROMHalfWord(REMOTE_TCP_SERVER_PORT_POS, remote_tcp_port);
    
    // Auto transmit
    writeEEPROMHalfWord(AUTO_TRANSMIT_FLAG_POS, auto_transmit);
    writeEEPROMHalfWord(AUTO_TRANSMIT_TIME_PERIOD_POS, transmit_period);
    
    // Remote UDP server
    writeEEPROMHalfWord(REMOTE_UDP_SERVER_IP_ADDR_POS+2*0, remote_udp_ip[0]);
    writeEEPROMHalfWord(REMOTE_UDP_SERVER_IP_ADDR_POS+2*1, remote_udp_ip[1]);
    writeEEPROMHalfWord(REMOTE_UDP_SERVER_IP_ADDR_POS+2*2, remote_udp_ip[2]);
    writeEEPROMHalfWord(REMOTE_UDP_SERVER_IP_ADDR_POS+2*3, remote_udp_ip[3]);
    
    // Remote UDP server port
    writeEEPROMHalfWord(REMOTE_UDP_SERVER_PORT_POS, remote_udp_port);
    
    // Enable modes
    writeEEPROMHalfWord(ENABLE_TCP_SERVER_POS, enable_tcp_server);
    writeEEPROMHalfWord(ENABLE_TCP_CLIENT_POS, enable_tcp_client);
    writeEEPROMHalfWord(ENABLE_UDP_SERVER_POS, enable_udp_server);
    writeEEPROMHalfWord(ENABLE_UDP_CLIENT_POS, enable_udp_client);
    
    // Serial inter-data period
    writeEEPROMHalfWord(INTER_DATA_PERIOD_POS, inter_data_period);
    
    // Special char
    writeEEPROMHalfWord(SPECIAL_CHAR_POS, special_char);
    
    // UART
    writeEEPROMHalfWord(UART_BAUDRATE_POS, uart_baudrate);
    writeEEPROMHalfWord(UART_BITS_POS, uart_bits);
    writeEEPROMHalfWord(UART_STOPBITS_POS, uart_stopbits);
    writeEEPROMHalfWord(UART_PARITY_POS, uart_parity);
    writeEEPROMHalfWord(UART_HANDSHAKE_POS, uart_handshake);
    
    // DHCP vs Static
    writeEEPROMHalfWord(ENABLE_DHCP_POS, enable_dhcp);
    
    disableEEPROMWriting();
    
    INFO("Successful");
}

/*!
 * Function to load module network configuration
 */
void read_device_configuration(void) {
    mbed_mac_address((char *)u8MacAddr);
    
    INFO("Loading network configuration...");
    
    // check if configured
    u16DeviceConfiguredFlag = readEEPROMHalfWord(DEVICE_CONFIGURED_FLAG_POS);
    
    // if not first run, load network config
    if (u16DeviceConfiguredFlag == DEFAULT_ENABLE_FLAG_VALUE) {
        INFO("User settings");
        
        // IP address
        u16IpAddr[0] = readEEPROMHalfWord(IP_ADDRESS_POS+2*0);
        u16IpAddr[1] = readEEPROMHalfWord(IP_ADDRESS_POS+2*1);
        u16IpAddr[2] = readEEPROMHalfWord(IP_ADDRESS_POS+2*2);
        u16IpAddr[3] = readEEPROMHalfWord(IP_ADDRESS_POS+2*3);
        u8IpAddr[0] = (uint8_t)(u16IpAddr[0] & 0x00FF);
        u8IpAddr[1] = (uint8_t)(u16IpAddr[1] & 0x00FF);
        u8IpAddr[2] = (uint8_t)(u16IpAddr[2] & 0x00FF);
        u8IpAddr[3] = (uint8_t)(u16IpAddr[3] & 0x00FF);
        
        // IP subnet
        u16IpSubnet[0] = readEEPROMHalfWord(IP_SUBNET_POS+2*0);
        u16IpSubnet[1] = readEEPROMHalfWord(IP_SUBNET_POS+2*1);
        u16IpSubnet[2] = readEEPROMHalfWord(IP_SUBNET_POS+2*2);
        u16IpSubnet[3] = readEEPROMHalfWord(IP_SUBNET_POS+2*3);
        
        // IP gateway
        u16IpGateway[0] = readEEPROMHalfWord(IP_GATEWAY_POS+2*0);
        u16IpGateway[1] = readEEPROMHalfWord(IP_GATEWAY_POS+2*1);
        u16IpGateway[2] = readEEPROMHalfWord(IP_GATEWAY_POS+2*2);
        u16IpGateway[3] = readEEPROMHalfWord(IP_GATEWAY_POS+2*3);
        
        // MAC address
        u16MacAddr[0] = readEEPROMHalfWord(MAC_ADDRESS_POS+2*0);
        u16MacAddr[1] = readEEPROMHalfWord(MAC_ADDRESS_POS+2*1);
        u16MacAddr[2] = readEEPROMHalfWord(MAC_ADDRESS_POS+2*2);
        u8MacAddr[0] = DEFAULT_MAC0; u8MacAddr[1] = DEFAULT_MAC1; u8MacAddr[2] = DEFAULT_MAC2;
        u8MacAddr[3] = (uint8_t)(u16MacAddr[0] & 0x00FF);
        u8MacAddr[4] = (uint8_t)(u16MacAddr[1] & 0x00FF);
        u8MacAddr[5] = (uint8_t)(u16MacAddr[2] & 0x00FF);
        
        // Ports
        u16LocalTcpServerPort = readEEPROMHalfWord(LOCAL_TCP_SERVER_PORT_POS);
        u16LocalUdpServerPort = readEEPROMHalfWord(LOCAL_UDP_SERVER_PORT_POS);
        
        // Remote TCP server
        u16RemoteTcpServerIpAddr[0] = readEEPROMHalfWord(REMOTE_TCP_SERVER_IP_ADDR_POS+2*0);
        u16RemoteTcpServerIpAddr[1] = readEEPROMHalfWord(REMOTE_TCP_SERVER_IP_ADDR_POS+2*1);
        u16RemoteTcpServerIpAddr[2] = readEEPROMHalfWord(REMOTE_TCP_SERVER_IP_ADDR_POS+2*2);
        u16RemoteTcpServerIpAddr[3] = readEEPROMHalfWord(REMOTE_TCP_SERVER_IP_ADDR_POS+2*3);
        u8RemoteTcpServerIpAddr[0] = (uint8_t)(u16RemoteTcpServerIpAddr[0] & 0x00FF);
        u8RemoteTcpServerIpAddr[1] = (uint8_t)(u16RemoteTcpServerIpAddr[1] & 0x00FF);
        u8RemoteTcpServerIpAddr[2] = (uint8_t)(u16RemoteTcpServerIpAddr[2] & 0x00FF);
        u8RemoteTcpServerIpAddr[3] = (uint8_t)(u16RemoteTcpServerIpAddr[3] & 0x00FF);
        
        // Remote TCP server port
        u16RemoteTcpServerPort = readEEPROMHalfWord(REMOTE_TCP_SERVER_PORT_POS);
        
        // Auto transmit
        u16AutoTransmitFlag = readEEPROMHalfWord(AUTO_TRANSMIT_FLAG_POS);
        u16TransmitPeriod = readEEPROMHalfWord(AUTO_TRANSMIT_TIME_PERIOD_POS);
        
        // Remote UDP server
        u16RemoteUdpServerIpAddr[0] = readEEPROMHalfWord(REMOTE_UDP_SERVER_IP_ADDR_POS+2*0);
        u16RemoteUdpServerIpAddr[1] = readEEPROMHalfWord(REMOTE_UDP_SERVER_IP_ADDR_POS+2*1);
        u16RemoteUdpServerIpAddr[2] = readEEPROMHalfWord(REMOTE_UDP_SERVER_IP_ADDR_POS+2*2);
        u16RemoteUdpServerIpAddr[3] = readEEPROMHalfWord(REMOTE_UDP_SERVER_IP_ADDR_POS+2*3);
        u8RemoteUdpServerIpAddr[0] = (uint8_t)(u16RemoteUdpServerIpAddr[0] & 0x00FF);
        u8RemoteUdpServerIpAddr[1] = (uint8_t)(u16RemoteUdpServerIpAddr[1] & 0x00FF);
        u8RemoteUdpServerIpAddr[2] = (uint8_t)(u16RemoteUdpServerIpAddr[2] & 0x00FF);
        u8RemoteUdpServerIpAddr[3] = (uint8_t)(u16RemoteUdpServerIpAddr[3] & 0x00FF);
        
        // Remote UDP server port
        u16RemoteUdpServerPort = readEEPROMHalfWord(REMOTE_UDP_SERVER_PORT_POS);
        
        // Enable modes
        u16EnableTcpServer = readEEPROMHalfWord(ENABLE_TCP_SERVER_POS);
        u16EnableTcpClient = readEEPROMHalfWord(ENABLE_TCP_CLIENT_POS);
        u16EnableUdpServer = readEEPROMHalfWord(ENABLE_UDP_SERVER_POS);
        u16EnableUdpClient = readEEPROMHalfWord(ENABLE_UDP_CLIENT_POS);
        
        // Serial inter-data period
        u16InterDataPeriod = readEEPROMHalfWord(INTER_DATA_PERIOD_POS);
        // Special character
        u16SpecialChar = readEEPROMHalfWord(SPECIAL_CHAR_POS);
        
        // UART
        u16UartBaudrate = readEEPROMHalfWord(UART_BAUDRATE_POS);
        u16UartBits = readEEPROMHalfWord(UART_BITS_POS);
        u16UartStopBits = readEEPROMHalfWord(UART_STOPBITS_POS);
        u16UartParity = readEEPROMHalfWord(UART_PARITY_POS);
        u16UartHandshake = readEEPROMHalfWord(UART_HANDSHAKE_POS);
        
        // DHCP
        u16EnableDHCP = readEEPROMHalfWord(ENABLE_DHCP_POS);
        
        // convert to strings
        sprintf(strIpAddr, "%d.%d.%d.%d", u8IpAddr[0], u8IpAddr[1], u8IpAddr[2], u8IpAddr[3]);
        sprintf(strIpSubnet, "%d.%d.%d.%d", (uint8_t)u16IpSubnet[0], (uint8_t)u16IpSubnet[1], (uint8_t)u16IpSubnet[2], (uint8_t)u16IpSubnet[3]);
        sprintf(strIpGateway, "%d.%d.%d.%d", (uint8_t)u16IpGateway[0], (uint8_t)u16IpGateway[1], (uint8_t)u16IpGateway[2], (uint8_t)u16IpGateway[3]);
        sprintf(strRemoteTcpServerIpAddr, "%d.%d.%d.%d", u8RemoteTcpServerIpAddr[0], u8RemoteTcpServerIpAddr[1], u8RemoteTcpServerIpAddr[2], u8RemoteTcpServerIpAddr[3]);
        sprintf(strRemoteUdpServerIpAddr, "%d.%d.%d.%d", u8RemoteUdpServerIpAddr[0], u8RemoteUdpServerIpAddr[1], u8RemoteUdpServerIpAddr[2], u8RemoteUdpServerIpAddr[3]);
        sprintf(strMacAddr, "%.2x:%.2x:%.2x:%.2x:%.2x:%.2x", u8MacAddr[0], u8MacAddr[1], u8MacAddr[2], u8MacAddr[3], u8MacAddr[4], u8MacAddr[5]);
    }
    // if ip is not configured, use default addresses
    else {
        INFO("No user settings, load defaults");
        u8MacAddr[0] = DEFAULT_MAC0; u8MacAddr[1] = DEFAULT_MAC1; u8MacAddr[2] = DEFAULT_MAC2;
        u8MacAddr[3] = DEFAULT_MAC3; u8MacAddr[4] = DEFAULT_MAC4; u8MacAddr[5] = DEFAULT_MAC5; 
        sprintf(strIpAddr, DEFAULT_IP_ADDRESS);
        sprintf(strIpSubnet, DEFAULT_IP_SUBNET);
        sprintf(strIpGateway, DEFAULT_IP_GATEWAY);
        sprintf(strMacAddr, "%.2x:%.2x:%.2x:%.2x:%.2x:%.2x", u8MacAddr[0], u8MacAddr[1], u8MacAddr[2], u8MacAddr[3], u8MacAddr[4], u8MacAddr[5]);
        
        u16LocalTcpServerPort = DEFAULT_LOCAL_TCP_SERVER_PORT;
        u16LocalUdpServerPort = DEFAULT_LOCAL_UDP_SERVER_PORT;
        
        sprintf(strRemoteTcpServerIpAddr, DEFAULT_REMOTE_TCP_SERVER_IP);
        u16RemoteTcpServerPort = DEFAULT_REMOTE_TCP_SERVER_PORT;
        
        sprintf(strRemoteUdpServerIpAddr, DEFAULT_REMOTE_UDP_SERVER_IP);
        u16RemoteUdpServerPort = DEFAULT_REMOTE_UDP_SERVER_PORT;
        
        u16AutoTransmitFlag = DEFAULT_DISABLE_FLAG_VALUE;
        u16TransmitPeriod = DEFAULT_TRANSMIT_PERIOD;
        // mode
        u16EnableTcpServer = DEFAULT_ENABLE_FLAG_VALUE;
        u16EnableTcpClient = DEFAULT_DISABLE_FLAG_VALUE;
        u16EnableUdpServer = DEFAULT_DISABLE_FLAG_VALUE;
        u16EnableUdpClient = DEFAULT_DISABLE_FLAG_VALUE;
        // signals
        u16InterDataPeriod = DEFAULT_INTER_DATA_PERIOD;
        u16SpecialChar = DEFAULT_SPECIAL_CHARACTER;
        // uart
        u16UartBaudrate = DEFAULT_UART_BAUDRATE;
        u16UartBits = DEFAULT_UART_BITS;
        u16UartStopBits = DEFAULT_UART_STOPBITS;
        u16UartParity = DEFAULT_UART_PARITY;
        u16UartHandshake = DEFAULT_UART_HANDSHAKE;
        
        u16EnableDHCP = DEFAULT_DISABLE_FLAG_VALUE;
    }
    
    INFO("Successful");
    INFO("IP: %s", strIpAddr);
    INFO("MASK: %s", strIpSubnet);
    INFO("GW: %s", strIpGateway);
    INFO("TCP server local port: %d", u16LocalTcpServerPort);
    INFO("UDP server local port: %d", u16LocalUdpServerPort);
    
    INFO("Remote TCP server: %s", strRemoteTcpServerIpAddr);
    INFO("Remote TCP server port: %d", u16RemoteTcpServerPort);
    
    INFO("Remote UDP server: %s", strRemoteUdpServerIpAddr);
    INFO("Remote UDP server port: %d", u16RemoteUdpServerPort);
    
    INFO("Auto transmit: %s", (u16AutoTransmitFlag == DEFAULT_ENABLE_FLAG_VALUE) ? "enabled" : "disable");
    INFO("Period: %d", u16TransmitPeriod);
    
    INFO("TCP server: %s", (u16EnableTcpServer == DEFAULT_ENABLE_FLAG_VALUE) ? "enabled" : "disable");
    INFO("TCP client: %s", (u16EnableTcpClient == DEFAULT_ENABLE_FLAG_VALUE) ? "enabled" : "disable");
    INFO("UDP server: %s", (u16EnableUdpServer == DEFAULT_ENABLE_FLAG_VALUE) ? "enabled" : "disable");
    INFO("UDP client: %s", (u16EnableUdpClient == DEFAULT_ENABLE_FLAG_VALUE) ? "enabled" : "disable");
    
    INFO("Serial inter-data period: %d", u16InterDataPeriod);
    INFO("Special character: %.2x", u16SpecialChar);
    
    INFO("DHCP: %s", (u16EnableDHCP == DEFAULT_ENABLE_FLAG_VALUE) ? "enabled" : "disable");
}

void reset_default_device_configuration() {
    uint16_t ip[4] = {192,168,0,120};
    uint16_t subnet[4] = {255,255,255,0};
    uint16_t gateway[4] = {192,168,0,1};
    uint16_t mac[3] = {u8MacAddr[3],u8MacAddr[4],u8MacAddr[5]};
    
    uint16_t tcp_port = DEFAULT_LOCAL_TCP_SERVER_PORT;
    uint16_t udp_port = DEFAULT_LOCAL_UDP_SERVER_PORT;
    
    uint16_t remote_tcp_ip[4] = {192,168,0,2};
    uint16_t remote_tcp_port = DEFAULT_REMOTE_TCP_SERVER_PORT;
    
    uint16_t remote_udp_ip[4] = {192,168,0,2};
    uint16_t remote_udp_port = DEFAULT_REMOTE_UDP_SERVER_PORT;
    
    uint16_t auto_transmit = DEFAULT_DISABLE_FLAG_VALUE;
    uint16_t transmit_period = DEFAULT_TRANSMIT_PERIOD;
    
    uint16_t enable_tcp_server = DEFAULT_ENABLE_FLAG_VALUE;
    uint16_t enable_tcp_client = DEFAULT_DISABLE_FLAG_VALUE;
    uint16_t enable_udp_server = DEFAULT_DISABLE_FLAG_VALUE;
    uint16_t enable_udp_client = DEFAULT_DISABLE_FLAG_VALUE;
    
    uint16_t inter_data_period = DEFAULT_INTER_DATA_PERIOD;
    uint16_t special_char = DEFAULT_SPECIAL_CHARACTER;
    
    uint16_t uart_baudrate = DEFAULT_UART_BAUDRATE;
    uint16_t uart_bits = DEFAULT_UART_BITS;
    uint16_t uart_stopbits = DEFAULT_UART_STOPBITS;
    uint16_t uart_parity = DEFAULT_UART_PARITY;
    uint16_t uart_handshake = DEFAULT_UART_HANDSHAKE;
    
    uint16_t enable_dhcp = DEFAULT_DISABLE_FLAG_VALUE;
    
    // erase the device configured flag
    erase_device_configuration();
    
    // write new one
    write_device_configuration(ip, subnet, gateway, mac,
            tcp_port, udp_port,
            remote_tcp_ip, remote_tcp_port, auto_transmit, transmit_period,
            remote_udp_ip, remote_udp_port,
            enable_tcp_server, enable_tcp_client, enable_udp_server, enable_udp_client,
            inter_data_period, special_char,
            uart_baudrate, uart_bits, uart_stopbits, uart_parity, uart_handshake,
            enable_dhcp);
}
