/// @file Watchdog.cpp provides the interface to the Watchdog module
///
/// This provides basic Watchdog service for the mbed. You can configure
/// various timeout intervals that meet your system needs. Additionally,
/// it is possible to identify if the Watchdog was the cause of any 
/// system restart.
/// 
/// Adapted from Simon's Watchdog code from http://mbed.org/forum/mbed/topic/508/
///
/// @note Copyright &copy; 2011 by Smartware Computing, all rights reserved.
///     This software may be used to derive new software, as long as
///     this copyright statement remains in the source file.
/// @author David Smart
///
#include "mbed.h"
#include "Watchdog.h"


/// Watchdog gets instantiated at the module level
Watchdog::Watchdog() {
    wdreset = false;
}

/// Load timeout value in watchdog timer and enable
void Watchdog::Configure(int pr) {
    // http://www.st.com/web/en/resource/technical/document/reference_manual/CD00171190.pdf
    IWDG->KR  = 0x5555;                                   // enable write to PR, RLR
    IWDG->PR  = pr;                                        // Init prescaler, page 486 Reference Manual
    IWDG->RLR = 0xFFF;                                    // Init RLR
    IWDG->KR  = 0xAAAA;                                   // Reload the watchdog
    IWDG->KR  = 0xCCCC;
}

/// "Service", "kick" or "feed" the dog - reset the watchdog timer
/// by writing this required bit pattern
void Watchdog::Service() {
    IWDG->KR  = 0xAAAA;
}

/// get the flag to indicate if the watchdog causes the reset
bool Watchdog::WatchdogCausedReset() {
    if (RCC->CSR & (1<<29)) {
        wdreset = true;
        RCC->CSR |= (1<<24); // clear reset flag
    }
    return wdreset;
}


