#include "mbed.h"
#include "PinDetect.h"
#include "MMA8451Q.h"
#include "MAG3110.h"
#include "TSISensor.h"

#define MMA8451_I2C_ADDRESS (0x1d<<1)

// Declare output LEDs
DigitalOut ledgreen(PTD5);
DigitalOut ledred(PTE29);

// Declare USB serial connection
Serial pc(USBTX,USBRX);

// Declare timer interrupt
Ticker timerAcc;
Ticker timerMag;
Ticker timerLight; 
Ticker timerTouch;
Ticker timerADC;

// Declare pointer variables
float xAcc;
float yAcc;
float zAcc;
int xMag;
int yMag;
int zMag;
float xLight; 
float xTouch; 
float xADC;

// Sampling rates
float accRate = 0.1;
float magRate = 0.1;
float lightRate = 0.1;
float touchRate = 0.1;
float adcRate = 0.1;

// Enables
int accEn = 1;
int magEn = 1;
int lightEn = 1;
int touchEn = 1;
int adcEn = 1;

// Receiving Data
const int bufferSize = 255;
char buffer[bufferSize];
int index = 0;
int received = 0;
int readEn = 0;
float temp;

// Declare Accelerometer pins and I2C address
MMA8451Q acc(PTE25, PTE24, MMA8451_I2C_ADDRESS, 0, 0);
// Declare Magnetometer pins
MAG3110 mag(PTE25, PTE24);
// Declare touch sensor pin
TSISensor touch;
// Declare light sensor pin
AnalogIn light(PTE22);
// Declare analog input pin
AnalogIn adc(PTB0);

// Functions
void init();
void printData();
void receiveHandler();
void processCommand();
void accTime();
void magTime();
void lightTime();
void touchTime();
void adcTime();

void init()
{
    // Attach timerAcc
    pc.baud(9600);
    timerAcc.attach(&accTime, accRate);
    timerMag.attach(&magTime, magRate);
    timerLight.attach(&lightTime, lightRate);
    timerTouch.attach(&touchTime, touchRate);
    timerADC.attach(&adcTime, adcRate);
    pc.attach(&receiveHandler, Serial::RxIrq);
    ledred = 0; 
    ledgreen = 0;   
}

int main() 
{
    // Initialize
    init();

    while(1)
    {
        __disable_irq();
        printData();
        __enable_irq();
        wait(0.05);
        if(received == 1){
            __disable_irq();    
            processCommand();
            __enable_irq();
        }
        //ledgreen = !ledgreen;
    }
}

void printData()
{
    pc.printf("/%f/%f/%f/%d/%d/%d/%f/%f/%f/%.3f/%.3f/%.3f/%.3f/%.3f/%d/%d/%d/%d/%d/\r\n", 
        xAcc, yAcc, zAcc, xMag, yMag, zMag, xLight, xTouch, xADC, 
        accRate, magRate, lightRate, touchRate, adcRate, 
        accEn, magEn, lightEn, touchEn, adcEn);
        
}  

void receiveHandler()
{
    while (pc.readable()){
        temp = pc.getc();
        //begin reading if char is @
        if (temp == '@')
        {
            readEn = 1;
        }
        //stop reading if char is #
        else if (temp == '#' && readEn == 1)
        {
            readEn = 0;
            received = 1;
            //ledred = 1;
            index = 0;
            return;
        }
        // if read enable is on, then read in data
        else if (readEn == 1)
        {
            buffer[index] = temp;
            index++;
        }
    }
    return;
}

void processCommand()
{
    //pc.printf("%s\r\n", buffer);
    char* cmd_index;
    char* cmd_val;
    char* disStr = "ddd";
    char* enStr = "eee";
    // read in first value (0, 1, 2, or 3)
    cmd_index = strtok(buffer, "x");
    //pc.printf("%s\r\n", cmd_index);
    //read in second value (integer larger than 050)
    cmd_val = strtok(NULL, "x");
    //pc.printf("%s\r\n", cmd_val);
    // ddd means disable
    if (strcmp(cmd_val, disStr) == 0)
    {
        ledgreen = 1;
        switch(*cmd_index)
        {
            case '1':  
                timerAcc.detach();
                accEn = 0;
                break;
            case '2':  
                timerMag.detach();
                magEn = 0;
                break;
            case '3': 
                timerLight.detach();
                lightEn = 0;
                break;
            case '4':  
                timerTouch.detach();
                touchEn = 0;
                break;
            case '5':
                timerADC.detach();
                adcEn = 0;
                break;
            default:
                //pc.printf("incorrect input\r\n");          
                break;
        }
    }
    // eee means enable
    else if (strcmp(cmd_val, enStr) == 0)
    {
        switch(*cmd_index)
        {
            case '1':  
                timerAcc.attach(&accTime, accRate); 
                accEn = 1;
                break;
            case '2':  
                timerMag.attach(&magTime, magRate);
                magEn = 1;
                break;
            case '3': 
                timerLight.attach(&lightTime, lightRate);
                lightEn = 1;
                break;
            case '4':  
                timerTouch.attach(&touchTime, touchRate);
                touchEn = 1;
                break;
            case '5':
                timerADC.attach(&adcTime, adcRate);
                adcEn = 1;
                break;
            default:
                //pc.printf("incorrect input\r\n");          
                break;
        }
    }
    temp = strtod(cmd_val, NULL)/1000;
    if (temp > 0.001 && strlen(cmd_val) >= 3)
    {
        switch(*cmd_index)
        {
            case '1':  
                accRate = temp;
                timerAcc.detach();
                timerAcc.attach(&accTime, accRate); 
                break;
            case '2':  
                magRate = temp;
                timerMag.detach();
                timerMag.attach(&magTime, magRate);
                break;
            case '3': 
                lightRate = temp;
                timerLight.detach();
                timerLight.attach(&lightTime, lightRate);
                break;
            case '4':  
                touchRate = temp;
                timerTouch.detach();
                timerTouch.attach(&touchTime, touchRate);
                break;
            case '5':
                adcRate = temp;
                timerADC.detach();
                timerADC.attach(&adcTime, adcRate);
            default:
                //pc.printf("incorrect input\r\n");          
                break;
        }
    }
        
    received = 0;
    memset(buffer, 0, bufferSize);
    //pc.printf("%s\r\n", buffer);                
}

void accTime() 
{
    xAcc = abs(acc.getAccX());
    yAcc = abs(acc.getAccY());
    zAcc = abs(acc.getAccZ());
    ledred = !ledred;
}

void magTime()
{
    xMag = mag.getXVal();
    yMag = mag.getYVal();
    zMag = mag.getZVal();
    //ledred = !ledred;
}

void lightTime()
{
    xLight = 1 - light.read();
}

void touchTime()
{
    xTouch = 1 - touch.readPercentage();
}

void adcTime()
{
    xADC = adc.read();
}