#include "mbed.h"
#include "FatfsIjfwConfigurable.h"
#include "UartIsp.h"
#include "I2CSlave2.h"
#include "I2c2mem.h"

// Pin definitions
DigitalOut ledPow(dp13);                // Power indicator
DigitalOut ledCard(dp14);               // Memory card access indicator

DigitalOut resetIJ(dp25);               // IchigoJam Reset Pin
DigitalOut ispIJ(dp26);                 // IchigoJam ISP Pin
RawSerial serial(dp16, dp15);           // tx, rx
I2CSlave2 i2cs(dp5, dp27);              // sda, scl

SPI spi(dp2, dp1, dp6);                 // SPI for memory card access (mosi, miso, sck)
DigitalOut cs(dp4);                     // Memory card cs pin
FatfsIjfwConfigurable fsij(&spi, &cs);  // Fatfs for Memory card


// file name
const char* CONFIG_FILE = "_config.txt";
const char* BINFILE = "jam.bin";


void error() {
    while(1) {
        ledCard = 1; ledPow = 1;
        wait(0.25);
        ledCard = 0; ledPow = 0;
        wait(0.25);
    }
}


int main() {
    // LED
    ledPow = 1; ledCard = 0;
    
    // Serial
    serial.baud(115200);

    // Read Config File
    ledCard = 1;
    fsij.readConfigFile(CONFIG_FILE);
    ledCard = 0;

    // Firmware update if bin file is found
    ledCard = 1;
    if (fsij.checkFirmFile(BINFILE)) {
        // Sequence of Entering UartISP mode
        ispIJ = 0; resetIJ = 0;     // ISP=L, Reset=L
        wait_ms(100);
        ispIJ = 0; resetIJ = 1;     // ISP=L, Reset=H
        wait_ms(100);
        ispIJ = 1; resetIJ = 1;     // ISP=H, Reset=H
        wait_ms(100);

        UartIsp uartisp(&serial, &fsij);                            // UartISP
        UartIsp::ISP_RESULT ret = uartisp.runUartIsp(BINFILE);      // Run UartISP process
        if (ret != UartIsp::NOERROR) {
            error();
        }
    
        // reset MCU
        ispIJ = 1; resetIJ = 0;     // ISP=H, Reset=L
        wait_ms(100);
        ispIJ = 1; resetIJ = 1;     // ISP=H, Reset=H
        wait_ms(100);

        // reset IJFW
        //NVIC_SystemReset();
    }
    ledCard = 0;

    // Data conversion I2C to/from memory card
    //   address 0xA? -> EEPROM Emulation
    //           0x22 -> Writing/Reading data of the specific file
    //           0x24 -> Continuous writing mode 
    I2c2mem i2c2m(&i2cs, &fsij, &ledCard);
    i2cs.address(0xA0, 0x8E);       // Slave address, mask

    while(1) {
        int rcv = i2cs.receive();
        int addr = i2cs.addressFromMaster();

        switch (rcv) {
            case I2CSlave2::WriteAddressed:
                if (addr == 0xA0) {
                    i2c2m.writeProgram(0);
                } else if (addr == 0xA8) {
                    i2c2m.writeProgram(1);
                } else if (addr == 0x22) {  // I2C address #11 in IchigoJam
                    i2c2m.writeData();
                } else if (addr == 0x24) {  // I2C address #12 in IchigoJam
                    i2c2m.writeDataContinuous();
                } else {
                    i2cs.stop();
                }
                break;
            case I2CSlave2::ReadAddressed:
                if (addr == 0xA1) {
                    i2c2m.readProgram();
                } else if (addr == 0xA9) {
                    i2c2m.readProgram();
                } else if (addr == 0x23) {  // I2C address #11 in IchigoJam
                    i2c2m.readData();
                } else if (addr == 0x25) {  // I2C address #12 in IchigoJam
                    i2c2m.reset();
                } else {
                    i2cs.stop();
                }
                break;
            case I2CSlave2::WriteGeneral:
                i2cs.stop();
                break;
            default:
                break;
        }

    }
}
