/*
 * Weather Station - mbed Weather Platform
 * Copyright (c) 2011 Hiroshi Suga
 * Released under the MIT License: http://mbed.org/license/mit
 */

/** @file
 * @brief Weather Station
 */

#include "mbed.h"
#include "weather.h"
#include "SDFileSystem.h"
#ifdef USE_USB
#include "MSCFileSystem.h"
#endif
#include "ConfigFile.h"

extern struct Sensor offset;
static SDFileSystem sd(p5, p6, p7, p8, "sd");
static LocalFileSystem *local;
#ifdef USE_USB
static MSCFileSystem *usb;
#endif
static DigitalIn sd_ins(p27);
struct Config conf;
static char filename[30] = "";

// date time from filesystem
int fstime (char *d, time_t *tim) {
    DIR_t dir;
    FILINFO fno;
#if _USE_LFN
    static char lfn[_MAX_LFN + 1];
    fno.lfname = lfn;
    fno.lfsize = sizeof(lfn);
#endif
    DWORD fattime = 0, ft;
    struct tm t;

    if (f_opendir(&dir, "") == FR_OK) {
        for (;;) {
            if (f_readdir(&dir, &fno) != FR_OK) break;
            if (fno.fname[0] == 0) break;
            if (fno.fdate != 0) {
            printf("%s %d\r\n", fno.fname, fno.fdate);
                ft = (fno.fdate << 16) | fno.ftime;
                if (ft > fattime) {
                    fattime = ft;
                }
            }
        }
        if (fattime) {
            t.tm_year = ((fattime >> 25) & 0x7f) + 80;
            t.tm_mon = ((fattime >> 21) & 0x0f) - 1; 
            t.tm_mday = (fattime >> 16) & 0x1f;
            t.tm_hour = (fattime >> 11) & 0x1f;
            t.tm_min = (fattime >> 5) & 0x3f;
            t.tm_sec = (fattime << 1) & 0x3e;
            *tim = mktime(&t);
            return 0;
        }
    }
    return -1;
}

// write CSV file
int write_log (const char *buf) {
    FILE *fp;

    if (filename[0] == 0) return 0;
#ifdef USE_SD_INS
    if (sd_ins != 0) return -1;
#endif

    LED_FILE_ACT_ON;
    fp = fopen(filename, "a");
    if (fp) {
        fprintf(fp, buf);
        fprintf(fp, "\r\n");
        fclose(fp);
    } else {
        LED_FILE_ACT_OFF;
        return -1;
    }
    LED_FILE_ACT_OFF;
    return 0;
}

int init_log () {
    int seq = 0;
    char dir[8];
    FILE *fp;

    if (conf.filetype) {
        if (conf.filetype == 1) {
            strcpy(dir, "/sd/");
        } else
        if (conf.filetype == 2) {
#ifdef USE_USB
            usb = new MSCFileSystem("usb");
            strcpy(dir, "/usb/");
#endif
        }

        // seq num
        strcpy(filename, dir);
        strcat(filename, "weather.seq");
        LED_FILE_ACT_ON;
        fp = fopen(filename, "r");
        if (fp) {
            fscanf(fp, "%d", &seq);
            fclose(fp);
        }
        seq ++;

        // save seq num
        fp = fopen(filename, "w");
        if (fp) {
            fprintf(fp, "%d", seq);
            fclose(fp);
        }
        LED_FILE_ACT_OFF;

        // csv filename
        sprintf(filename, "%sw%05d.csv", dir, seq);
        pc.printf("CSV Filename: %s\r\n", filename);
    }
    return 0;
}

char* chop (char *s) {
    int i;

    for (i = strlen(s) - 1; i >= 0; i --) {
        if (s[i] == ' ' || s[i] == '\n' || s[i] == '\r') {
            s[i] = 0;
        } else {
            break;
        }
    }
    return s;
}

int init_conf () {
    ConfigFile cfg;
    int i;
    char buf[CF_VALUE_SIZE], key[CF_KEY_SIZE];
    int ip0, ip1, ip2, ip3;

    memset(&conf, 0, sizeof(conf));
    conf.interval = 60;
    strcpy(conf.csv_mesg, CSV_MESG);

#ifdef USE_SD_INS
    sd_ins.mode(PullUp);
#endif

    // open config
    LED_FILE_ACT_ON;
#ifdef USE_SD_INS
    if (sd_ins == 0 && cfg.read("/sd/" CONFIG_FILE)) {
#else
    if (cfg.read("/sd/" CONFIG_FILE)) {
#endif
        // from sd

        strcpy(conf.dir, "/sd/");
        LED_FILE_ACT_OFF;

    } else {
        local = new LocalFileSystem("local");
        if (cfg.read("/local/" CONFIG_FILE)) {
            // from usb

            strcpy(conf.dir, "/local/");
            LED_FILE_ACT_OFF;

        } else {
            // none
            LED_FILE_ACT_OFF;
            return -1;
        }
    }

    // load config
    if (cfg.getValue("INTERVAL", buf, sizeof_1(buf))) {
        conf.interval = atoi(buf);
    }
    if (cfg.getValue("BAUD", buf, sizeof_1(buf))) {
        conf.baud = atoi(buf);
    }

    if (cfg.getValue("FILE", buf, sizeof_1(buf))) {
        chop(buf);
        if (strcmp(buf, "SD") == 0) {
            conf.filetype = 1;
        } else
        if (strcmp(buf, "USB") == 0) {
            conf.filetype = 2;
        }
    }
    if (cfg.getValue("CSV_MESG", conf.csv_mesg, sizeof_1(conf.csv_mesg))) {
        chop(conf.csv_mesg);
    }

    if (cfg.getValue("IPADDRESS", buf, sizeof_1(buf))) {
        if (strncmp(buf, "DHCP", 4) == 0) {
            conf.ipaddr = IpAddr(255, 255, 255, 255);
        } else {
            sscanf(buf, "%d.%d.%d.%d", &ip0, &ip1, &ip2, &ip3);
            conf.ipaddr = IpAddr(ip0, ip1, ip2, ip3);
        }
    }
    if (cfg.getValue("NETMASK", buf, sizeof_1(buf))) {
        sscanf(buf, "%d.%d.%d.%d", &ip0, &ip1, &ip2, &ip3);
        conf.netmask = IpAddr(ip0, ip1, ip2, ip3);
    }
    if (cfg.getValue("GATEWAY", buf, sizeof_1(buf))) {
        sscanf(buf, "%d.%d.%d.%d", &ip0, &ip1, &ip2, &ip3);
        conf.gateway = IpAddr(ip0, ip1, ip2, ip3);
    }
    if (cfg.getValue("NAMESERVER", buf, sizeof_1(buf))) {
        sscanf(buf, "%d.%d.%d.%d", &ip0, &ip1, &ip2, &ip3);
        conf.nameserver = IpAddr(ip0, ip1, ip2, ip3);
    }

    cfg.getValue("NTPSERVER", conf.ntpserver, sizeof_1(conf.ntpserver));
    chop(conf.ntpserver);
    cfg.getValue("SNMP_COMMNAME", conf.snmp_commname, sizeof_1(conf.snmp_commname));
    chop(conf.snmp_commname);

    cfg.getValue("PACHUBE_APIKEY", conf.pachube_apikey, sizeof_1(conf.pachube_apikey));
    chop(conf.pachube_apikey);
    cfg.getValue("PACHUBE_FEEDID", conf.pachube_feedid, sizeof_1(conf.pachube_feedid));
    chop(conf.pachube_feedid);
    if (cfg.getValue("PACHUBE_MESG", conf.pachube_mesg, sizeof_1(conf.pachube_mesg))) {
        chop(conf.pachube_mesg);
    } else {
        strcpy(conf.pachube_mesg, PACHUBE_MESG);
    }

    cfg.getValue("TWITTER_USER", conf.twitter_user, sizeof_1(conf.twitter_user));
    chop(conf.twitter_user);
    cfg.getValue("TWITTER_PWD", conf.twitter_pwd, sizeof_1(conf.twitter_pwd));
    chop(conf.twitter_pwd);
    cfg.getValue("TWITTER_MESG", conf.twitter_mesg[0], sizeof_1(conf.twitter_mesg[0]));
    chop(conf.twitter_mesg[0]);
    for (i = 0; i < CF_TWITTER_NUM; i ++) {
        sprintf(key, "TWITTER_MESG[%d]", i);
        cfg.getValue(key, conf.twitter_mesg[i], sizeof_1(conf.twitter_mesg[i]));
        chop(conf.twitter_mesg[i]);
    }

    cfg.getValue("STATIONS_ID", conf.stations_id, sizeof_1(conf.stations_id));
    chop(conf.stations_id);
    cfg.getValue("STATIONS_PIN", conf.stations_pin, sizeof_1(conf.stations_pin));
    chop(conf.stations_pin);

    if (cfg.getValue("LCD_TYPE", buf, sizeof_1(buf))) {
        conf.lcdtype = (enum I2CLCDType)atoi(buf);
    } else {
        conf.lcdtype = LCD16x2;
    }
    if (cfg.getValue("LCD_CONF", buf, sizeof_1(buf))) {
        conf.lcdconf = (enum I2CLCDConfig)atoi(buf);
    } else {
        conf.lcdconf = LCDCFG_3V;
    }
    if (cfg.getValue("LCD_MESG", conf.lcd_mesg, sizeof_1(conf.lcd_mesg))) {
        chop(conf.lcd_mesg);
    }
    if (cfg.getValue("LEDDISP_MESG", conf.leddisp_mesg, sizeof_1(conf.leddisp_mesg))) {
        chop(conf.leddisp_mesg);
    }

    if (cfg.getValue("XBEE", buf, sizeof_1(buf))) {
        conf.xbeebaud = atoi(buf);
    }

    if (cfg.getValue("INPUT", buf, sizeof(buf))) {
        conf.inputtype = (enum eINPUTTYPE)atoi(buf);
    }

    memset(&offset, 0, sizeof(offset));
    if (cfg.getValue("OFFSET[P]", buf, sizeof_1(buf))) {
        offset.pres = atof(buf);
    }
    if (cfg.getValue("OFFSET[T]", buf, sizeof_1(buf))) {
        offset.temp = atof(buf);
    }
    if (cfg.getValue("OFFSET[H]", buf, sizeof_1(buf))) {
        offset.humi = atof(buf);
    }
    if (cfg.getValue("OFFSET[V]", buf, sizeof_1(buf))) {
        offset.vane = atof(buf);
    }

#ifdef USE_EMAIL
    cfg.getValue("SMTPSERVER", conf.smtphost, sizeof_1(conf.smtphost));
    chop(conf.smtphost);
    if (cfg.getValue("SMTPPORT", buf, sizeof_1(buf))) {
        conf.smtpport = atoi(buf);
    } else {
        conf.smtpport = 25;
    }
    cfg.getValue("SMTPUSER", conf.smtpuser, sizeof_1(conf.smtpuser));
    chop(conf.smtpuser);
    cfg.getValue("SMTPPWD", conf.smtppwd, sizeof_1(conf.smtppwd));
    chop(conf.smtppwd);
    cfg.getValue("MAIL_FROM", conf.mailfrom, sizeof_1(conf.mailfrom));
    chop(conf.mailfrom);
    for (i = 0; i < CF_MAIL_NUM; i ++) {
        snprintf(key, sizeof_1(key), "MAIL_MESG[%d]", i);
        cfg.getValue(key, conf.mailmesg[i], sizeof_1(conf.mailmesg[i]));
        chop(conf.mailmesg[i]);
        snprintf(key, sizeof_1(key), "MAIL_TO[%d]", i);
        cfg.getValue(key, conf.mailto[i], sizeof_1(conf.mailto[i]));
        chop(conf.mailto[i]);
    }
#endif

#ifdef DEBUG
    pc.printf("Configration: %d\r\n", cfg.getCount());
#endif
    return 0;
}
