/*
 * Weather Station - mbed Weather Platform
 * Copyright (c) 2011 Hiroshi Suga
 * Released under the MIT License: http://mbed.org/license/mit
 */

/** @file
 * @brief Weather Station
 */

#include "mbed.h"
#include "ConfigFile.h"
#include "EthernetNetIf.h"
#include "I2CLCD.h"

#undef DEBUG

#define PCB_LOT 0 // PCB version 0:i, 1:ro, 2:ha

#define REG_LIGHT 1 // k ohm
#define REG_UV 100 // k ohm
#define REG_MOIST 10 // k ohm

#define USE_IL // IL interpreter
#define USE_USB // USB memory
#define USE_DISPLAY // I2C LCD, LED display module
#define USE_HTTP // HTTP Client (Weather Stations, Pachube, Twitter)
#define USE_NTP // NTP Client
#undef USE_SNMP // SNMP Agent
#define USE_EMAIL // send mail

#if PCB_LOT >= 1
#define USE_3LED // 3 leds level meter
#define USE_SD_INS // SD card insert sw
#define USE_EXINOUT // 2 input, 2 output
#endif


#define CONFIG_FILE "weather.cfg"
#define IL_FILE "weather.asm"

#define CSV_MESG "%y-%m-%d %h:%i:%s,%.2P,%.2T,%.2H,%.2A,%.2V,%.2R,%.2L,%.2U,%.2M,%.2p"
#define PACHUBE_MESG "%.2P,%.2T,%.2H,%.2A,%.2R"

#define TIMEZONE 9 // hour

#define CF_KEY_SIZE 32
#define CF_VALUE_SIZE 128
#define FORMAT_STR_SIZE 140

#define CF_TWITTER_NUM 5
#define CF_MAIL_NUM 5
#define INPUT_NUM 2
#define OUTPUT_NUM 2

#define LED_FREQ 4 // x 0.1s
#define NET_TIMEOUT 15000 // ms

#define LED_BUSY_ON led1 = 0
#define LED_BUSY_OFF led1 = 1
#ifdef USE_3LED
#define LED_FILE_ACT_ON led_yellow = 1; led2 = 1
#define LED_FILE_ACT_OFF led_yellow = 0; led2 = 0
#else
#define LED_FILE_ACT_ON led2 = 1
#define LED_FILE_ACT_OFF led2 = 0
#endif
#define LED_NET_ACT_ON led_yk = 0
#define LED_NET_ACT_OFF led_yk = 1
#define LED_NET_G_ON led_gayk = 1; led_gkya = 0
#define LED_NET_Y_ON led_gayk = 0; led_gkya = 1
#define LED_NET_GY_OFF led_gayk = 0; led_gkya = 0

#define sizeof_1(x) (sizeof(x) - 1)
#define sizeof_len(x) (sizeof(x) - 1 - strlen(x))

enum eINPUTTYPE {
    INPUT_MOIST = 0,
    INPUT_FALL = 1,
    INPUT_RISE = 2,
    INPUT_CPM = 4,
    INPUT_EXINOUT = 8,
};

struct Sensor {
    float pres, temp, humi, light, anemo, vane, rain, uv, moist, temp2;
    time_t sec;
    int input[INPUT_NUM], output[OUTPUT_NUM];
};

struct Config {
    IpAddr ipaddr, netmask, gateway, nameserver;
    char dir[8];
    int interval, baud, xbeebaud, filetype;
    char filename[32];
    char csv_mesg[CF_VALUE_SIZE], pachube_mesg[CF_VALUE_SIZE];
    char ntpserver[40];
    char pachube_apikey[80], pachube_feedid[8];
    char twitter_user[40], twitter_pwd[40], twitter_mesg[CF_TWITTER_NUM][CF_VALUE_SIZE];
    char stations_id[8], stations_pin[40];
    char snmp_commname[32];
    char lcd_mesg[CF_VALUE_SIZE];
    enum I2CLCDType lcdtype;
    enum I2CLCDConfig lcdconf;
    char leddisp_mesg[CF_VALUE_SIZE];
    enum eINPUTTYPE inputtype;
#ifdef USE_EMAIL
    char smtphost[40], smtpuser[40], smtppwd[40];
    int smtpport;
    char mailfrom[40], mailmesg[CF_MAIL_NUM][40], mailto[CF_MAIL_NUM][40];
#endif
};

extern const char VERSION[];
extern Serial pc;
extern I2C i2c;
extern EthernetNetIf *eth;
extern PwmOut led1;
extern DigitalOut led2, led3, led4;
extern DigitalOut led_gayk, led_gkya, led_yk;
#ifdef USE_3LED
extern DigitalOut led_red, led_yellow;
#endif
extern struct Sensor sensor, offset, sensor_old;
extern struct Config conf;
extern volatile uint32_t locUpTime;
extern char csv[];


int init_conf ();
int init_log ();
int write_log (const char *buf);
int fstime (char *d, time_t *tim);
char* chop (char *s);

int init_net ();
void pool_net ();
int weatherstations ();
int pachube ();
int twitter (int num);
int email (int num);
int snmp_init (char *commname);
int ntp (char *hostname);

int init_sensor ();
int update_sensor ();
char *format_str (const char *fmt, char *buf, int len);

int init_ilip (char *dir);
void exec_ilip (int enable);
void pool_ilip ();

int init_display ();
void pool_display ();
void update_display ();
