/*
 * Weather Station - mbed Weather Platform
 * Copyright (c) 2011 Hiroshi Suga
 * Released under the MIT License: http://mbed.org/license/mit
 */

/** @file
 * @brief Weather Station
 */

#include "mbed.h"
#include <new>
#include "weather.h"
#include "EthernetPowerControl.h"

const char VERSION[] = "mbed Weather Platform 0.4.0 (C) 2011 Suga koubou";
Serial pc(USBTX, USBRX);
PwmOut led1(LED1);
DigitalOut led2(LED2), led3(LED3), led4(LED4);
#ifdef USE_3LED
DigitalOut led_red(p28), led_yellow(p23);
#endif
volatile uint32_t locUpTime = 0;
char csv[FORMAT_STR_SIZE];


void no_memory () {
    printf("panic: can't allocate to memory!\r\n");
    exit(-1);
}

// Interrupt 10Hz
void isr_timer () {

    // uptime
    locUpTime = locUpTime + 10;

    pool_ilip();
    pool_display();

    LED_NET_ACT_OFF;
}

void init () {

    init_conf();

    if (conf.baud) {
        pc.baud(conf.baud);
    }

    if (time(NULL) == -1) {
        time_t sec;
        if (! fstime(conf.dir, &sec)) {
            set_time(sec);
        }
    }

    init_sensor();
    init_ilip(conf.dir);
    init_log();
    init_display();

    if (init_net()) {
        pc.printf("Power down: ethernet\r\n");
        PHY_PowerDown();
    }

    pc.printf("Interval: %d sec.\r\n", conf.interval);
}

int main() {
    Timer timer;
    Ticker ticker;
    int ledlevel = 0, ledflg = 0;

    set_new_handler(no_memory); // new handler function

    LED_BUSY_ON;
    init();
    LED_BUSY_OFF;

    ticker.attach(&isr_timer, 0.1); // Interrupt 10Hz
    timer.start();
    while(1) {
        // main loop
        LED_BUSY_ON;
        pool_net();

        __disable_irq();
        update_sensor();
        __enable_irq();

        // create CSV
        format_str(conf.csv_mesg, csv, sizeof_1(csv));
        pc.printf(csv);
        pc.printf("\r\n");
        if (write_log(csv)) {
            pc.printf("error: can't open file.\r\n");
        }

        // I2C LCD or LED
        update_display();

        // in/out
        exec_ilip(1);

        LED_BUSY_OFF;

        while (locUpTime % 100 != 0) {
            pool_net();
            wait_ms(100);
        }

        // interval (wait)
        while (timer.read() < conf.interval) {
            ledlevel = ledlevel + (ledflg ? -5 : 5);
            if (ledlevel > 100) {
                ledlevel = 100;
                ledflg = 1;
            } else
            if (ledlevel < 0) {
                ledlevel = 0;
                ledflg = 0;
            }
            led1 = ledlevel / 100.0;
            
            pool_net();
            wait_ms(100);

            // in/out (timer) 1s
            if (locUpTime % 100 == 0) {
                exec_ilip(0);
            }

            // for debug
            if (pc.readable()) {
                int i;
                i = pc.getc();
                if (i == ' ') {
                    break;
                } else {
                    pc.printf("( %d )\r\n", (int)timer.read());
                }
            }
        }
        timer.reset();
    }
}
