/*
 * Weather Station - mbed Weather Platform
 * Copyright (c) 2011 Hiroshi Suga
 * Released under the MIT License: http://mbed.org/license/mit
 */

/** @file
 * @brief Weather Station
 */

#include "mbed.h"
#include "weather.h"
#ifdef USE_IL
#include "ILinterpreter.h"

static ILinterpreter ilip;
#endif
static Serial xbee(p13, p14);
static volatile int measure_flg = 0, exinout_flg = 0;
#ifdef USE_EXINOUT
static DigitalIn *exin1, *exin2;
static DigitalOut *exout1, *exout2;
#endif

#ifdef USE_IL
// input relay
float cb_input(char key, int keynum, eEXPRESSION exp, int old) {
    float value = 0;
    Sensor *s;

    if (old) {
        s = &sensor_old;
    } else {
        s = &sensor;
    }

    switch (key) {
    case 'A': // sensor
        switch (keynum) {
        case 0:
            value = s->pres;
            break;
        case 1:
            value = s->temp;
            break;
        case 2:
            value = s->humi;
            break;
        case 3:
            value = s->anemo;
            break;
        case 4:
            value = s->vane;
            break;
        case 5:
            value = s->rain;
            break;
        case 6:
            value = s->light;
            break;
        case 7:
            value = s->uv;
            break;
        case 8:
            value = s->moist;
            break;
        case 99:
            value = measure_flg;
            break;
        }
        break;
        
    case 'I': // input
        if (keynum >= INPUT_NUM) break;
        value = s->input[keynum];
        break;

    case 'Q': // output
        if (keynum >= OUTPUT_NUM) break;
        value = s->output[keynum];
        break;
    }

    return value;
}

// output relay
void cb_output(char key, int keynum, int reg, eMNEMONIC mne) {

#ifdef DEBUG
    pc.printf("OUT [%c %d] %d %d\r\n", key, keynum, reg, mne);
#endif
    pool_net();

    switch (key) {
    case 'Q': // output
        if (keynum >= OUTPUT_NUM) break;
        if (mne == MNE_OUT) {
            sensor.output[keynum] = reg;
        } else
        if (mne == MNE_SET && reg) {
            sensor.output[keynum] = 1;
        } else
        if (mne == MNE_RST && reg) {
            sensor.output[keynum] = 0;
        }
        break;

    case 'P': // Pachube
        if (mne == MNE_OUT && reg) {
            pachube();
        }
        break;

    case 'S': // Weather Stations
        if (mne == MNE_OUT && reg) {
            weatherstations();
        }
        break;

    case 'W': // Twitter
        if (mne == MNE_OUT && reg) {
            twitter(keynum);
        }
        break;

    case 'B': // XBee
        if (mne == MNE_OUT && reg) {
            xbee.printf(csv);
            xbee.printf("\r\n");
        }
        break;

    case 'E': // E-mail
        if (mne == MNE_OUT && reg) {
            email(keynum);
        }
        break;

    case 'x': // NTP
        if (mne == MNE_OUT && reg && conf.ntpserver[0]) {
            ntp(conf.ntpserver);
        }
        break;

    }

    pool_net();
}
#endif

void pool_ilip () {
#ifdef USE_IL
    ilip.pool();
#endif
}

// execute IL
void exec_ilip (int enable) {
    Sensor sensor_tmp;

    measure_flg = enable;

#ifdef USE_EXINOUT
    // set input
    if (exinout_flg) {
        sensor.input[0] = *exin1 ? 0 : 1;
        sensor.input[1] = *exin2 ? 0 : 1;
    }
#endif

    sensor_tmp = sensor;

#ifdef USE_IL
    // every time
    if (ilip.exec()) {
        printf("IL: out of stack\r\n");
    }
#else
    if (enable) {
        weatherstations();
        pachube(csv);
        email(0);
    }
#endif

#ifdef USE_EXINOUT
    // set output
    if (exinout_flg) {
        led3 = *exout1 = sensor.output[0];
        led4 = *exout2 = sensor.output[1];
    }
#endif

    sensor_old = sensor_tmp;
}

// init IL
int init_ilip (char *dir) {
    int r;
    char buf[40];

    if (conf.xbeebaud) {
        xbee.baud(conf.xbeebaud);
    }

#ifdef USE_EXINOUT
    if (conf.inputtype == INPUT_EXINOUT) {
        exout1 = new DigitalOut(p30);
        exout2 = new DigitalOut(p29);
        exin1 = new DigitalIn(p18);
        exin1->mode(PullUp);
        exin2 = new DigitalIn(p19);
        exin2->mode(PullUp);
        exinout_flg = 1;
    }
#endif

#ifdef USE_IL
    strcpy(buf, dir);
    strcat(buf, IL_FILE);
    r = ilip.load(buf);
    if (r > 0) {
        pc.printf("IL load: %d\r\n", r);
        ilip.attach(cb_input, cb_output);
    }
#endif

    return 0;
}
