/*
 * Weather Station - mbed Weather Platform
 * Copyright (c) 2011 Hiroshi Suga
 * Released under the MIT License: http://mbed.org/license/mit
 */

/** @file
 * @brief Weather Station
 */

#include "mbed.h"
#include "weather.h"
#include "EthernetNetIf.h"
#ifdef USE_NTP
#include "TinySNTP.h"
#endif
#ifdef USE_HTTP
#include "TinyHTTP.h"
#endif
#ifdef USE_EMAIL
#include "TinySMTP.h"
#endif
#include "TCPSocket.h"

EthernetNetIf *eth; 
DigitalOut led_gayk(p24),led_gkya(p25), led_yk(p26);
static DigitalIn eth_link(P1_25), eth_speed(P1_26);
static volatile int ethernet_flg = 0;

int weatherstations () {
#ifdef USE_HTTP
    Host host;
    char buf[FORMAT_STR_SIZE];
    char head[80];

    if (! ethernet_flg) return 0;
    if (! conf.stations_id[0] || ! conf.stations_pin[0]) {
        return -1;
    }

    LED_NET_ACT_ON;

    format_str("d0=%.2P&d1=%.2T&d2=%.2H&d3=%.2A&d4=%.2V&d5=%.2R&d6=%.2L&d7=%.2U&d8=%.2M&d9=%.2p", buf, sizeof_1(buf));
    snprintf(&buf[strlen(buf)], sizeof_len(buf),
      "&fcd=%s&pin=%s", conf.stations_id, conf.stations_pin);

#ifdef DEBUG
    pc.printf("S: %s %s\r\n%s\r\n", conf.stations_id, conf.stations_pin, buf);
#endif
//    http->resetRequestHeaders();
//    http->basicAuth(NULL, NULL);

    strncpy(head, "Content-type: application/x-www-form-urlencoded\r\n", sizeof_1(head));

    host.setName("weather.sugakoubou.com");
    host.setPort(HTTP_PORT);
    return httpRequest(METHOD_POST, &host, "/p", head, buf) == 200 ? 0 : -1;
#endif
}

int pachube () {
#ifdef USE_HTTP
    Host host;
    char buf[FORMAT_STR_SIZE];
    char uri[40], head[160];

    if (! ethernet_flg) return 0;
    if (! conf.pachube_apikey[0] || ! conf.pachube_feedid[0]) {
        return -1;
    }

    LED_NET_ACT_ON;
    // body
    format_str(conf.pachube_mesg, buf, sizeof_1(buf));
    // header
    snprintf(head, sizeof(head), "Content-type: text/csv\r\nX-PachubeApiKey: %s\r\n", conf.pachube_apikey);
    // uri
    snprintf(uri, sizeof(uri), "/v1/feeds/%s.csv?_method=put", conf.pachube_feedid);
#ifdef DEBUG
    pc.printf("P: %s %s %s\r\n", conf.pachube_apikey, conf.pachube_feedid, uri);
#endif

    host.setName("api.pachube.com");
    host.setPort(HTTP_PORT);
    return httpRequest(METHOD_POST, &host, uri, head, buf) == 200 ? 0 : -1;
#endif
}

int twitter (int num) {
#ifdef USE_HTTP
    Host host;
    char buf[FORMAT_STR_SIZE];
    char msg[FORMAT_STR_SIZE];
    char head[160];

    if (! ethernet_flg || num >= CF_TWITTER_NUM) return 0;
    if (! conf.twitter_user[0] || ! conf.twitter_pwd[0] || ! conf.twitter_mesg[num][0]) {
        return -1;
    }

    LED_NET_ACT_ON;
    // header
    createauth(conf.twitter_user, conf.twitter_pwd, head, sizeof(head));
    strncat(head, "Content-type: application/x-www-form-urlencoded\r\n", sizeof_len(head));
    // post data
    format_str(conf.twitter_mesg[num], msg, sizeof_1(msg));
    strcpy(buf, "status=");
    urlencode(msg, &buf[strlen(buf)], sizeof_len(buf));
#ifdef DEBUG
    pc.printf("T: %s %s %s\r\n%s\r\n", conf.twitter_user, conf.twitter_pwd, conf.twitter_mesg[num], buf);
#endif

    host.setName("api.supertweet.net");
    host.setPort(HTTP_PORT);
    return httpRequest(METHOD_POST, &host, "/1/statuses/update.xml", head, buf) == 200 ? 0 : -1;
#endif
}

int email (int num) {
#ifdef USE_EMAIL
    Host host;
    char buf[FORMAT_STR_SIZE];

    if (! ethernet_flg || num >= CF_MAIL_NUM) return 0;
    if (! conf.smtphost[0] || ! conf.mailfrom[0] ||
      ! conf.mailto[num][0] || ! conf.mailmesg[num][0]) {
        return -1;
    }

    LED_NET_ACT_ON;

    snprintf(buf, sizeof(buf), "From: %s\nTo: %s\nSubject: Message from weather\n\n", conf.mailfrom, conf.mailto[0]);
    format_str(conf.mailmesg[num], &buf[strlen(buf)], sizeof_len(buf));
#ifdef DEBUG
    pc.printf("SMTP %s %s %s\r\n%s\r\n", conf.smtphost, conf.mailfrom, conf.mailto[num], buf);
#endif

    host.setName(conf.smtphost);
    host.setPort(conf.smtpport);
    return sendmail(conf.mailfrom, conf.mailto[num], buf, &host, conf.smtpuser, conf.smtppwd);
#endif
}

int ntp (char *hostname) {
#ifdef USE_NTP
    time_t sec;

    if (! ethernet_flg) return 0;
    pc.printf("NTP: %s\r\n", hostname);

    if (! ntpdate(hostname, &sec)) {
        sec = sec + (60 * 60 * TIMEZONE);
        set_time(sec);
#ifdef DEBUG
        pc.printf("NTP success %s\r\n", ctime(&sec));
#endif
    } else {
        pc.printf("NTP failure\r\n");
        return -1;
    }

/*
//    NTPClient ntp;
    Host ntpserver;
    NTPResult ret;
    time_t sec;

    if (! ethernet_flg) return 0;
    pc.printf("NTP: %s\r\n", hostname);

    LED_NET_ACT_ON;
    ntpserver.setName(hostname);
    ntpserver.setPort(123);

    ret = ntp->setTime(ntpserver);
    if (ret == NTP_OK) {
        sec = time(NULL) + (60 * 60 * TIMEZONE);
        set_time(sec);
    }

    if (ret != NTP_OK) {
        pc.printf("NTP failure: %d\r\n", ret);
        return -1;
#ifdef DEBUG
    } else {
        pc.printf("NTP success %s\r\n", ctime(&sec));
#endif
    }
*/
#endif
    return 0;
}

int init_net () {
    EthernetErr ethErr;
    
    eth_link.mode(PullUp);
    eth_speed.mode(PullUp);

    if (conf.ipaddr[0]) {

        if (! eth_link) {
            LED_NET_G_ON;
        }
        LED_NET_ACT_ON;

        if (conf.ipaddr[0] == 255) {
            // dhcp ip address
            eth = new EthernetNetIf;
        } else {
            // static ip address
            eth = new EthernetNetIf(conf.ipaddr, conf.netmask, conf.gateway, conf.nameserver);
        }

        ethErr = eth->setup();
        if (ethErr) {
            // error
            LED_NET_Y_ON;
            return -1;
        }

        pc.printf("\r\nEthernet: %d.%d.%d.%d\r\n", eth->getIp()[0], eth->getIp()[1], eth->getIp()[2], eth->getIp()[3]);
        ethernet_flg = 1;
        pool_net();

#ifdef USE_HTTP
//        http = new HTTPClient;
#endif
//        ntp = new NTPClient;
    }

    if (conf.ntpserver[0]) {
        ntp(conf.ntpserver);
    }

    if (conf.snmp_commname[0]) {
        snmp_init(conf.snmp_commname);
    }

    return 0;
}

void pool_net () {

    if (ethernet_flg) {
        Net::poll();

        if (! eth_link) {
            LED_NET_G_ON;
        } else {
            LED_NET_GY_OFF;
        }
    }
}
