/* USBKeyboard.h */
/* USB device example: Standard keyboard */
/* Copyright (c) 2011 ARM Limited. All rights reserved. */

#ifndef USBKEYBOARD_H
#define USBKEYBOARD_H

#include "USBHID.h"
#include "Stream.h"

/* Modifiers */
enum MODIFIER_KEY
{
    KEY_CTRL = 1,
    KEY_SHIFT = 2,
    KEY_ALT = 4,
};


enum MEDIA_KEY
{
    KEY_NEXT_TRACK,     /*!< next Track Button */
    KEY_PREVIOUS_TRACK, /*!< Previous track Button */
    KEY_STOP,           /*!< Stop Button */
    KEY_PLAY_PAUSE,     /*!< Play/Pause Button */
    KEY_MUTE,           /*!< Mute Button */
    KEY_VOLUME_UP,      /*!< Volume Up Button */
    KEY_VOLUME_DOWN,    /*!< Volume Down Button */
};

enum FUNCTION_KEY
{
    KEY_F1 = 128,   /* F1 key */
    KEY_F2,         /* F2 key */
    KEY_F3,         /* F3 key */
    KEY_F4,         /* F4 key */
    KEY_F5,         /* F5 key */
    KEY_F6,         /* F6 key */
    KEY_F7,         /* F7 key */
    KEY_F8,         /* F8 key */
    KEY_F9,         /* F9 key */
    KEY_F10,        /* F10 key */
    KEY_F11,        /* F11 key */
    KEY_F12,        /* F12 key */
    KEY_PRINT_SCREEN,   /* Print Screen key */
    KEY_INSERT,         /* Insert key */
    KEY_HOME,           /* Home key */
    KEY_PAGE_UP,        /* Page Up key */
    KEY_PAGE_DOWN,      /* Page Down key */
};

/** 
 * USBKeyboard example
 * @code
 *
 * #include "mbed.h"
 * #include "USBKeyboard.h"
 *
 * USBKeyboard key;
 * 
 * int main(void)
 * {
 *   while (1)
 *   {
 *       key.printf("Hello World\r\n");
 *       wait(1);
 *   }
 * }
 *
 * @endcode
 */
class USBKeyboard: public USBHID, public Stream
{
    public:
    
        /**
        *   Constructor
        *
        * @param vendor_id Your vendor_id (default: 0x1234)
        * @param product_id Your product_id (default: 0x0001)
        * @param product_release Your preoduct_release (default: 0x0001)
        *
        */
        USBKeyboard(uint16_t vendor_id = 0x1234, uint16_t product_id = 0x0003, uint16_t product_release = 0x0001): USBHID(0, 0, vendor_id, product_id, product_release){};
        
        /**
        * To send a character defined by a modifier(CTRL, SHIFT, ALT) and the key 
        *
        * @code
        * //To send CTRL + s (save)
        *  keyboard.keyCode('s', KEY_CTRL);
        * @endcode
        *
        * @param modifier bit 0: KEY_CTRL, bit 1: KEY_SHIFT, bit 2: KEY_ALT (default: 0)
        * @param key character to send
        * @returns true if there is no error, false otherwise
        */
        bool keyCode(uint8_t key, uint8_t modifier = 0);
        
        /**
        * Send a character
        *
        * @param c character to be sent
        * @returns true if there is no error, false otherwise
        */
        virtual int _putc(int c);
        
        /**
        * Control media keys
        *
        * @param key media key pressed (KEY_NEXT_TRACK, KEY_PREVIOUS_TRACK, KEY_STOP, KEY_PLAY_PAUSE, KEY_MUTE, KEY_VOLUME_UP, KEY_VOLUME_DOWN)
        * @returns true if there is no error, false otherwise
        */
        bool mediaControl(MEDIA_KEY key);
        
        /*
        * To define the report descriptor. Warning: this method has to store the length of the report descriptor in reportLength.
        *
        * @returns pointer to the report descriptor
        */
        virtual uint8_t * reportDesc();
        
   private:
        //dummy otherwise it doesn,t compile (we must define all methods of an abstract class)
        virtual int _getc() { return -1;}
};

#endif


