#include "stdint.h"
#include "USBCDC.h"
#include "USBBusInterface.h"
#include "USBDevice.h"

void USBCDC::attach(void (*fptr)(char*, int)) {
    cdc_evt = fptr;
}

USBCDC::USBCDC(uint16_t vendor_id, uint16_t product_id, uint16_t product_release): USBDevice(vendor_id, product_id, product_release) {
    cdc_evt = NULL;
}

bool USBCDC::EPBULK_OUT_callback() {
    uint8_t buf[64];
    uint16_t len;

    read(EPBULK_OUT, buf, &len, MAX_PACKET_SIZE_EPBULK);

    if (cdc_evt) {
        cdc_evt((char*)buf, len);
    }

    // We reactivate the endpoint to receive next characters
    readStart(EPBULK_OUT, MAX_PACKET_SIZE_EPBULK);
    return true;
}

bool USBCDC::USBCallback_request () {
    bool success = false;
    CONTROL_TRANSFER *transfer = getTransferPtr();
    
//    DBG("USBCallback_request: type %x, request %x\r\n", transfer->setup.bmRequestType.Type, transfer->setup.bRequest);
    /* Process standard requests */
    if ((transfer->setup.bmRequestType.Type == STANDARD_TYPE))
    {
        switch (transfer->setup.bRequest)
        {
            default:
                break;
        }
    }

    if (transfer->setup.bmRequestType.Type == CLASS_TYPE)
    {
        switch (transfer->setup.bRequest)
        {
            case 0x20: // CDC set line coding
            case 0x22: // CDC set control line state
            case 0x23: // CDC Alt-B
//                DBG("val=%x idx=%x len=%x\r\n", transfer->setup.wValue, transfer->setup.wIndex, transfer->setup.wLength);
                success = true;
                break;
            case 0x21: // CDC get line coding
                static uint8_t LineConfig[] = {0x00, 0xc0, 0x01, 0x00, 0, 0, 8};
                transfer->remaining = 7;
                transfer->ptr = LineConfig;
                transfer->direction = DEVICE_TO_HOST;
                success = true;
                break;
            default:
                break;
        }
    }

    return success;
}

bool USBCDC::USBCallback_setConfiguration(uint8_t configuration) {
    // Called in ISR context
    // Set configuration. Return false if the
    // configuration is not supported.
    if (configuration != DEFAULT_CONFIGURATION) {
        return false;
    }

    addEndpoint(EPINT_IN, MAX_PACKET_SIZE_EPINT);

    // Configure endpoints > 0
    addEndpoint(EPBULK_IN, MAX_PACKET_SIZE_EPBULK);
    addEndpoint(EPBULK_OUT, MAX_PACKET_SIZE_EPBULK);

    // We activate the endpoint to be able to receive data
    readStart(EPBULK_OUT, MAX_PACKET_SIZE_EPBULK);

    return true;
}


uint8_t * USBCDC::stringIinterfaceDesc() {
    static uint8_t stringIinterfaceDescriptor[] = {
        14,                           //bLength
        STRING_DESCRIPTOR,              //bDescriptorType 0x03
        'C',0,'D',0,'C',0,'D',0,'E',0,'V',0   //bString iInterface - Audio
    };
    return stringIinterfaceDescriptor;
}

uint8_t * USBCDC::stringIproductDesc() {
    static uint8_t stringIproductDescriptor[] = {
        24,                                                       //bLength
        STRING_DESCRIPTOR,                                          //bDescriptorType 0x03
        'M',0,'b',0,'e',0,'d',0,' ',0,'D',0,'e',0,'v',0,'i',0,'c',0,'e',0 //bString iProduct - Mbed Audio
    };
    return stringIproductDescriptor;
}


uint8_t * USBCDC::configurationDesc() {
    static uint8_t configDescriptor[] = {
        // configuration descriptor
        0x09, 0x02, 67, 0x00, 0x02, 0x01, 0x00, 0xC0, 0x32,

        // CDC
        // control class interface
        0x09, 0x04, 0x00, 0x00, 0x01, 0x02, 0x02, 0x01, 0x00,
        // header functional descriptor
        0x05, 0x24, 0x00, 0x10, 0x01,
        // call management functional descriptor
        0x05, 0x24, 0x01, 0x01, 0x01,
        // ACM functional descriptor
        0x04, 0x24, 0x02, 0x02,
        // union functional descriptor
        0x05, 0x24, 0x06, 0x00, 0x01,
        // notification EP
        0x07, 0x05, 0x81, 0x03, 0x10, 0x00, 0x0A,
        // data class interface descriptor
        0x09, 0x04, 0x01, 0x00, 0x02, 0x0A, 0x00, 0x00, 0x00,
        // data EP OUT
        0x07, 0x05, 0x02, 0x02, 0x40, 0x00, 0x00,
        // data EP IN
        0x07, 0x05, 0x82, 0x02, 0x40, 0x00, 0x00,
    };
    return configDescriptor;
}

uint8_t * USBCDC::deviceDesc() {
    static uint8_t deviceDescriptor[] = {
        DEVICE_DESCRIPTOR_LENGTH,       /* bLength */
        DEVICE_DESCRIPTOR,              /* bDescriptorType */
        LSB(0x01),           /* bcdUSB (LSB) */
        MSB(0x01),           /* bcdUSB (MSB) */
        0x02,                           /* bDeviceClass */
        0x00,                           /* bDeviceSubClass */
        0x00,                           /* bDeviceprotocol */
        MAX_PACKET_SIZE_EP0,            /* bMaxPacketSize0 */
        LSB(VENDOR_ID),                 /* idVendor (LSB) */
        MSB(VENDOR_ID),                 /* idVendor (MSB) */
        LSB(PRODUCT_ID),                /* idProduct (LSB) */
        MSB(PRODUCT_ID),                /* idProduct (MSB) */
        LSB(PRODUCT_RELEASE),           /* bcdDevice (LSB) */
        MSB(PRODUCT_RELEASE),           /* bcdDevice (MSB) */
        STRING_OFFSET_IMANUFACTURER,    /* iManufacturer */
        STRING_OFFSET_IPRODUCT,         /* iProduct */
        STRING_OFFSET_ISERIAL,          /* iSerialNumber */
        0x01                            /* bNumConfigurations */
    };
    return deviceDescriptor;
}
