/* mbed SpiOLED Library, for a spi LCD based on Winstar WEH001602 series
 *   Originl: http://mbed.org/users/simon/libraries/TextLCD/latest
 *   Modified by Suga
 */
/* mbed TextLCD Library, for a 4-bit LCD based on HD44780
 * Copyright (c) 2007-2010, sford, http://mbed.org
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "SpiOLED.h"
#include "mbed.h"

SpiOLED::SpiOLED(PinName mosi, PinName miso, PinName sclk, PinName cs, LCDType type) :
        _spi(mosi, miso, sclk), _cs(cs), _type(type) {

    _cs = 1;
    _spi.format(10, 3);

    wait_ms(100);        // Wait 15ms to ensure powered up

    writeCommand(0x38); // Function set 001 BW N F - -
    writeCommand(0x0C); // 0000 1 D C B
    writeCommand(0x06); // Entry mode : 0000 01 I/D S
    writeCommand(0x10); // Cursor and Display Shift 0001 S/C R/L 0 0
    writeCommand(0x17); // DCDC on
    wait_ms(10);
    writeCommand(0x02); // Home
    cls();
}

void SpiOLED::character(int column, int row, int c) {
    int a = address(column, row);
    writeCommand(a);
    writeData(c);
}

void SpiOLED::cls() {
    writeCommand(0x01); // cls, and set cursor to 0
    wait_ms(10);     // This command takes 1.64 ms
    locate(0, 0);
}

void SpiOLED::locate(int column, int row) {
    _column = column;
    _row = row;
}

int SpiOLED::_putc(int value) {
    if (value == '\n') {
        _column = 0;
        _row++;
        if (_row >= rows()) {
            _row = 0;
        }
    } else {
        character(_column, _row, value);
        _column++;
        if (_column >= columns()) {
            _column = 0;
            _row++;
            if (_row >= rows()) {
                _row = 0;
            }
        }
    }
    return value;
}

int SpiOLED::_getc() {
    return -1;
}

void SpiOLED::writeByte(int value, int rs) {
    int dat;

    dat = value & 0xff;
    if (rs) dat |= 0x200;
    _spi.write(dat);
}

void SpiOLED::writeCommand(int command) {
    _cs = 0;
    wait_us(1);
    writeByte(command, 0);
    _cs = 1;
}

void SpiOLED::writeData(int data) {
    _cs = 0;
    wait_us(1);
    writeByte(data, 1);
    _cs = 1;
}

int SpiOLED::isBusy() {
    int dat;

    _cs = 0;
    wait_us(1);
    _spi.write(0x100); // rs=0, rw=1
    dat = _spi.write(0x100);
    _cs = 1;
    dat = dat >> 2;
    return dat & 0x80;
}

int SpiOLED::address(int column, int row) {
    switch (_type) {
        case LCD20x4:
            switch (row) {
                case 0:
                    return 0x80 + column;
                case 1:
                    return 0xc0 + column;
                case 2:
                    return 0x94 + column;
                case 3:
                    return 0xd4 + column;
            }
        case LCD16x2B:
            return 0x80 + (row * 40) + column;
        case LCD16x2:
        case LCD20x2:
        case LCD8x2:
        default:
            return 0x80 + (row * 0x40) + column;
    }
}

int SpiOLED::columns() {
    switch (_type) {
        case LCD8x2:
            return 8;
        case LCD20x4:
        case LCD20x2:
            return 20;
        case LCD16x2:
        case LCD16x2B:
        default:
            return 16;
    }
}

int SpiOLED::rows() {
    switch (_type) {
        case LCD20x4:
            return 4;
        case LCD16x2:
        case LCD16x2B:
        case LCD20x2:
        case LCD8x2:
        default:
            return 2;
    }
}

void SpiOLED::cursor (bool flg) {
    if (flg) {
        writeCommand(0x0F); // 0000 1 D C B
    } else {
        writeCommand(0x0C); // 0000 1 D C B
    }
}
