/*
 * Serial RAM (SPI SRAM) library
 * Copyright (c) 2013 Hiroshi Suga
 * Released under the MIT License: http://mbed.org/license/mit
 */

/** @file
 * @brief Serial RAM (SPI SRAM) library
 *   23K256, 23LC1024 (Microchip)
 *   support FIFO
 *   support DMA need http://mbed.org/users/AjK/code/MODDMA/
 */

#ifndef __SerRAM_h__
#define __SerRAM_h__

#if defined(TARGET_LPC1768) || defined(TARGET_LPC2368)
#define RAM_USE_DMA
#define RAM_DMA_SIZE 64
#endif

#include "mbed.h"

#ifdef RAM_USE_DMA
#include "MODDMA.h"
#endif

class SerRAM {
public:
    /**
     * @param mosi MOSI port (p5, p10)
     * @param miso MISO port (p6, p11)
     * @param sck SCK port (p7, p12)
     * @param cs CS port
     * @param size Memory size (Kbits) default: 256
     */
    SerRAM (PinName mosi, PinName miso, PinName sck, PinName cs, int size = 256);
    SerRAM (SPI& spi, PinName cs, int size = 256);

    /**
     * @param addr address
     * @param dat data
     * @return 0:success, -1:failure
     */
    int write (int addr, int dat);
    /**
     * @param addr address
     * @param buf buffer
     * @param len length
     * @param async block (DMA)
     * @return >=0:success, -1:failure
     */
    int write (int addr, char *buf, int len, bool async = false);
    /**
     * @param addr address
     * @return data
     */
    int read (int addr);
    /**
     * @param addr address
     * @param buf buffer
     * @param len length
     * @param async block (DMA)
     * @return 0:success, -1:failure
     */
    int read (int addr, char *buf, int len, bool async = false);

    int setStatus (int status);
    int getStatus ();

    static SerRAM * getInstance() {
        return _inst;
    };

#ifdef RAM_USE_DMA
    bool isBusy () {
        return dmabusy;
    };
#endif

private:
    SPI _spi;
    DigitalOut _cs;
    int _size;
    int _alloc;
    static SerRAM * _inst;

#ifdef RAM_USE_DMA
    void tc0_callback ();
    void tc1_callback ();
    void err_callback ();

    LPC_SSP_TypeDef *_ssp;
    MODDMA dma;
    MODDMA_Config *dmacfg0, *dmacfg1;
    MODDMA::GPDMA_CONNECTION dmacon0, dmacon1;
    volatile bool dmabusy;
#endif
};

#endif
