/*
 * SPI PRAM NP8P128A13TSM60E (Micron)
 *   128Mbit
 */
#include "mbed.h"

#define CMD_WREN    0x06    // Write enable
#define CMD_WRDI    0x04    // Write disable
#define CMD_RDID    0x9f    // Read identification 
#define CMD_RDSR    0x05    // Read status register
#define CMD_WRSR    0x01    // Write status registe
#define CMD_READ    0x03    // Read data bytes
#define CMD_FREAD   0x0b    // Read data bytes at higher speed
#define CMD_PP      0x02    // Page program (legacy program)
#define CMD_PP_BA   0x22    // Page program (bit-alterable write)
#define CMD_PP_1S   0xd1    // Page program (On all 1s)
#define CMD_SE      0xd8    // Sector erase

DigitalOut myled(LED1);
Serial pc(USBTX, USBRX);

SPI spi(p11, p12, p13); // mosi, miso, sclk
DigitalOut cs(p14), hold(p15), reset(p16);

int ram_status () {
    int r;

    cs = 0;
    spi.write(CMD_RDSR);
    r = spi.write(0);
    cs = 1;
    return r;
}

int ram_write (int addr, char *buf, int len) {
    int i;

    while (ram_status() & 1) {
        // write in progress
    }

    cs = 0;
    spi.write(CMD_WREN);
    cs = 1;
    wait_us(10);

    cs = 0;
    spi.write(CMD_PP_BA);
    spi.write((addr >> 16) & 0xff);
    spi.write((addr >> 8) & 0xff);
    spi.write(addr & 0xff);

    len = len - (addr & 0x3f);
    for (i = 0; i < len; i ++) {
        spi.write(buf[i]);
    }
    cs = 1;
    return i;
}

int ram_read (int addr, char *buf, int len) {
    int i;

    cs = 0;
    spi.write(CMD_READ);
    spi.write((addr >> 16) & 0xff);
    spi.write((addr >> 8) & 0xff);
    spi.write(addr & 0xff);

    for (i = 0; i < len; i ++) {
        buf[i] = spi.write(0);
    }
    cs = 1;
    return i;
}

int main() {
    int i, j;
    char buf[64];
    Timer t;

    cs = 1;
    hold = 1;
    reset = 0;
    pc.baud(115200);
    spi.frequency(16000000);
    wait_ms(10);
    reset = 1;
    wait_ms(500);
    
    cs = 0;
    spi.write(CMD_RDID);
    printf("RAM Manufacturer ID : %02x\r\n", spi.write(0));
    i = (spi.write(0) << 8) | spi.write(0);
    printf("RAM Device ID : %04x\r\n", i);
    cs = 1;
    wait_ms(10);

    printf("Status Register : %02x\r\n", ram_status());

    printf("\r\nHELLO test\r\n");
    
    printf("RAM write\r\n");
    strcpy(buf, "Hello!");
    ram_write(0, buf, 6);
    
    for (i = 0; i < 64; i ++) {
        buf[i] = i;
    }
    ram_write(6, buf, 64 - 6);

    wait(1);
    memset(buf, 0, 64);
    
    printf("RAM read\r\n");
    ram_read(0, buf, 64);
    for (i = 0; i < 64; i ++) {
        printf(" %02x", buf[i]);
        if ((i & 0x0f) == 0x0f)
            printf("\r\n");
    }

    wait(1);

    printf("\r\nWrite/Read time\r\n");

    printf("RAM write\r\n");
    t.reset();
    t.start();
    for (i = 0; i < 0x1000000; i += 64) {
        buf[0] = (i >> 6) & 0xff;
        ram_write(i, buf, 64);
        if ((i & 0x0ffff) == 0) printf(".");
    }
    t.stop();
    printf("\r\ntime %f, %f KBytes/sec\r\n", t.read(), (float)0x1000000 / 1024 / t.read());

    wait(1);

    printf("RAM read\r\n");
    t.reset();
    t.start();
    for (i = 0; i < 0x1000000; i += 64) {
        ram_read(i, buf, 64);
        if (buf[0] != ((i >> 6) & 0xff)) {
            printf("error %d\r\n", i);
            break;
        }
        if ((i & 0x0ffff) == 0) printf(".");
    }
    t.stop();
    printf("\r\ntime %f, %f KBytes/sec\r\n", t.read(), (float)0x1000000 / 1024 / t.read());

}
