/*
 * Memory File System Library, for SPI PRAM NP8P128A13 (Micron)
 * Copyright (c) 2012 Hiroshi Suga
 * Released under the MIT License: http://mbed.org/license/mit
 */

#include "mbed.h"
#include "PRAMFileSystem.h"

PRAMFileSystem::PRAMFileSystem (PinName mosi, PinName miso, PinName sclk, PinName cs, const char* name) :
  FATFileSystem(name), _spi(mosi, miso, sclk), _cs(cs) {
    _cs = 1;
}

int PRAMFileSystem::_init () {
    int i;

    _spi.frequency(16000000);
    _cs = 1;
    wait_ms(10);

    _cs = 0;
    _spi.write(CMD_RDID);
    i = _spi.write(0);
    i = (_spi.write(0) << 8) | _spi.write(0);
    _cs = 1;
    if (i == 0 || i == 0xffff) {
        return -1;
    }
 
    return 0;
}

int PRAMFileSystem::disk_initialize() {
    if (_init()) {
        return 1;
    }
    _sectors = 128 * 1024 * 1024 / 8 / 512;
    return 0;
}

int PRAMFileSystem::disk_write(const char *buffer, int block_number) {
    int i, addr;
    
    addr = block_number * 512;
    for (i = 0; i < 512; i += 64) {
        _write(addr + i, &buffer[i], 64);    
    }
    return 0;    
}

int PRAMFileSystem::disk_read(char *buffer, int block_number) {        
    int i, addr;
    
    addr = block_number * 512;
    for (i = 0; i < 512; i += 64) {
        _read(addr + i, &buffer[i], 64);    
    }
    return 0;
}

int PRAMFileSystem::disk_status() { return 0; }
int PRAMFileSystem::disk_sync() { return 0; }
int PRAMFileSystem::disk_sectors() { return _sectors; }


int PRAMFileSystem::_status () {
    int r;

    _cs = 0;
    _spi.write(CMD_RDSR);
    r = _spi.write(0);
    _cs = 1;
    return r;
}

int PRAMFileSystem::_write (int addr, const char *buf, int len) {
    int i;

    while (_status() & 1) {
        // write in progress
        wait_us(1);
    }

    _cs = 0;
    _spi.write(CMD_WREN);
    _cs = 1;
    wait_us(1);

    _cs = 0;
    _spi.write(CMD_PP_BA);
    _spi.write((addr >> 16) & 0xff);
    _spi.write((addr >> 8) & 0xff);
    _spi.write(addr & 0xff);

    len = len - (addr & 0x3f);
    for (i = 0; i < len; i ++) {
        _spi.write(buf[i]);
    }
    _cs = 1;
    return i;
}

int PRAMFileSystem::_read (int addr, char *buf, int len) {
    int i;

    while (_status() & 1) {
        // write in progress
        wait_us(1);
    }

    _cs = 0;
    _spi.write(CMD_READ);
    _spi.write((addr >> 16) & 0xff);
    _spi.write((addr >> 8) & 0xff);
    _spi.write(addr & 0xff);

    for (i = 0; i < len; i ++) {
        buf[i] = _spi.write(0);
    }
    _cs = 1;
    return i;
}
