/**
 * NEC Near Field Communication RF module library for mbed
 * Copyright (c) 2015 Suga
 * Released under the MIT License: http://mbed.org/license/mit
 */

/** @file
 * @brief NEC Near Field Communication RF module library for mbed
 * @note H001-000003-001 (950MHz), TY24FM-E2024 (2.4GHz), H001-000013-001 (920MHz)
 */

#include "mbed.h"

#define NEC_MAXLENGTH 240
#define NEC_HEADER_SIZE 13
#define NEC_SYSTEMID  0x0000
#define NEC_PRODUCTID 0x8341
#define NEC_DUMMYID 0xffffffff

#define NEC_TIMEOUT 3 // sec

//#define DEBUG_DUMP
//#define DBG(...) printf("" __VA_ARGS__) 
#define DBG(...)

// host to network short
#define htons( x ) ( (( (x) << 8 ) & 0xFF00) | (( (x) >> 8 ) & 0x00FF) )
#define ntohs( x ) htons(x)
// host to network long
#define htonl( x ) ( (( (x) << 24 ) & 0xFF000000)  \
                   | (( (x) <<  8 ) & 0x00FF0000)  \
                   | (( (x) >>  8 ) & 0x0000FF00)  \
                   | (( (x) >> 24 ) & 0x000000FF)  )
#define ntohl( x ) htonl(x)


/**
 * NECnfc class
 */
class NECnfc {
public:
    enum NECTYPE {
        TYPE_950MHz,
        TYPE_2400MHz,
        TYPE_920MHz,
    };

    enum NECMSG {
        NECMSG_ACK           = 0x00,
        NECMSG_NOACK         = 0x01,
        NECMSG_SEARCH        = 0x10,
        NECMSG_SEND_DAT      = 0x11,
        NECMSG_RESEND        = 0x12,
        NECMSG_SEND_NOACK    = 0x13,
        NECMSG_ENERGY_DETECT = 0x16,
        NECMSG_SEND_CMD      = 0x17,
        NECMSG_WRITE_CHANNEL = 0x20,
        NECMSG_WRITE_RFCONF  = 0x21,
        NECMSG_READ_RSSI     = 0x24,
        NECMSG_READ_CONFIG   = 0x29,
        NECMSG_WRITE_CONFIG  = 0x2A,
        NECMSG_READ_DEFAULT  = 0x7D,
        NECMSG_WRITE_DEFAULT = 0x7E,
        NECMSG_RESET         = 0x77,
    };

    enum NECPWR {
        PWR_LOW           = 0x00,
        PWR_MID           = 0x01,
        PWR_HIGH          = 0x02,
        PWR_MAX           = 0x03,
    };

    enum NECBAUD {
        BAUD_9600           = 0x02,
        BAUD_50k            = 0x06,
        BAUD_100k           = 0x07,
    };

    enum NECUART {
        UART_4800           = 0x01,
        UART_9600           = 0x02,
        UART_19200          = 0x04,
        UART_38400          = 0x05,
        UART_56700          = 0x06,
        UART_115200         = 0x08,
    };

    enum Mode {
        MODE_READY,
        MODE_DATA,
    };

    struct ifMessage {
        uint16_t start;
        uint8_t  length;
        uint8_t  msgid;
        uint8_t  msgno;
        uint32_t dstid;
        uint32_t srcid;
        uint8_t  parameter[NEC_MAXLENGTH + 2];
    } __attribute__((packed));

    // ----- NECnfc.cpp -----
    NECnfc (PinName tx, PinName rx, PinName reset, int baud = 38400, NECTYPE type = TYPE_920MHz);

    NECnfc (PinName tx, PinName rx, PinName reset, PinName wakeup, PinName mode, int baud = 38400, NECTYPE type = TYPE_920MHz);

    void poll ();
    int sendData(int dest, const char *data, int len);
    int readData(int *dest, int *src, char *data, int len);

    void attach (void(*fptr)() = NULL) {
        _func.attach(fptr);
    }
    template<typename T>
    void attach (T* tptr, void (T::*mptr)()) {
        if ((mptr != NULL) && (tptr != NULL)) {
            _func.attach(tptr, mptr);
        }
    }

    // ----- NECnfc_msg.cpp -----
    int send (NECMSG msgid, unsigned int dest, const char *param, int len);

    // ----- NECnfc_util.cpp -----
    int setRfConfig (enum NECPWR power, int ch, NECBAUD baud);
    unsigned int getId ();
    unsigned int getSystemId ();
    int setSystemId (unsigned int id);
    int setSleepMode (int sleep_time, int rev_time); // sleep * 1000ms, rev * 1ms
    int getRssi ();
    int search ();

    // ----- NECnfc_hal.cpp -----
    int sleep (int wait = 0);
    int wakeup (int wait = 0);

protected:
    RawSerial _nec;
    DigitalInOut *_reset;
    DigitalOut *_wakeup;
    DigitalIn *_wmode;
    enum NECTYPE _type;
    FunctionPointer _func;

    enum Mode _mode;
    struct ifMessage _rxmsg;
    char *_rxbuf;
    int _rxlen;
    int _rxcount;
    int _msgno;
    volatile int _ack, _noack, _resend;
    int _txmsg;
    unsigned int _id;
    int _rssi;
    int _received;

    // ----- NECnfc_msg.cpp -----
    void recvData (char c);
    void parseMessage ();

    // ----- NECnfc_hal.cpp -----
    void setReset (bool flg);
    void isrUart ();
    int getUart ();
    void putUart (char c);
    int lockUart (int ms);
    void unlockUart ();
    void initUart (PinName reset, PinName wakeup, PinName mode, int baud);

};
