/**
 * NEC Near Field Communication RF module library for mbed
 * Copyright (c) 2015 Suga
 * Released under the MIT License: http://mbed.org/license/mit
 */

/** @file
 * @brief NEC Near Field Communication RF module library for mbed
 * @note H001-000003-001 (950MHz), TY24FM-E2024 (2.4GHz), H001-000013-001 (920MHz)
 */

#include "mbed.h"
#include "NECnfc.h"
#include <string.h>


NECnfc::NECnfc (PinName tx, PinName rx, PinName reset, int baud, NECTYPE type) : _nec(tx, rx) {

    _rxbuf = (char*)&_rxmsg;
    _type = type;
    _msgno = 0;
    _id = NEC_DUMMYID;
    _received = 0;
    _rxcount = 0;

    initUart(reset, NC, NC, baud);
    setReset(true);
    wait_ms(20);
    setReset(false);
    wait_ms(100);
}


NECnfc::NECnfc (PinName tx, PinName rx, PinName reset, PinName wakeup, PinName mode, int baud, NECTYPE type) : _nec(tx, rx) {

    _rxbuf = (char*)&_rxmsg;
    _type = type;
    _msgno = 0;
    _id = NEC_DUMMYID;
    _received = 0;
    _rxcount = 0;

    initUart(reset, wakeup, mode, baud);
    setReset(true);
    wait_ms(20);
    setReset(false);
    wait_ms(100);
}

void NECnfc::poll () {

    if (_received) {
        _func.call();
        _received = 0;
    }
}

int NECnfc::sendData(int dest, const char *data, int len) {

    if (dest == NEC_DUMMYID) {
        if (send(NECMSG_SEND_NOACK, dest, data, len)) {
            return -1;
        }
    } else {
        if (send(NECMSG_SEND_DAT, dest, data, len)) {
            return -1;
        }
        _rssi = _rxmsg.parameter[2];
    }
    return 0;
}

int NECnfc::readData(int *dest, int *src, char *data, int len) {
    int r;

    if (!_received) return -1;
    if (_rxmsg.msgid != NECMSG_SEND_DAT && _rxmsg.msgid != NECMSG_SEND_NOACK) return -1;

    r = _rxmsg.length - NEC_HEADER_SIZE;
    if (r > len) r = len;
    memcpy(data, _rxmsg.parameter, r);
    if (dest) *dest = ntohl(_rxmsg.dstid);
    if (src) *src = ntohl(_rxmsg.srcid);
    return r;
}
