/*
 * MPU (Memory Protection Unit)
 *
 * Reference:
 *   http://infocenter.arm.com/help/index.jsp?topic=/com.arm.doc.ddi0337gj/BIHHGADD.html
 *   http://www.phoenix-c.or.jp/~olfa/lpcxpresso/27_mpu.html
 */

#include "mbed.h"

DigitalOut myled(LED1);
Serial pc(USBTX, USBRX);

int mpu_config (int region, void *addr, int size, bool readable, bool writable, bool executable, int sub) {
    int a, s, ap, xn;

    for (s = 4; s < 0x20; s ++) {
        if (size == (1<<(s + 1))) break;
    }
    if (s == 0x20) error("Invalid size");

    a = (int)addr & (0xffffffff<<(s + 1));
    if (a != (int)addr) error("Invalid addr");

    if (size <= 128) {
        sub = 0;
    }

    ap = 0;
    if (readable) {
        ap = writable ? 0x03 : 0x07;
    }
    xn = executable ? 0 : 1;

    MPU->CTRL = (1<<2)|(1<<0);
    MPU->RBAR = a|(1<<4)|(region & 0x0f);
    MPU->RASR = (xn<<28)|(ap<<24)|(0x1<<19)|((sub & 0xff)<<8)|(s<<1)|(1<<0);
    return 0;
}


extern "C" {
void HardFault_Handler () {
    error(" *** HardFault\r\n");
}
void MemManage_Handler () {
    error(" *** MemManage\r\n");
}
}

int main() {
    char *buf1 = (char*)0x2007c000;
    char *buf2 = (char*)0x20080000;
    int i;
    volatile int c;

    pc.baud(115200);
    pc.printf("--- mem test\r\n");

    if (mpu_config(0, buf1, 0x4000, true, false, false, 0x01)) return -1;
    if (mpu_config(1, buf2, 0x4000, false, false, false, 0x03)) return -1;
    SCB->SHCSR |= (1<<16); // Enable Memory Management Fault

    pc.printf("\r\nwrite 1\r\n");
    for (i = 0; i < 0x4000; i += 1024) {
        pc.printf(" %d", i);
        c = buf1[i];
    }

    pc.printf("\r\nread 1\r\n");
    for (i = 0; i < 0x4000; i += 1024) {
        pc.printf(" %d", i);
        buf1[i] = c;
    }

    pc.printf("\r\nwrite 2\r\n");
    for (i = 0; i < 0x4000; i += 1024) {
        pc.printf(" %d", i);
        c = buf2[i];
    }

    pc.printf("\r\nread 2\r\n");
    for (i = 0; i < 0x4000; i += 1024) {
        pc.printf(" %d", i);
        buf2[i] = c;
    }
}
