/*
 * Instruction List interpreter library
 * Copyright (c) 2011 Hiroshi Suga
 * Released under the MIT License: http://mbed.org/license/mit
 */

/** @file
 * @brief Instruction List interpreter
 */

#ifndef ILinterpreter_H
#define ILinterpreter_H

#include "mbed.h"

#define IL_NUM 100
#define IL_RELAY_NUM 10
#define IL_TIMER_NUM 10
#define IL_COUNTER_NUM 10
#define IL_STACK 10

enum eMNEMONIC {
    MNE_NULL,
    MNE_DEF,
    MNE_LD, MNE_LDI, MNE_LDP, MNE_LDF,
    MNE_ALD, MNE_ALDI, MNE_ALDP, MNE_ALDF,
    MNE_OR, MNE_ORI, MNE_ORP, MNE_ORF,
    MNE_AND, MNE_ANI, MNE_ANDP, MNE_ANDF,
    MNE_ORB, MNE_ANB,
    MNE_INV,
    MNE_MPS, MNE_MRD, MNE_MPP,
    MNE_OUT, MNE_SET, MNE_RST,
    MNE_END,
};

enum eEXPRESSION {
    EXP_NULL,
    EXP_EQ, EXP_NE,
    EXP_LE, EXP_LT,
    EXP_GE, EXP_GT,
    EXP_MOD, EXP_NMOD,
};

struct tIL {
    enum eMNEMONIC mnemonic;
    char key;
    char keynum;
    enum eEXPRESSION expression;
    float value;
};

struct tInOut {
    time_t sec;
    int relay[IL_RELAY_NUM];
    int timer_flg[IL_TIMER_NUM];
    unsigned int timer_set[IL_TIMER_NUM], timer_cnt[IL_TIMER_NUM];
    unsigned int count_set[IL_COUNTER_NUM], count_cnt[IL_COUNTER_NUM], count_rev[IL_COUNTER_NUM];
};


/** ILinterpreter class
 */
class ILinterpreter {
public:
    ILinterpreter ();

    /** exec IL sequence
     * @retval 0 success
     * @retval -1 error
     */
    int exec ();

    /** set call back function
     * @param pf_i input function (input relay)
     * @param pf_o output function (output relay)
     * @return pointer of tInOut (internal relay)
     */
    struct tInOut* attach (float (*pf_i)(char, int, eEXPRESSION, int), void (*pf_o)(char, int, int, eMNEMONIC));

    /** timer interval (call 10Hz)
     */
    void pool ();

    /** load IL file
     * @param file file name
     * @retval 0 success
     * @retval -1 error
     */
    int load (char *file);

protected:
    int il_count;
    struct tIL il[IL_NUM];
    struct tInOut inout, inout_old;
    int stack[IL_STACK];
    int addr;

    int input (tInOut *io, int i, int old = 0);
    void output (int i, int reg, eMNEMONIC mne);
    void load_exp (int i, char *buf);
    int push (int dat);
    int pop (int *dat);
    int read (int *dat);

    float (*cb_input)(char key, int keynum, eEXPRESSION exp, int old);
    void (*cb_output)(char key, int keynum, int reg, eMNEMONIC mne);

private:

};

#endif
