/**
* @author Daniel Worrall
*
* @section LICENSE
*
* Copyright (c) 2011 mbed
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
* THE SOFTWARE.
*
* @section DESCRIPTION
*    I2S port abstraction library h file for NXP LPC1768
*
*/ 

#include "mbed.h"
#include "math.h"

#ifndef __MBED_I2SSLAVE_H__
#define __MBED_I2SSLAVE_H__

extern FunctionPointer akjnh3489v8ncv;

/** I2S class defined on the LPC1768 port
 *
 */
class I2SSlave{

    public:
        //Constructor
        /** Create an I2S object
         *
         * @param tx_sda Transmitter serial data line
         * @param tx_ws Transmitter word select line
         * @param clk Shared transmitter/receiver clock line
         * @param rx_sda Receiver serial data line
         * @param rx_ws Receiver word select line
         */
        I2SSlave(PinName tx_sda, PinName tx_ws, PinName clk, PinName rx_sda, PinName rx_ws);
        /** Set the data transmission format
         *
         * @param bit Set the number of bits per write
         * @param mode Set STEREO (0) or MONO (1) mode
         */
        void format(int bit, bool mode);
        /** Write a buffer to the I2S port
         *
         * @param buffer Address of buffer to pass to I2S port
         * @param from Start position in buffer to read from
         * @param length Length of buffer (MUST not exceed 8 words, each 32 bits long)
         *
         * Note: sending 8 words to the TXFIFO will trigger an interrupt!
         */
        void write(int* buffer, int from,  int length);
        /** Activate I2S port for data streaming 
         *
         * @param mode Mode to enable - NONE, TRANSMIT only, RECEIVE only, BOTH
         * Enables tx/rx interrupts
         */
        void start(int mode);
        /** Deactivate I2S port from data streaming 
         *
         * Disable all interrupts
         */
        void stop(void);
        /** Load receive FIFO data into receiver buffer
         *
         */
        void read(void);
        /** Attach a void/void function or void/void static memeber function to an interrupt generated by the I2SxxFIFOs
         *
         * @param function Function to attach
         *
         * <code> myI2sObject.attach(&myfunction); </code>
         * OR <code> myI2sObject.attach(&myClass::myStaticMemberFunction); </code>
         */
        void attach(void(*fptr)(void)){
            akjnh3489v8ncv.attach(fptr);
        }
        /** Attach a nonstatic void/void member function to an interrupt generated by the I2SxxFIFOs
         *
         * @param tptr Object pointer
         * @param mptr Member function pointer
         *
         * e.g. <code>myI2sObject.attach(&myObject, &myClass::myNonstaticMemberFunction); </code>where myObject is an object of myClass
         */
        template<typename T>
        void attach(T *tptr, void(T::*mptr)(void)){
            akjnh3489v8ncv.attach(tptr, mptr);
        }
        /** Return contents of I2S status register
         *
         * @returns Content of I2SSTATE register
         *
         * bit0: receive/transmit interrupt active
         * bit1: receive/transmit DMA request 1
         * bit2: receive/transmit DMA request 2
         * bit[11:8]: receive FIFO level
         * bit[19:16]: transmit FIFO level
         */
        int status(void);
        
        //Receiver buffer
        int rxBuffer[4];
        //defines
        #define STEREO  0
        #define MONO    1
        
        #define I2SFIFO_EMPTY    0
        #define I2SFIFO_FULL     8
        
        #define RAM_LENGTH  1024
        #define RAM_LIMIT   (RAM_LENGTH - 1) 
        #define PTR_MAX     ((RAM_LENGTH / 8) - 1)
        
        #define NONE        0
        #define TRANSMIT    1
        #define RECEIVE     2
        #define BOTH        3

    private:        
        /** I2S intitalize function
         *
         * @param tx_sda Transmitter serial data line
         * @param tx_ws Transmitter word select line
         * @param clk Shared transmitter/receiver clock line
         * @param rx_sda Receiver serial data line
         * @param rx_ws Receiver word select line
         * @return Returns 0 for successful initialisation, -1 for an error
         */
        int initialize_(PinName tx_sda, PinName tx_ws, PinName clk, PinName rx_sda, PinName rx_ws);
        /** Set internal clock divide by rate
         *
         * @param divideBy Divide by 1, 2, 4 or 8
         */
        void setClocks_(int divideBy);
        /** Set up the pins on the processor itself
         *
         * @param tx_sda Transmitter serial data line
         * @param tx_ws Transmitter word select line
         * @param clk Shared transmitter/receiver clock line
         * @param rx_sda Receiver serial data line
         * @param rx_ws Receiver word select line
         */
        void setPins_(PinName tx_sda, PinName tx_ws, PinName clk, PinName rx_sda, PinName rx_ws);
        /** Set the data transmission format 
         *
         * @param bit Set the number of bits per write
         * @param mode Set STEREO (0) or MONO (1) mode
         */
        void format_(int bit, bool mode);
        /** Set slave mode
         *
         */
        void modeConfig_(void);
        /** Store PinName values
         * 
         * @param tx_sda Transmitter serial data line
         * @param tx_ws Transmitter word select line
         * @param clk Shared transmitter/receiver clock line
         * @param rx_sda Receiver serial data line
         * @param rx_ws Receiver word select line
         */
        void storePins_(PinName tx_sda, PinName tx_ws, PinName clk, PinName rx_sda, PinName rx_ws);
        //variables
        int bit_;
        bool mode_;
        PinName tx_sda_; 
        PinName tx_ws_; 
        PinName clk_;
        PinName rx_sda_;
        PinName rx_ws_;
};

#endif /*__MBED_I2S_H__*/