#include "mbed.h"
#include "DMX.h"

#define RDM_MANUFACTURER_ID 0x7ff7
#define RDM_DEVICE_MODEL    0x0001
#define RDM_SW_VERSION      0x01020304 // V.1.2.3.4
#define RDM_DMX_FOOTPRINT   4 // slots
#define RDM_START_ADDRESS   1

Serial pc(USBTX, USBRX);
DigitalOut led1(LED1), led2(LED2), led3(LED3), led4(LED4);
DMX dmx(p9, p10, p21); // rx, tx, xmit
//DMX dmx(p9, p10);

char uid[] = {(RDM_MANUFACTURER_ID >> 8) & 0xff, RDM_MANUFACTURER_ID & 0xff, 0x12, 0x34, 0x56, 0x78};
int dmx_address = 0;
int dmx_footprint = 10;

int sendRdmDeviceInfoResponse (struct RDM_DATA *rdm) {
    unsigned char data[20];

    data[ 0] = 1; // RDM Protocol Version
    data[ 1] = 0;
    data[ 2] = (RDM_DEVICE_MODEL >> 8) & 0xff; // Device model
    data[ 3] = RDM_DEVICE_MODEL & 0xff;
    data[ 4] = 1; // Product Category
    data[ 5] = 0;
    data[ 6] = RDM_SW_VERSION & 0xff; // software version
    data[ 7] = (RDM_SW_VERSION >> 8) & 0xff;
    data[ 8] = (RDM_SW_VERSION >> 16) & 0xff;
    data[ 9] = (RDM_SW_VERSION >> 24) & 0xff;
    data[10] = (dmx_footprint >> 8) & 0xff; // DMX512 Footprint
    data[11] = dmx_footprint & 0xff;
    data[12] = 1; // DMX512 Personality
    data[13] = 1;
    data[14] = ((dmx_address + 1) >> 8) & 0xff; // DMX512 Start Address
    data[15] = (dmx_address + 1) & 0xff;
    data[16] = 0; // Sub-Device Count
    data[17] = 0;
    data[18] = 0; // Sensor Count
    return dmx.sendRdmMsg(rdm, E120_GET_COMMAND_RESPONSE, data, 19);
}

void rdmParser (struct RDM_DATA *rdm) {
    unsigned char data[20];

    led3 = 1;
    switch (rdm->CmdClass) {
    case E120_GET_COMMAND:
        switch (__REV16(rdm->Parameter)) {
        case E120_SUPPORTED_PARAMETERS:
            uint16_t *tmp = (uint16_t*)data;
            tmp[0] = E120_MANUFACTURER_LABEL;
            tmp[1] = E120_DEVICE_MODEL_DESCRIPTION;
            tmp[2] = E120_DEVICE_LABEL;
            tmp[3] = E120_SOFTWARE_VERSION_LABEL;
            tmp[4] = E120_BOOT_SOFTWARE_VERSION_ID;
            tmp[5] = E120_BOOT_SOFTWARE_VERSION_LABEL;
            for (int i = 0; i < 6; i ++) {
                tmp[i] = __REV16(tmp[i]);
            }
            dmx.sendRdmMsg(rdm, E120_GET_COMMAND_RESPONSE, data, 2 * 6);
            break;

        case E120_IDENTIFY_DEVICE:
            data[0] = led4;
            dmx.sendRdmMsg(rdm, E120_GET_COMMAND_RESPONSE, data, 1);
            break;

        case E120_DEVICE_INFO:
            sendRdmDeviceInfoResponse(rdm);
            break;

        case E120_DMX_START_ADDRESS:
            data[0] = ((dmx_address + 1) >> 8) & 0xff;
            data[1] = (dmx_address + 1) & 0xff;
            dmx.sendRdmMsg(rdm, E120_GET_COMMAND_RESPONSE, data, 2);
            break;

        case E120_SOFTWARE_VERSION_LABEL:
            strcpy((char*)data, "DMX 1.2.3.4");
            dmx.sendRdmMsg(rdm, E120_GET_COMMAND_RESPONSE, data, strlen((char*)data));
            break;

        case E120_MANUFACTURER_LABEL:
            strcpy((char*)data, "ARM Mbed");
            dmx.sendRdmMsg(rdm, E120_GET_COMMAND_RESPONSE, data, strlen((char*)data));
            break;

        case E120_DEVICE_MODEL_DESCRIPTION:
            strcpy((char*)data, "model");
            dmx.sendRdmMsg(rdm, E120_GET_COMMAND_RESPONSE, data, strlen((char*)data));
            break;

        case E120_DEVICE_LABEL:
            strcpy((char*)data, "device");
            dmx.sendRdmMsg(rdm, E120_GET_COMMAND_RESPONSE, data, strlen((char*)data));
            break;

        case E120_BOOT_SOFTWARE_VERSION_ID:
            data[0] = RDM_SW_VERSION & 0xff; // software version
            data[1] = (RDM_SW_VERSION >> 8) & 0xff;
            data[2] = (RDM_SW_VERSION >> 16) & 0xff;
            data[3] = (RDM_SW_VERSION >> 24) & 0xff;
            dmx.sendRdmMsg(rdm, E120_GET_COMMAND_RESPONSE, data, 4);
            break;

        case E120_BOOT_SOFTWARE_VERSION_LABEL:
            strcpy((char*)data, "1.2.3.4");
            dmx.sendRdmMsg(rdm, E120_GET_COMMAND_RESPONSE, data, strlen((char*)data));
            break;

        case E120_QUEUED_MESSAGE:
        case E120_STATUS_MESSAGES:
            data[0] = E120_STATUS_NONE;
            dmx.sendRdmMsg(rdm, E120_GET_COMMAND_RESPONSE, data, 1);
            break;

        default:
            printf("Get %04x\r\n", __REV16(rdm->Parameter));
            break;
        }
        break;

    case E120_SET_COMMAND:
        switch (__REV16(rdm->Parameter)) {
        case E120_IDENTIFY_DEVICE:
            if (rdm->DataLength == 1) {
                led4 = rdm->Data[0];
                dmx.sendRdmMsg(rdm, E120_SET_COMMAND_RESPONSE, rdm->Data, rdm->DataLength);
            }
            break;

        case E120_DMX_START_ADDRESS:
            if (rdm->DataLength == 2) {
                dmx_address = ((rdm->Data[0] << 8) | rdm->Data[1]) - 1;
                dmx.sendRdmMsg(rdm, E120_SET_COMMAND_RESPONSE, rdm->Data, rdm->DataLength);
                printf("Set addr %04x\r\n", dmx_address);
            }
            break;

        default:
            printf("Set %04x\r\n", __REV16(rdm->Parameter));
            break;
        }
    }

printf("rdm len=%d cmdcls=%02x param=%04x datlen=%d\r\n", rdm->Length, rdm->CmdClass, __REV16(rdm->Parameter), rdm->DataLength);
for (int i = 0; i < rdm->DataLength; i ++) {
    printf(" %02x", rdm->Data[i]);
}
printf("\r\n");
}


void isr_timer () {
    led2 = led3 = 0;
}

int main() {
    Ticker t;
    unsigned char uid_buf[10 * 6];

    pc.baud(115200);
    pc.printf("*** DMX RDM\r\n");
    dmx.attachRdmCallback(&rdmParser, uid);
//    dmx.start();
    led1 = 1;
    t.attach(&isr_timer, 0.1);

    for (;;) {
        dmx.pollRdm();

        if (dmx.isReceived()) {
            led2 = 1;
            printf("dmx ");
            for (int i = 0; i < dmx_footprint; i ++) {
                printf(" %02x", dmx.get(dmx_address + i));
            }
            printf("\r\n");
        }

        if (pc.readable()) {
            switch (pc.getc()) {
            case 's':
                dmx.start();
                break;

            case 't':
                dmx.stop();
                break;

            case 'd':
                int n = dmx.rdmDiscovery(uid_buf, 10);

                printf("found %d\r\n", n);
                for (int i = 0; i < n * 6; i ++) {
                    printf(" %02x", uid_buf[i]);
                }
                printf("\r\n");
                break;
            }
        }
    }
}
