/*
 * DMX Station - mbed DMX Platform
 * Copyright (c) 2012 Hiroshi Suga
 * Released under the MIT License: http://mbed.org/license/mit
 */

/*
 * "patch#.csv" (#=dest port)
 *  |dest ch,src port,src ch|
 *
 * dest port: 1=dmx out 1 / 2=dmx out 2 / 3=ArtNet 1 / 4=ArtNet 2 / 5=RF(NEC950MHz)
 * src port: 1=dmx in 1 / 2=dmx in 2 / 3=ArtNet 1 / 4=ArtNet 2 / 5=UDP2DMX
 */
#include "dbg.h"
#include "mbed.h"
#include "SDFileSystem.h"
#include "DmxArtNet.h"
#include "DMX.h"
#include "dmx_patch.h"


SDFileSystem sd(p5, p6, p7, p8, "sd");

struct TPatch dmx_patch[DEST_NUM][512] __attribute__((section("AHBSRAM0")));

extern DmxArtNet art;
extern DMX dmx1;
extern DMX dmx2;

extern volatile int patch_update;
extern volatile int art_update[2];
extern volatile int rf_update;
extern char dmx_art[2][512];
extern char dmx_udp[512];
extern char dmx_rf[512];


int load_patch () {
    int i, j, dest, src, port;
    FILE *fp;
    char file[32];

    for (i = 0; i < DEST_NUM; i ++) {
        for (j = 0; j < 512; j ++) {
            dmx_patch[i][j].port = i < 2 ? i + 2 : i - 2;
            dmx_patch[i][j].addr = j;
        }
    }

    if (sd.disk_initialize()) {
        DBG("no SD\r\n");
        return -1;
    }

    for (i = 0; i < DEST_NUM; i ++) {
        sprintf(file, "/sd/patch%d.csv", i + 1);
        fp = fopen(file, "r");
        if (! fp) continue;
        
        while (1) {
            // load csv
            if (feof(fp)) break;
            if (fscanf(fp, "%d,%d,%d", &dest, &port, &src) == 3) {
                if (dest < 1 && dest > 512) continue;
                if (src < 1 && src > 512) continue;
                if (port < 1 && port > 5) continue;
                dmx_patch[i][dest - 1].port = port - 1;
                dmx_patch[i][dest - 1].addr = src - 1;
                DBG("%d %d <- %d %d\r\n", i + 1, dest, port, src);
            }
        }
        fclose(fp);
    }
    return 0;
}

void patch () {
    int i, j, dmx;

    for (i = 0; i < DEST_NUM; i ++) {
        for (j = 0; j < 512; j ++) {
            // src
            switch (dmx_patch[i][j].port) {
            case 0: // dmx in 1
                dmx = dmx1.get(dmx_patch[i][j].addr);
                break;
            case 1: // dmx in 2
                dmx = dmx2.get(dmx_patch[i][j].addr);
                break;
            case 2: // ArtNet 1
                dmx = art.DmxIn[0][dmx_patch[i][j].addr];
                break;
            case 3: // ArtNet 2
                dmx = art.DmxIn[1][dmx_patch[i][j].addr];
                break;
            case 4: // UDP2DMX
                dmx = dmx_udp[dmx_patch[i][j].addr];
                break;
            default:
                continue;
            }
            
            // dest
            switch (i) {
            case 0: // dmx out 1
                dmx1.put(j, dmx);
                break;
            case 1: // dmx out 2
                dmx2.put(j, dmx);
                break;
            case 2: // ArtNet 1
                if (dmx_art[0][j] != dmx) {
                    dmx_art[0][j] = dmx;
                    art_update[0] = 1;
                }
                break;
            case 3: // ArtNet 2
                if (dmx_art[1][j] != dmx) {
                    dmx_art[1][j] = dmx;
                    art_update[1] = 1;
                }
                break;
            case 4: // RF(NEC950MHz)
                if (dmx_rf[j] != dmx) {
                    dmx_rf[j] = dmx;
                    rf_update = 1;
                }
                break;
            }
        }
    }
}
