/*
 * DMX512, RDM send/recv library
 * Copyright (c) 2017 Hiroshi Suga
 * Released under the MIT License: http://mbed.org/license/mit
 */

/** @file
 * @brief DMX512 send/recv
 */
 
#ifndef __DMX_H__
#define __DMX_H__

#define RDM_ENABLE

#include "mbed.h"
#include "RawSerial.h"

#ifdef RDM_ENABLE
#include "RDM.h"
#endif

//#define DMX_UART_DIRECT

#define DMX_SIZE        512
#define DMX_START_CODE  0

#define RDM_START_CODE  E120_SC_RDM
#define RDM_SUB_CODE    E120_SC_SUB_MESSAGE
#define XMIT_TX         1
#define XMIT_RX         0

#define DMX_TIME_BREAK  100 // 100us (88us-1s)
#define DMX_TIME_MAB    12 // 12us (8us-1s)
#define DMX_TIME_MBB    200 // 10us (0-1s)
#define RDM_TIME_DELAY  100

enum DMX_MODE {
    DMX_MODE_BEGIN,
    DMX_MODE_START,
    DMX_MODE_BREAK,
    DMX_MODE_MAB,
    DMX_MODE_DATA,
    DMX_MODE_ERROR,
    DMX_MODE_STOP,
    DMX_MODE_RDMSUB,
    DMX_MODE_RDM,
};

#ifdef RDM_ENABLE
struct RDM_DATA {
//    uint8_t StartCode;
    uint8_t SubStartCode;
    uint8_t Length;
    uint8_t DestID[6];
    uint8_t SourceID[6];
    uint8_t TransactionNo;
    uint8_t ResponseType;
    uint8_t MessageCount;
    uint16_t SubDev;
    uint8_t CmdClass;
    uint16_t Parameter;
    uint8_t DataLength;
    uint8_t Data[231];
} __attribute__((packed));
#endif

/** DMX512 class (sender/client)
 */
class DMX {
public:
    /** init DMX class
     * @param p_tx TX serial port (p9, p13, p28)
     * @param p_rx RX serial port (p10, p14, p27)
     * @param p_re data enable/~receive enable
     */
    DMX (PinName p_tx, PinName p_rx, PinName p_xmit = NC); 

    /** Send the data
     * @param addr DMX data address (0-511)
     * @param data DMX data (0-255)
     */
    void put (int addr, int data);
    /** Send the data
     * @param buf DMX data buffer
     * @param addr DMX data address
     * @param len data length
     */
    void put (unsigned char *buf, int addr = 0, int len = DMX_SIZE);

    /** Send the data
     * @param addr DMX data address (0-511)
     * @return DMX data (0-255)
     */
    int get (int addr);
    /** Send the data
     * @param buf DMX data buffer
     * @param addr DMX data address
     * @param len data length
     */
    void get (unsigned char *buf, int addr = 0, int len = DMX_SIZE);

    /** Start DMX send operation
     */
    void start ();
    /** Stop DMX send operation
     */
    void stop ();
    /** Clear DMX data
     */
    void clear ();

    int isReceived ();
    int isSent ();
    unsigned char *getRxBuffer ();
    unsigned char *getTxBuffer ();
    int setTimingParameters (int breaktime, int mab, int mbb);

#ifdef RDM_ENABLE
    void pollRdm ();
    int sendRdmMsg (struct RDM_DATA *rdm, int CmdClass, unsigned char *data, int len);
    int sendRdmMsg (unsigned char *dest, int CmdClass, int Parameter, int Type, unsigned char *data, int len, int block = 0);
    void attachRdmCallback (void (*handler)(struct RDM_DATA *), char *uid);

    int sendRdmDiscMute (unsigned char *dest, int mute);
    int rdmDiscovery (unsigned char *buf, int size);
#endif

protected:

    void int_timer ();
    void int_tx ();
    void int_rx ();

//    Serial _dmx;
    RawSerial _dmx;
    DigitalOut *_xmit;
    Timeout timeout01;
    volatile DMX_MODE mode_tx, mode_rx;
    volatile int addr_tx, addr_rx;
    unsigned char data_tx[DMX_SIZE];
    unsigned char data_rx[DMX_SIZE];
    volatile int is_received, is_sent, is_rdm_received;
    int time_break, time_mab, time_mbb;

#ifdef RDM_ENABLE
    volatile int mode_rdm;
    unsigned char rdm_uid[6];
    int rdm_mute, rdm_msgcount, rdm_transno;
    unsigned char data_rdm[64];
    unsigned char found_uid[6];
    unsigned char *buf_uid;
    int buf_uid_count, buf_uid_size;

    int calcCrc (unsigned char *buf, int len, int offset = RDM_START_CODE);
    void rdmStart (int block = 0);
    void rdmWaitResponse (int ms);
    int sendRdmDiscResponse (struct RDM_DATA *rdm);
    int sendRdmDiscMuteResponse (struct RDM_DATA *rdm);

    void int_rdm ();
    int sendRdmDiscovery (uint64_t uid_begin, uint64_t uid_end);
    int rdmDiscoverySub (uint64_t uid_begin, uint64_t uid_end, int ttl);

    void (*cb_RdmParser)(struct RDM_DATA *rdm);
#endif

private:
#if defined(TARGET_LPC1768) || defined(TARGET_LPC2368) || defined(TARGET_LPC4088)
    LPC_UART_TypeDef *_uart;
#elif defined(TARGET_LPC11UXX)
    LPC_USART_Type *_uart;
#elif defined(TARGET_LPC11XX)
    LPC_UART_TypeDef *_uart;
#else
#error "this CPU not supported."
#endif

};

#endif