/*
 * DALI send/recv library
 * Copyright (c) 2020 Hiroshi Suga
 * Released under the MIT License: http://mbed.org/license/mit
 */

/** @file
 * @brief DALI send/recv
 */
 
 #ifndef _DALI_H_
#define _DALI_H_

#include "CBuffer.h"

class DALI {
public:
    enum DALI_FRAME {
        FORWARD_SHORT_DAP, // DIRECT ARC POWER
        FORWARD_SHORT_IAP, // COMMAND
        FORWARD_GROUP_DAP,
        FORWARD_GROUP_IAP,
        BACKWARD,
    };

    enum DALI_COMMAND {
        OFF                 = 0x00,
        UP                  = 0x01,
        DOWN                = 0x02,
        STEP_UP             = 0x03,
        STEP_DOWN           = 0x04,
        RECALL_MAX_LEVEL    = 0x05,
        RECALL_MIN_LEVEL    = 0x06,
        STEP_DOWN_AND_OFF   = 0x07,
        ON_AND_STEP_UP      = 0x08,
        RESET               = 0x20,
        QUERY_STATUS        = 0x90,
    };

    /** init DALI class
     * @param tx TX port
     * @param rx RX port (interrupt)
     */
    DALI (PinName tx, PinName rx);

    /** Recv the data
     * @param frame enum DALI_FRAME 
     * @param addr DALI address (short:0-63, group:0-15,63 broadcast)
     * @param value DALI value
     */
    int read (enum DALI_FRAME *frame, int *addr, int *value);

    int readable ();

    /** Send the data
     * @param frame enum DALI_FRAME 
     * @param addr DALI address (short:0-63, group:0-15,63 broadcast)
     * @param value DALI value
     */
    int write (enum DALI_FRAME frame, int addr, int value);

    int writable ();

private:
    InterruptIn _rx;
    DigitalOut _tx;
    Timeout _timer;
    Ticker _ticker;

    CircBuffer<int> *recv_buf;
    CircBuffer<int> *send_buf;

    volatile int mode;
    int count;
    int timeflg;
    int recv_bit;
    int recv_data;

    int send_data;
    int send_bit;
    int halfbit;
    volatile int busy;

    void isr_rx ();
    void isr_timer ();
    void isr_timeout ();
    void isr_send ();

};

#endif
