#include "mbed.h"
#include "MjGP2Y0E03.h"

BusOut      leds( LED1, LED2, LED3 );
DigitalOut  vitalsign( LED4 );
AnalogIn    key0( p19 );
AnalogOut   ao( p18 );

I2C         i2c( p28, p27 );
MjGP2Y0E03  distanceSensor(&i2c, 0x80);

#define PI          3.1415926535897932384626433
#define K           3.0
#define K2          12.0

#define SQUELCH     0.0

#define N_SAMPLES   16384
short           *sample     = (short *)0x2007c000;

Ticker          in;
Ticker          out;
int             in_flag     = 0;
float           amp;
unsigned int    count    = 0;
float           ci  = 1.0;
float           ref_tone;

#define     SHORT_HIGH  0
#define     SHORT_LOW   1
#define     CONTROL_DIRECTION   SHORT_LOW

//#define     OPERATION_AID

enum Tone_list {
    Tone_A      = 0,
    Tone_Ais,
    Tone_H,
    Tone_C,
    Tone_Cis,
    Tone_D,
    Tone_Dis,
    Tone_E,
    Tone_Eis,
    Tone_F,
    Tone_G,
    Tone_Gis,
};

#define     REFERENCE_TONE  ((float)Tone_C)
#define     REFERENCE_PITCH 442.0

void init( void );
void set_in_flag();
void data_output();
int tone_indicator( float ci );

int main()
{
    float   ai;
    int     vc  = 0;

    init();

    while(1) {
        if ( in_flag ) {
            in_flag = 0;

            //
            //  output amplitude
            //
            amp = (key0 < SQUELCH) ? 0.0 : key0 * 2.0;

            //
            //  output frequency
            //
            ai  = distanceSensor.rd();

#if (CONTROL_DIRECTION == SHORT_HIGH)
            ci  = (2.25 - ai * 2.0);
#else
            ci  = (0.75 + ai * 2.0);
#endif

#ifdef  OPERATION_AID
            leds    = tone_indicator( ci );
#else
            //
            //  reference tone indicator
            //
            leds    = 0x0;
            if ( (0.943874 < ci) && (ci < 1.05946) ) {
                leds    = 0x1;
            }
            if ( (0.971532 < ci) && (ci < 1.0293) ) {
                leds    = 0x3;
            }
            if ( (0.985663 < ci) && (ci < 1.01455) ) {
                leds    = 0x7;
            }
#endif
            vitalsign   = (vc++ >> 4) & 0x1;
        }
    }
}



float waveform_generator( int i )
{
#define WAVEFORM_SIN
//#define WAVEFORM_SAWTOOTH
//#define WAVEFORM_CUSTOM_HARMONICS

    float   r;

#ifdef  WAVEFORM_SAWTOOTH
    r   = (float)i / (float)N_SAMPLES;
#endif

#ifdef  WAVEFORM_SIN
    r   = sin( 2.0 * PI * ((float)i / (float)N_SAMPLES) );
#endif

#ifdef  WAVEFORM_CUSTOM_HARMONICS

    typedef struct  element_st {
        float   frequency;
        float   amplitude;
        float   phase;
    }
    element;

#define REF_AMPLITUDE   1.0
    static element e[] = {
        { 1.0,      REF_AMPLITUDE / 1.0,    0 * PI },
//        { 2.0,      REF_AMPLITUDE / 2.0,    0 * PI },
        { 3.0,      REF_AMPLITUDE / 3.0,    0 * PI },
//        { 4.0,      REF_AMPLITUDE / 4.0,    0 * PI },
        { 5.0,      REF_AMPLITUDE / 5.0,    0 * PI },
//        { 6.0,      REF_AMPLITUDE / 4.0,    0 * PI },
        { 7.0,      REF_AMPLITUDE / 7.0,    0 * PI },
//        { 3.33333,  REF_AMPLITUDE / 2.0,    0 * PI },//  nonintegral harmonics
//        { 11.9311,  REF_AMPLITUDE / 2.0,    0 * PI },//  nonintegral harmonics
    };

    float   f;
    r   = 0.0;

    for ( int x = 0; x < sizeof( e ) / sizeof( element ); x++ ) {
        f   = e[ x ].frequency * 2.0 * PI * ((float)i / (float)N_SAMPLES);
        r  += e[ x ].amplitude * cos( e[ x ].phase ) * sin( f );
        r  += e[ x ].amplitude * sin( e[ x ].phase ) * cos( f );
    }
#endif
    return ( r );
}

void init( void )
{
    //
    //  prepare waveform table in RAM
    //
    for ( int i = 0; i < N_SAMPLES; i++ ) {
        sample[ i ] = (short)(32767.0 * waveform_generator( i ));
    }

    //
    //  set reference tone coefficient
    //
    ref_tone   = (REFERENCE_PITCH * pow( 2.0, REFERENCE_TONE / 12.0 )) /K2 * K;

    //  I2C clock frequency (option)
    i2c.frequency( 400 * 1000 );

    //
    //  set periodic interrupt to sample user inputs
    //
    in.attach( &set_in_flag, 1.0 / 200.0 );

    //
    //  set periodic interrupt to output sound data
    //
    out.attach( &data_output, 1.0 / ((float)(N_SAMPLES / K) * K2) );

//    printf( "output sampling freq = %f kHz\r\n", ((float)(N_SAMPLES / K) * K2) / 1000.0 );
}

void set_in_flag()
{
    in_flag    = 1;
}

void data_output()
{
    ao      = 0.5 + (amp * ((float)sample[ count & 0x3FFF ] / 65536.0));
    count  += (int)(ref_tone * ci);
}


int tone_indicator( float ci )
{
    static float   indicator_reference[]    = {
        1.0,
        pow( 2.0, 2.0 / 12.0 ),
        pow( 2.0, 4.0 / 12.0 ),
        pow( 2.0, 5.0 / 12.0 ),
        pow( 2.0, 7.0 / 12.0 ),
        pow( 2.0, 9.0 / 12.0 ),
        pow( 2.0, 11.0 / 12.0 ),
    };
    static float    ratio   = pow( 2.0, 1.0 / 48.0 );
    int             i;

    ci  = (ci < 1.0) ? ci + 1.0 : ci;
    ci  = (2.0 < ci) ? ci - 1.0 : ci;

    for ( i = 0; i < sizeof( indicator_reference ) / sizeof( float ); i++ ) {
        if ( (indicator_reference[ i ] / ratio < ci) && (ci < indicator_reference[ i ] * ratio) )
            return ( i + 1 );
    }
    return ( 0 );
}