/** サンプルコード：ライブラリ開発の例
 *
 *  @author  Tedd OKANO
 *  @version 1.1
 *  @date    11-Nov-2014
 *
 *  ライブラリ開発の例として，コードをステップ・バイ・ステップで解説しています
 *  各ステップはリポジトリ履歴で追うことができます
 *  I2Cインターフェースを持った温度センサのLM75Bをターゲットとしています
 *
 *  LM75Bの詳細は
 *    http://www.nxp.com/documents/data_sheet/LM75B.pdf
 */

#include "mbed.h"

/** デフォルト・スレーブアドレス */
#define     ADDRESS_LM75B   0x90

/** LM75Bのレジスタ名とアドレス */
#define     LM75B_Conf      0x01
#define     LM75B_Temp      0x00
#define     LM75B_Tos       0x03
#define     LM75B_Thyst     0x02

/** test_LM75Bクラスライブラリ
 *
 *  クラスライブラリは非常にシンプルなインターフェースを提供します
 *
 *  コード例:
 *  @code
 *  #include "mbed.h"
 *  #include "test_LM75B.h"
 *  
 *  test_LM75B  temp0( p28, p27 );
 *  
 *  I2C         i2c( p28, p27 );
 *  test_LM75B  temp1( i2c );
 *  
 *  
 *  int main()
 *  {
 *      float   t0;
 *      float   t1;
 *  
 *      i2c.frequency( 400 * 1000 );
 *  
 *      while(1) {
 *          t0   = temp0;
 *          t1   = temp1;
 *          printf( "temp = %7.3f, %7.3f\r\n", t0, t1 );
 *          wait( 1 );
 *      }
 *  }
 *  @endcode
 */
class test_LM75B
{
public:

    /** I2Cピンとスレーブアドレスを指定し，インスタンスを作成します
     *
     * @param sda I2C-bus SDAピン
     * @param scl I2C-bus SCLピン
     * @param address (オプション) I2C-bus スレーブアドレス (デフォルト: 0x90)
     */
    test_LM75B( PinName sda, PinName scl, char address = ADDRESS_LM75B );

    /** I2Cオブジェクトとスレーブアドレスを指定し，インスタンスを作成します
     *
     * @param i2c_obj I2C オブジェクト (インスタンス)
     * @param address (オプション) I2C-bus スレーブアドレス (デフォルト: 0x90)
     */
    test_LM75B( I2C &i2c_obj, char address = ADDRESS_LM75B );
    
    /** デストラクタ
     */
    ~test_LM75B();
    
    /** 初期化
     */    
    void    init( void );

    /** 温度の読み出し
     * 
     *  @return 摂氏温度を返します(float型) 
     */
    float   read( void );

    /** 温度の読み出し
     * 
     *  @return オブジェクトが読みだした値を返すようにしています 
     */
    operator float( void );
    
private:
    I2C     *i2c_p;
    I2C     &i2c;
    char    adr;
};
