/*
 *  Operation sample code for SC18IS606 library, using EEPROM (AT25010B)
 *
 *  @author     Akifumi (Tedd) OKANO, NXP Semiconductors
 *  @version    0.2
 *  @date       28-July-2021
 *
 *  SC18IS606 is an "I2C-bus to SPI bridge"
 *  http://www.nxp.com/ (product infomation page will be updated later)
 */

#include "mbed.h"
#include "SC18IS606.h"

I2C         i2c( p28, p27 );
InterruptIn int_line( p21 );
SC18IS606   bridge( i2c );

#define I2C_FREQUENCY       (400 * 1000)    //  Hz
#define SLAVE_SELECT_NUM    1               //  AT25010B is connected to /SS1 pin

#define WRITE_DATA_LENGTH   8                       //  1 or 8 (single byte or page write)
#define TRANSFER_LENGTH     (2 + WRITE_DATA_LENGTH) //  AT25010B needs command(1 byte), address(1 byte) + data(1 or 8 byte)
#define READ_DATA_LENGTH    128
#define RCV_LENGTH          (2 + READ_DATA_LENGTH)  //  AT25010B returnss 2 bytes dummy + data(1 or 8 byte)  

void    data_check( char *data, int length );

volatile int    int_flag    = false;

void int_handler()
{
    int_flag    = true;
}

void wait_transfer_done( void )
{
    while ( !int_flag )
        ;

    bridge.clear_interrupt();
    int_flag    = false;
}

void hardware_settings( void )
{
    int_line.mode( PullUp );
    int_line.fall( &int_handler );
    i2c.frequency( I2C_FREQUENCY );
}

int main()
{
    printf( "SC18IS606 Hello\r\n" );

    hardware_settings();
    bridge.install_wait_func( wait_transfer_done );

    char    snd_data[ TRANSFER_LENGTH ];
    char    rcv_data[ RCV_LENGTH ];
    char    write_enable    = 0x06;

    snd_data[ 0 ]   = 0x02;

    int count   = 0;

    while ( true ) {
        for ( int d = 0; d < READ_DATA_LENGTH; d += WRITE_DATA_LENGTH ) {
            snd_data[ 1 ]   = d;
            for ( int i = 0; i < WRITE_DATA_LENGTH; i++ ) {
                snd_data[ i + 2 ]  = (count & 0x1) ? ~(d + i) : (d + i);
            }

            bridge.transfer( SLAVE_SELECT_NUM, &write_enable, sizeof( write_enable ) );
            bridge.transfer( SLAVE_SELECT_NUM, snd_data, sizeof( snd_data ) );

            wait_ms(1);
        }

        rcv_data[ 0 ]  = 0x03;
        rcv_data[ 1 ]  = 0x00;

        bridge.transfer( SLAVE_SELECT_NUM, rcv_data, sizeof( rcv_data ) );
        bridge.read_buffer( rcv_data, sizeof( rcv_data ) );

        data_check( rcv_data + 2, sizeof( rcv_data ) - 2 );
        count++;
    }
}


void data_check( char *data, int length )
{
    printf( "\r\ndata_check:" );
    for ( int i = 0; i < length; i++ ) {
        if ( !(i % 16) )
            printf( "\r\n %02X :", i );
        printf( " %02X", data[ i ] );
    }
}
