/*
 *  SC18IS606 library
 *
 *  @author     Akifumi (Tedd) OKANO, NXP Semiconductors
 *  @version    0.2
 *  @date       28-July-2021
 *
 *  SC18IS606 is an "I2C-bus to SPI bridge"
 *  http://www.nxp.com/ (product infomation page will be updated later)
 */

#include    "mbed.h"

#ifndef        MBED_SC18IS606
#define        MBED_SC18IS606

/** SC18IS606 class
 *
 *  This is a driver code for the SC18IS606:  *
 *  Example:
 *  @code
 *  #include "mbed.h"
 *  #include "SC18IS606.h"
 *  
 *  I2C         i2c( p28, p27 );
 *  InterruptIn int_line( p21 );
 *  SC18IS606   bridge( i2c );      //  make a SC18IS606 instance as "bridge"
 *  
 *  #define I2C_FREQUENCY       (400 * 1000)    // Hz
 *  #define SLAVE_SELECT_NUM    0
 *  #define DATA_LENGTH         256
 *  
 *  void    data_check( char *data, int length );
 *  
 *  volatile int    int_flag    = false;
 *  
 *  void int_handler()
 *  {
 *      int_flag    = true;
 *  }
 *  
 *  void wait_transfer_done( void )
 *  {
 *      while ( !int_flag )
 *          ;
 *  
 *      bridge.clear_interrupt();
 *      int_flag    = false;
 *  }
 *  
 *  void hardware_settings( void )
 *  {
 *      int_line.mode( PullUp );
 *      int_line.fall( &int_handler );
 *      i2c.frequency( I2C_FREQUENCY );
 *  }
 *  
 *  int main()
 *  {
 *      printf( "SC18IS606 Hello\r\n" );
 *  
 *      hardware_settings();
 *      bridge.install_wait_func( wait_transfer_done );
 *  
 *      char    snd_data[ DATA_LENGTH ];
 *      char    rcv_data[ DATA_LENGTH ];
 *  
 *      for ( int i = 0; i < DATA_LENGTH; i++ ) {
 *          snd_data[ i ]  = i;
 *      }
 *  
 *      while(1) {
 *          bridge.transfer( SLAVE_SELECT_NUM, snd_data, sizeof( snd_data ) );
 *          bridge.read_buffer( rcv_data, sizeof( rcv_data ) );
 *          data_check( rcv_data, DATA_LENGTH );
 *      }
 *  }
 *  
 *  void data_check( char *data, int length )
 *  {
 *      for ( int i = 0; i < length; i++ ) {
 *          if ( !(i % 16) )
 *              printf( "\r\n %02X :", i );
 *          printf( " %02X", data[ i ] );
 *      }
 *  }
 *  @endcode
 */

class SC18IS606
{
public:

    /** Function IDs */
    typedef enum {
        SPI_read_and_write      = 0x00,
        Configure_SPI_Interface = 0xF0,
        Clear_Interrupt,
        Idle_mode,
        GPIO_Write              = 0xF4,
        GPIO_Read,
        GPIO_Enable,
        GPIO_Configuration,
        Read_Version            = 0xFE
    }
    FunctionID;

    /** Error Code */
    typedef enum {
        NO_ERROR    = 0x00,
    }
    ErrorCode;

    /** Create a SC18IS606 instance connected to specified I2C pins
     *
     *  @param sda          I2C-bus SDA pin
     *  @param scl          I2C-bus SCL pin
     *  @param i2c_address  I2C slave address (option, default = 0x50)
     */
    SC18IS606( PinName sda, PinName scl, char i2c_address = DEFAULT_I2C_SLAVE_ADDRESS );

    /** Create a SC18IS606 instance connected to specified I2C instance
     *
     *  @param i2c          I2C object (instance)
     *  @param i2c_address  I2C slave address (option, default = 0x50)
     */
    SC18IS606( I2C &i2c, char i2c_address = DEFAULT_I2C_SLAVE_ADDRESS );

    /** Destractor
     */
    ~SC18IS606();

    /** Installing "wait function" for "SC18IS606::transfer()"
     *  "SC18IS606::transfer()" will callback install function to monitor SPI transfer completion
     *
     *  @param block        pointer to a function to monitor SPI transfer completion
     */
    void install_wait_func( void (*block)( void ) )
    {
        wait_transfer_completion    = block;
        }

    /** Transfer (execute SPI transfer)
     *
     *  @param slave_select_num SPI slave select number (0 ~ 2)
     *  @param send_data_ptr    Send_data_ptr
     *  @param length           Length of data array
     *  @return                 I2C error state
     */
    int transfer( int slave_select_num, char *send_data_ptr, int length );

    /** Read buffer (reading out received data from buffer)
     *
     *  @param receive_data_ptr Receive_data_ptr
     *  @param length           Length of data array
     *  @return                 I2C error state
     */
    int read_buffer( char *receive_data_ptr, int length );

    /** Set congiguration
     *
     *  @param data     Donfig data byte
     *  @return         I2C error state
     */
    int     config( FunctionID fid, char data );

    /** Clear interrupt
     *
     *  @return         I2C error state
     */
    int     clear_interrupt( void );

    /** Read version
     *
     *  @return         version info in string
     */
    char* read_version( void );

private:

    enum {
        DEFAULT_I2C_SLAVE_ADDRESS   = 0x50
    };

    int     init( void);

    I2C     *i2c_p;
    I2C     &i2c;
    char    device_address;
    void    (*wait_transfer_completion)( void );
};

#endif  // end of "#ifndef MBED_SC18IS606"
