/*
 *  SC18IS606 library
 *
 *  @author     Akifumi (Tedd) OKANO, NXP Semiconductors
 *  @version    0.1
 *  @date       13-July-2021
 *
 *  SC18IS606 is an "I2C-bus to SPI bridge"
 *  http://www.nxp.com/ (product infomation page will be updated later)
 */

#include    "mbed.h"
#include    "SC18IS606.h"

SC18IS606::SC18IS606( PinName sda, PinName scl, char i2c_address )
    : i2c_p( new I2C( sda, scl ) ), i2c( *i2c_p ), device_address( i2c_address )
{
    init();
}

SC18IS606::SC18IS606( I2C &i2c_, char i2c_address )
    : i2c_p( NULL ), i2c( i2c_ ), device_address( i2c_address )
{
    init();
}

SC18IS606::~SC18IS606()
{
    if ( NULL != i2c_p )
        delete  i2c_p;
}

int SC18IS606::init( void )
{
    int err;
    
    wait_transfer_completion    = NULL;
    err = clear_interrupt();
    
    return err;
}

int SC18IS606::transfer( int slave_select_num, char *send_data_ptr, int length )
{
    char    *p;
    int     err;

    p   = new char[ length + 1 ];
    
    *p  = SPI_read_and_write | (0x1 << slave_select_num);
    memcpy( p + 1, send_data_ptr, length );
    err = i2c.write( device_address, p, length + 1 );
    delete[]    p;
    
    if ( err )
        return err;
    
    if ( NULL != wait_transfer_completion )
        (*wait_transfer_completion)();

    return err;
}

int SC18IS606::read_buffer( char *receive_data_ptr, int length )
{
    int     err;

    if ( receive_data_ptr )
        err = i2c.read( device_address, receive_data_ptr, length );
    
    return err;
}

int SC18IS606::config( FunctionID fid, char data )
{
    char    s[ 2 ];
    int     err;

    s[ 0 ]  = fid;
    s[ 1 ]  = data;
    err = i2c.write( device_address, s, sizeof( s ) );
    
    return err;
}

int SC18IS606::clear_interrupt( void )
{
    char    c   = Clear_Interrupt;
    int     err;
    
    err = i2c.write( device_address, &c, sizeof( c ) );    

    return err;
}

char* SC18IS606::read_version( void )
{
    static char s[ 16 ];
    char        func_id = Read_Version;
    int         err;
    
    err  = i2c.write( device_address, &func_id, sizeof( func_id ) );

    if ( NULL != wait_transfer_completion )
        (*wait_transfer_completion)();

    err |= read_buffer( s, sizeof( s ) );
    
    if ( err )
        s[ 0 ]  = 0;
    
    return s;
}

