/*
 *  LCD module "SB1602E" test application
 *
 *  SB1602E is an I2C based low voltage text LCD panel (based Sitronix ST7032 chip)
 *  The module by StrawberryLinux
 *  http://strawberry-linux.com/catalog/items?code=27002 (Online shop page (Japanese))
 *  http://strawberry-linux.com/pub/ST7032i.pdf (datasheet of the chip)
 *
 *  Copyright (c) 2010 Tedd OKANO
 *  Released under the MIT License: http://mbed.org/license/mit
 *
 *  revision 1.0  22-Jan-2010   a. 1st release
 *  revision 1.1  23-Jan-2010   a. class and app name has been changed from lcd_SB1602E to TextLCD_SB1602E
 *                              b. printf() test added
 *                              c. copyright notice added
 */

#include    "mbed.h"
#include    "SB1602E.h"

#define     TEST_LOOP                    1
#define     DEFAULT_TIME_INTERVAL        1.0
#define     DEFAULT_TIME_INTERVAL_PUTx   0.2

#define     CONTRAST_ADJUST_TEST
#define     PRINTF_TEST
#define     CGRAM_FUNCTION_TEST
#define     PUTS_TEST_WITH_ESCAPE
#define     PUTS_TEST_WITHOUT_ESCAPE
#define     PUTC_TEST
//#define     ICON_TEST  (This has not been tested by Okano, sorry)

Serial  pc( USBTX, USBRX ); // tx, rx
SB1602E lcd( p9, p10 );     //  SDA, SCL
//SB1602E lcd( p28, p27 );  //  SDA, SCL

static char    cg[8][8]        =    {// I hope I can improve this bitmap sometime in future (^^;
    { 0x1E, 0x1F, 0x1F, 0x1D, 0x1C, 0x1C, 0x1C, 0x1C },
    { 0x01, 0x02, 0x13, 0x1B, 0x1F, 0x1F, 0x0F, 0x15 },
    { 0x19, 0x1D, 0x1F, 0x0F, 0x1F, 0x0F, 0x1D, 0x18 },
    { 0x1F, 0x1D, 0x16, 0x1E, 0x17, 0x1F, 0x1A, 0x1C },
    { 0x1E, 0x1F, 0x07, 0x07, 0x1F, 0x1E, 0x00, 0x00 },
    { 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55 },
    { 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11 },
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
};

void lcd_test( void );
void set_lines_fixed_value( char line, char value );
void block_pattaern( char direction, char polarity );


int main() {
    lcd.puts( 0, "TextLCD SB1602E \r" );

    for ( int i = 10; i > 0; i-- ) {
        lcd.printf( 1, "  count down: %2d\r", i );
        wait( 1.0 );
    }

    do {
        lcd_test();
    } while ( TEST_LOOP );
}

void lcd_test( void ) {
    char    s[ 18 ];
    int        i;

    pc.printf( "LCD test runing\n" );
    lcd.contrast( DEFAULT_CONTRAST );

#ifdef CONTRAST_ADJUST_TEST
    pc.printf( "  CONTRAST_ADJUST_TEST\n" );
    lcd.clear();
    lcd.puts( 1, "\x07\x07\x07\x07\x07 contrast()\r" );
    wait( DEFAULT_TIME_INTERVAL );

    lcd.puts( 0, "ABCDEFGHIJKLMNOP\r" );

    for ( int t = 0; t < 2; t++ ) {
        for ( i = 0; i <= 0x3F; i++ ) {

            lcd.printf( 1, "  cntrst = 0x%02X\r", i );
            lcd.contrast( i );
            wait ( 0.02 );
        }

        for ( i = 0x3E; i > 0 ; i-- ) {
            sprintf( s, "  cntrst = 0x%02X\r", i );
            lcd.puts( 1, s );
            lcd.contrast( i );
            wait ( 0.02 );
        }
    }

    lcd.contrast( DEFAULT_CONTRAST );

#endif
#ifdef PRINTF_TEST

    pc.printf( "  PRINTF_TEST\n" );
    lcd.clear();
    lcd.puts( 1, "\x07\x07\x07\x07\x07\x07\x07 printf()\r" );
    wait( DEFAULT_TIME_INTERVAL );

    lcd.printf( 0, "%%6.4f, %%c, %%X\"\r" );
    for ( i = -37; i < 37; i++ ) {
        lcd.printf( 1, "%6.4f, %c, %x\r", (float)i / 37.0, 'a' + i, i + 37 );
        wait ( 0.1 );
    }

#endif
#ifdef CGRAM_FUNCTION_TEST
    pc.printf( "  CGRAM_FUNCTION_TEST\n" );
    lcd.clear();
    lcd.puts( 1, "\x07\x07\x07\x07 set_CGRAM()\r" );
    wait( DEFAULT_TIME_INTERVAL );

    for ( i = 0; i < 8; i++ )
        lcd.put_custom_char( i, cg[ i ], i, 0 );

    for ( i = 0; i < 8; i++ )
        lcd.put_custom_char( i, cg[ i ], i + 8, 0 );

    for ( i = 0; i < 8; i++ )
        lcd.put_custom_char( i, cg[ i ], i, 1 );

    for ( i = 0; i < 8; i++ )
        lcd.put_custom_char( i, cg[ i ], i + 8, 1 );

    wait( DEFAULT_TIME_INTERVAL );
    lcd.clear();
    lcd.set_CGRAM( 7, '\x1F' );

    lcd.set_CGRAM( 0, '\0' );
    lcd.set_CGRAM( 1, '\0' );
    lcd.set_CGRAM( 0, '\0' );
    lcd.set_CGRAM( 1, '\0' );
    set_lines_fixed_value( 0, 1 );
    set_lines_fixed_value( 1, 0 );

    block_pattaern( 0, 0 );
    block_pattaern( 0, 1 );
    block_pattaern( 1, 0 );
    block_pattaern( 1, 1 );


#endif
#ifdef PUTS_TEST_WITH_ESCAPE
    pc.printf( "  PUTS_TEST_WITH_ESCAPE\n" );
    lcd.clear();
    lcd.puts( 1, "\x07\x07\x07 puts(\"..\\r\")\r" );
    wait( DEFAULT_TIME_INTERVAL );

    lcd.puts( 0, "\rWelcome to mbed!\r" );
    lcd.puts( 1, "\rwww.mbed.com\r" );
    wait( DEFAULT_TIME_INTERVAL );
    lcd.puts( 0, "       This is a\r" );
    lcd.puts( 1, "        LCD test\r" );
    wait( DEFAULT_TIME_INTERVAL );
    lcd.puts( 0, "  22-Jan-2010\r" );
    lcd.puts( 1, "  Hello world!\r" );
    wait( DEFAULT_TIME_INTERVAL );

#endif
#ifdef PUTS_TEST_WITHOUT_ESCAPE
    pc.printf( "  PUTS_TEST_WITHOUT_ESCAPE\n" );
    lcd.clear();
    lcd.puts( 0, "\r" );
    lcd.puts( 1, "\x07\x07\x07\x07\x07 puts(\"..\")\r" );
    wait( DEFAULT_TIME_INTERVAL );

    lcd.puts( 0, "0" );
    lcd.puts( 1, "ABC" );
    wait( DEFAULT_TIME_INTERVAL_PUTx );
    lcd.puts( 0, "1" );
    lcd.puts( 1, "DEF" );
    wait( DEFAULT_TIME_INTERVAL_PUTx );
    lcd.puts( 0, "2" );
    lcd.puts( 1, "GHI" );
    wait( DEFAULT_TIME_INTERVAL_PUTx );
    lcd.puts( 0, "3" );
    lcd.puts( 1, "JKL" );
    wait( DEFAULT_TIME_INTERVAL_PUTx );
    lcd.puts( 0, "4" );
    lcd.puts( 1, "MNO" );
    wait( DEFAULT_TIME_INTERVAL_PUTx );
    lcd.puts( 0, "5" );
    lcd.puts( 1, "PQR" );
    wait( DEFAULT_TIME_INTERVAL_PUTx );
    lcd.puts( 0, "6" );
    wait( DEFAULT_TIME_INTERVAL_PUTx );
    lcd.puts( 0, "7" );
    wait( DEFAULT_TIME_INTERVAL_PUTx );
    lcd.puts( 0, "8" );
    wait( DEFAULT_TIME_INTERVAL_PUTx );
    lcd.puts( 0, "9" );
    wait( DEFAULT_TIME_INTERVAL_PUTx );
    lcd.puts( 0, "A" );
    wait( DEFAULT_TIME_INTERVAL_PUTx );
    lcd.puts( 0, "B" );
    wait( DEFAULT_TIME_INTERVAL_PUTx );
    lcd.puts( 0, "C" );
    wait( DEFAULT_TIME_INTERVAL_PUTx );
    lcd.puts( 0, "D" );
    wait( DEFAULT_TIME_INTERVAL_PUTx );
    lcd.puts( 0, "E" );
    wait( DEFAULT_TIME_INTERVAL_PUTx );
    lcd.puts( 0, "F" );
    wait( DEFAULT_TIME_INTERVAL_PUTx );

#endif
#ifdef PUTC_TEST
    pc.printf( "  PUTC_TEST\n" );
    lcd.clear();
    lcd.puts( 1, "\x07\x07\x07\x07\x07\x07\x07\x07\x07 putc()\r" );
    wait( DEFAULT_TIME_INTERVAL );

    for ( i = 0; i < 16; i++ ) {
        lcd.putc( 0, '0' + i );
        lcd.putc( 1, 'A' + i );
        wait( DEFAULT_TIME_INTERVAL_PUTx );
    }
#endif
#ifdef ICON_TEST
    pc.printf( "  ICON_TEST\n" );
    lcd.clear();
    lcd.puts( 1, "\x07\x07\x07\x07\x07\x07 puticon()\r" );
    unsigned short flg = 0;
    for (i = 0; i < 13; i++) {
        flg |= (1 << i);
        lcd.puticon(flg);
        wait( DEFAULT_TIME_INTERVAL_PUTx );
    }
    wait( DEFAULT_TIME_INTERVAL );
    lcd.puticon(0);
#endif
}

void set_lines_fixed_value( char line, char value ) {
    for ( int i = 0; i < 16; i++ )
        lcd.putc( line, value );

    lcd.puts( line, "\r" );
}


void block_pattaern( char direction, char polarity ) {
    char    c[ 8 ];

    for ( int x = 0; x < 2; x++ ) {
        for ( int i = 0; i < 8; i++ ) {
            for ( int j = 0; j < 8; j++ ) {
                if ( j <= i )
                    c[ (direction ? j : 7 - j) ]    = (polarity ?  0x00: ~0x00 );
                else
                    c[ (direction ? j : 7 - j) ]    = (polarity ? ~0x00 : 0x00 );
            }

            lcd.set_CGRAM( (x ? (direction ? 0 : 1) : (direction ? 1 : 0)), c );
            wait( 0.05 );
        }
    }
}

