#ifndef MBED_MARY_CAMERA
#define MBED_MARY_CAMERA

#define SPI_FREQUENCY       6000000

/** MARY_CAMERA class
 *
 *  MARY_CAMERA driver library
 *  This driver has been made to get camera data from MARY CAMERA.
 *
 *  Example:
 *  @code
 *
 *  #include "mbed.h"
 *  #include "MARMEX_OB_oled.h"
 *  #include "MARY_CAMERA.h"
 *
 *  MARMEX_OB_oled  oled1( p5, p7,  p20, p16, p15 ); // mosi, sclk, cs, rst, power_control     -- maple-mini-type-b-slot1
 *  MARY_CAMERA     camera( p5, p6, p7, p22, p26, p28, p27 ); // mosi, miso, sclk, cs, reset, I2C_SDA, I2C_SCL
 *  BusOut          led( LED3, LED4 );
 *
 *  #define X_OFFSET        ((MARY_CAMERA::PIXEL_PER_LINE - MARMEX_OB_oled::WIDTH ) / 2)
 *  #define Y_OFFSET        ((MARY_CAMERA::LINE_PER_FRAME - MARMEX_OB_oled::HEIGHT) / 2)
 *
 *  int main()
 *  {
 *      led    = 0x3;
 *
 *      oled1.cls();
 *
 *      short   buf[ MARMEX_OB_oled::WIDTH ];
 *
 *      while ( 1 ) {
 *
 *          led    = 0x1;
 *          camera.open_transfer();
 *
 *          for ( int line = 0; line < 128; line++  ) {
 *              camera.transfer_a_line( buf, line + Y_OFFSET, X_OFFSET, 128 );
 *              oled1.blit565( 0, line, 128, 1, buf );
 *          }
 *
 *          camera.close_transfer();
 *          led    = 0x2;
 *     }
 *  }
 *  @endcode
 */


#define  DEFAULT_PICTURE_SIZE   QCIF
//#define  DEFAULT_PICTURE_SIZE   VGA
//#define  DEFAULT_PICTURE_SIZE   QVGA
//#define  DEFAULT_PICTURE_SIZE   QQVGA

class MARY_CAMERA
{
public:
    typedef enum  {
        QCIF    = 1,            /**< QCIF       */
        VGA,                    /**< VGA        */
        QVGA,                   /**< QVGA       */
        QQVGA,                  /**< QQVGA      */
    } CameraResolution;

    typedef enum  {
        OFF    = 0,             /**< ON         */
        ON,                     /**< OFF        */
    } SwitchState;

    /** General parameters for MARY_CAMERA */
    enum  {
        BYTE_PER_PIXEL  = 2,    /**< bytes per pixel  */

        QCIF_PIXEL_PER_LINE     = 176,  /**< pixels in a line  */
        QCIF_LINE_PER_FRAME     = 144,  /**< pixels in a line  */
        VGA_PIXEL_PER_LINE      = 640,  /**< pixels in a line  */
        VGA_LINE_PER_FRAME      = 480,  /**< pixels in a line  */
    };

    /** General parameters for MARMEX_OB_oled */
    enum  {
        NO_ERROR        = 0     /**< zero  */
    };

    /** Create a MARY_CAMERA instance connected to specified SPI, DigitalOut and I2C pins
     *
     * @param SPI_mosi   SPI-bus MOSI pin
     * @param SPI_miso   SPI-bus MISO pin
     * @param SPI_sclk   SPI-bus SCLK pin
     * @param SPI_cs     SPI-bus Chip Select pin
     * @param cam_reset  Camera reset pin
     * @param I2C_sda    I2C-bus SDA pin
     * @param I2C_scl    I2C-bus SCL pin
     */

    MARY_CAMERA(
        PinName SPI_mosi,
        PinName SPI_miso,
        PinName SPI_sck,
        PinName SPI_cs,
        PinName cam_reset,
        PinName I2C_sda,
        PinName I2C_scl
    );

    /** Initialization
     *
     *  Performs MARY_CAMERA reset and initializations
     *  This function is called from MARY_CAMERA constoructor. So user don't have to call in the user code.
     *
     *  This function takes about 2 seconds because there is 99 times I2C access with 20ms interval.
     *
     *  @param res   select camera resolution : QCIF(default), VGA, QVGA or QQVGA
     */
    int init( CameraResolution res = QCIF );

    /** Color bar ON/OFF
     *
     *  Set colorbar ON/OFF
     *
     *  @param sw   turn-ON or -OFF colorbar : ON or OFF
     */
    void colorbar( SwitchState sw );

    /** Get holizontal size
     *
     *  Returns image horizontal size (pixels)
     *
     *  @return holizontal size
     */
    int horizontal_size( void );
    
    /** Get vertical size
     *
     *  Returns image vertical size (pixels)
     *
     *  @return vertical size
     */
    int vertical_size( void );

    /** Check camera availability
     *
     *  Returns last I2C access result
     *  This returns non-zero value id the camera initialization failed
     *
     *  This function takes about 2 seconds because there is 99 times I2C access with 20ms interval.
     *
     *  @return error code in init function (I2C API return value)
     */
    int ready( void );

    /** Open transfer
     *
     *  Let the MARY_CAMERA get ready to transfer the data.
     *  When this function is called, the camera will stop updating buffer at end of frame.
     *
     *  @return error code in init function (I2C API return value)
     */
    void open_transfer( void );

    /** Close transfer
     *
     *  Letting the MARY_CAMERA to know the data transfer done.
     *  This function should be called when the data transfer done to resume the buffer update by camera
     */
    void close_transfer( void );

    /** Transfer a line
     *
     *  Reads 1 line data from MARY_CAMERA
     *  This function should be called when the data transfer done to resume the buffer update by camera
     *
     * @param *p            pointer to array of short
     * @param line_number   to select which line want to read
     * @param x_offset      holizontal offset (from left) to start the read
     * @param n_of_pixels   pixels to be read
     */
    void transfer_a_line( short *p, int line_number, int x_offset, int n_of_pixels );

private:
    int     send_spi( char data );
    void    write_register( char reg, char value );
    int     read_register( char reg );
    void    set_address( int address );

    SPI         _spi;
    DigitalOut  _cs;
    DigitalOut  _reset;
    I2C         _i2c;
    int         _error_state;
    int         _horizontal_size;
    int         _vertical_size;
};

#endif  //  MBED_MARY_CAMERA