#include    "mbed.h"
#include    "MARY_CAMERA.h"


#define CAM_I2C_ADDR    0x42

#define     COMMAND_WRITE                   0x00
#define     COMMAND_READ                    0x80
#define     COMMAND_ADDR_INCREMENT          0x20

#define     MEMORY_ADDR_LOW__REGISTER       0x0
#define     MEMORY_ADDR_MID__REGISTER       0x1
#define     MEMORY_ADDR_HIGH_REGISTER       0x2
#define     CAMERA_DATA_REGISTER            0x8
#define     CONTROL_DATA_REGISTER           0x3
#define     STATUS_REGISTER                 0x4

#define     CONTROL__PAUSE_BUFFER_UPDATE    0x01
#define     CONTROL__RESUME_BUFFER_UPDATE   0x00


MARY_CAMERA::MARY_CAMERA(
    PinName SPI_mosi,
    PinName SPI_miso,
    PinName SPI_sck,
    PinName SPI_cs,
    PinName cam_reset,
    PinName I2C_sda,
    PinName I2C_scl
) :
    _spi( SPI_mosi, SPI_miso, SPI_sck ),
    _cs( SPI_cs ),
    _reset( cam_reset ),
    _i2c( I2C_sda, I2C_scl )
{
#ifdef IGNORE_INITIALIZATION_ERROR
    init();
#else
    if ( 0 != init() )
        error( "camera initialization failed." );
#endif
}

int MARY_CAMERA::init( CameraResolution res )
{
#define     PARAM_NUM               99
#define     RES_CHANGE_PARAM_NUM    12
#define     RESET_PULSE_WIDTH       100     //  mili-seconds
#define     RESET_RECOVERY_TIME     100     //  mili-seconds
#define     COMMAND_INTERVAL        20      //  mili-seconds

    char camera_register_setting[ PARAM_NUM ][ 2 ] = {
        { 0x01, 0x40 }, { 0x02, 0x60 }, { 0x03, 0x02 }, { 0x0C, 0x0C },
//      { 0x0E, 0x61 }, { 0x0F, 0x4B }, { 0x11, 0x81 }, { 0x12, 0x04 },
//      { 0x0E, 0x61 }, { 0x0F, 0x4B }, { 0x11, 0xC0 }, { 0x12, 0x04 },
        { 0x0E, 0x61 }, { 0x0F, 0x4B }, { 0x11, 0x80 }, { 0x12, 0x04 },
        { 0x15, 0x00 }, { 0x16, 0x02 }, { 0x17, 0x39 }, { 0x18, 0x03 },
        { 0x19, 0x03 }, { 0x1A, 0x7B }, { 0x1E, 0x37 }, { 0x21, 0x02 },
        { 0x22, 0x91 }, { 0x29, 0x07 }, { 0x32, 0x80 }, { 0x33, 0x0B },
        { 0x34, 0x11 }, { 0x35, 0x0B }, { 0x37, 0x1D }, { 0x38, 0x71 },
        { 0x39, 0x2A }, { 0x3B, 0x12 }, { 0x3C, 0x78 }, { 0x3D, 0xC3 },
        { 0x3E, 0x11 }, { 0x3F, 0x00 }, { 0x40, 0xD0 }, { 0x41, 0x08 },
        { 0x41, 0x38 }, { 0x43, 0x0A }, { 0x44, 0xF0 }, { 0x45, 0x34 },
        { 0x46, 0x58 }, { 0x47, 0x28 }, { 0x48, 0x3A }, { 0x4B, 0x09 },
        { 0x4C, 0x00 }, { 0x4D, 0x40 }, { 0x4E, 0x20 }, { 0x4F, 0x80 },
        { 0x50, 0x80 }, { 0x51, 0x00 }, { 0x52, 0x22 }, { 0x53, 0x5E },
        { 0x54, 0x80 }, { 0x56, 0x40 }, { 0x58, 0x9E }, { 0x59, 0x88 },
        { 0x5A, 0x88 }, { 0x5B, 0x44 }, { 0x5C, 0x67 }, { 0x5D, 0x49 },
        { 0x5E, 0x0E }, { 0x69, 0x00 }, { 0x6A, 0x40 }, { 0x6B, 0x0A },
        { 0x6C, 0x0A }, { 0x6D, 0x55 }, { 0x6E, 0x11 }, { 0x6F, 0x9F },
        { 0x70, 0x3A }, { 0x71, 0x35 }, { 0x72, 0x11 }, { 0x73, 0xF1 },
        { 0x74, 0x10 }, { 0x75, 0x05 }, { 0x76, 0xE1 }, { 0x77, 0x01 },
        { 0x78, 0x04 }, { 0x79, 0x01 }, { 0x8D, 0x4F }, { 0x8E, 0x00 },
        { 0x8F, 0x00 }, { 0x90, 0x00 }, { 0x91, 0x00 }, { 0x96, 0x00 },
        { 0x96, 0x00 }, { 0x97, 0x30 }, { 0x98, 0x20 }, { 0x99, 0x30 },
        { 0x9A, 0x00 }, { 0x9A, 0x84 }, { 0x9B, 0x29 }, { 0x9C, 0x03 },
        { 0x9D, 0x4C }, { 0x9E, 0x3F }, { 0xA2, 0x52 }, { 0xA4, 0x88 },
        { 0xB0, 0x84 }, { 0xB1, 0x0C }, { 0xB2, 0x0E }, { 0xB3, 0x82 },
        { 0xB8, 0x0A }, { 0xC8, 0xF0 }, { 0xC9, 0x60 },
    };
    const char  res_change_param[ 5 ][ RES_CHANGE_PARAM_NUM ] = {
        { 0x17, 0x18, 0x32, 0x19, 0x1a, 0x03, 0x0c, 0x3e, 0x71, 0x72, 0x73, 0xa2 }, //  register addr
        { 0x39, 0x03, 0x80, 0x03, 0x7b, 0x02, 0x0c, 0x11, 0x35, 0x11, 0xf1, 0x52 }, //  QSIF
        { 0x13, 0x01, 0xb6, 0x02, 0x7a, 0x0a, 0x00, 0x00, 0x35, 0x11, 0xf0, 0x02 }, //  VGA
        { 0x16, 0x04, 0x80, 0x02, 0x7a, 0x0a, 0x04, 0x19, 0x35, 0x11, 0xf1, 0x02 }, //  QVGA
        { 0x16, 0x04, 0xa4, 0x02, 0x7a, 0x0a, 0x04, 0x1a, 0x35, 0x22, 0xf2, 0x02 }, //  QQVGA
    };
    const char camera_reset_command[] = { 0x12, 0x80 };

    //  SPI settings

    _cs     = 1;                        //  set ChipSelect signal HIGH
    _spi.format( 8 );                   //  camera SPI : 8bits/transfer
    _spi.frequency( SPI_FREQUENCY );    //  SPI frequency setting

    //  reset

    _reset = 0;
    wait_ms( RESET_PULSE_WIDTH );   //  assert RESET signal
    _reset = 1;
    wait_ms( RESET_RECOVERY_TIME ); //  deassert RESET signal

    if ( 0 != (_error_state = _i2c.write( CAM_I2C_ADDR, camera_reset_command, 2 )) )
        return _error_state;    //  return non-zero if I2C access failed

    wait_ms( 100 ); //  deassert RESET signal

    _horizontal_size    = QCIF_PIXEL_PER_LINE;
    _vertical_size      = QCIF_LINE_PER_FRAME;

#ifdef UNIFIED_RESOLUTION_CHANGE
    if ( QCIF != res ) {
        for ( int i = 0; i < RES_CHANGE_PARAM_NUM; i++ ) {
            for ( int j = 0; j < PARAM_NUM; j++ ) {
                if ( camera_register_setting[ j ][ 0 ] == res_change_param[ 0 ][ i ] ) {
                    camera_register_setting[ j ][ 1 ]   = res_change_param[ res ][ i ];
                }
            }
        }
    }

    switch ( res ) {
        case QCIF:
            _horizontal_size    = QCIF_PIXEL_PER_LINE;
            _vertical_size      = QCIF_LINE_PER_FRAME;
            break;
        case VGA:
            _horizontal_size    = VGA_PIXEL_PER_LINE;
            _vertical_size      = VGA_LINE_PER_FRAME;
            break;
        case QVGA:
            _horizontal_size    = VGA_PIXEL_PER_LINE / 2;
            _vertical_size      = VGA_LINE_PER_FRAME / 2;
            break;
        case QQVGA:
            _horizontal_size    = VGA_PIXEL_PER_LINE / 4;
            _vertical_size      = VGA_LINE_PER_FRAME / 4;
            break;
    }

#endif

    for ( int i = 0; i < PARAM_NUM; i++ ) {
        if ( 0 != (_error_state = _i2c.write( CAM_I2C_ADDR, camera_register_setting[ i ], 2 )) )
            break;

        wait_ms( COMMAND_INTERVAL );  //  camera register writing requires this interval
    }

#ifndef UNIFIED_RESOLUTION_CHANGE
    if ( QCIF != res ) {
        char    d[ 2 ];
        for ( int i = 0; i < RES_CHANGE_PARAM_NUM; i++ ) {
            d[ 0 ]  = res_change_param[  0  ][ i ];
            d[ 1 ]  = res_change_param[ res ][ i ];

            if ( 0 != (_error_state = _i2c.write( CAM_I2C_ADDR, d, 2 )) )
                break;

            wait_ms( COMMAND_INTERVAL );  //  camera register writing requires this interval
        }
    }

    switch ( res ) {
        case QCIF:
            _horizontal_size    = QCIF_PIXEL_PER_LINE;
            _vertical_size      = QCIF_LINE_PER_FRAME;
            break;
        case VGA:
            _horizontal_size    = VGA_PIXEL_PER_LINE;
            _vertical_size      = VGA_LINE_PER_FRAME;
            break;
        case QVGA:
            _horizontal_size    = VGA_PIXEL_PER_LINE / 2;
            _vertical_size      = VGA_LINE_PER_FRAME / 2;
            break;
        case QQVGA:
            _horizontal_size    = VGA_PIXEL_PER_LINE / 4;
            _vertical_size      = VGA_LINE_PER_FRAME / 4;
            break;
    }

printf( "resolution : h,v = %d,%d\r\n", _horizontal_size, _vertical_size );
#endif

    return _error_state;    //  return non-zero if I2C access failed
}

void MARY_CAMERA::colorbar( SwitchState sw )
{
    char    s[ 2 ];

    s[ 0 ]  = 0x12;
    s[ 1 ]  = sw ? 0x06 : 0x04;

    _error_state = _i2c.write( CAM_I2C_ADDR, s, 2 );
}

int MARY_CAMERA::horizontal_size( void )
{
    return _horizontal_size;    //  return last state of I2C access
}

int MARY_CAMERA::vertical_size( void )
{
    return _vertical_size;    //  return last state of I2C access
}

int MARY_CAMERA::ready( void )
{
    return _error_state;    //  return last state of I2C access
}

extern int read_order_change;

void MARY_CAMERA::transfer_a_line( short *p, int line_number, int x_offset, int n_of_pixels )
{
#if 0

    char    tmp;

    if ( line_number < 0 )
        return;

    //  set camera module's buffer address
    set_address( line_number * horizontal_size() * BYTE_PER_PIXEL + x_offset * BYTE_PER_PIXEL );

    //  put a read command, first return byte should be ignored
    read_register( CAMERA_DATA_REGISTER );

    for( int x = 0; x < n_of_pixels; x++ ) {
        //  perform 2 bytes read. a pixel data is in RGB565 format (16bits)
        tmp     = read_register( CAMERA_DATA_REGISTER );                //  read lower byte
        *p++    = (read_register( CAMERA_DATA_REGISTER ) << 8) | tmp;   //  read upper byte
    }

#else


    short    tmp;

    if ( line_number < 0 )
        return;

    //  set camera module's buffer address
    set_address( line_number * horizontal_size() * BYTE_PER_PIXEL + x_offset * BYTE_PER_PIXEL );

    //  put a read command, first return byte should be ignored
    read_register( CAMERA_DATA_REGISTER );


    if ( read_order_change ) {

        read_register( CAMERA_DATA_REGISTER );

        for( int x = 0; x < n_of_pixels; x++ ) {
            //  perform 2 bytes read. a pixel data is in RGB565 format (16bits)
            tmp     = read_register( CAMERA_DATA_REGISTER ) << 8;                //  read lower byte
            *p++    = (read_register( CAMERA_DATA_REGISTER ) << 0) | tmp;   //  read upper byte
        }

    } else {

        for( int x = 0; x < n_of_pixels; x++ ) {
            //  perform 2 bytes read. a pixel data is in RGB565 format (16bits)
            tmp     = read_register( CAMERA_DATA_REGISTER );                //  read lower byte
            *p++    = (read_register( CAMERA_DATA_REGISTER ) << 8) | tmp;   //  read upper byte
        }

    }

#endif
}

void MARY_CAMERA::open_transfer( void )
{
    //  send command to pause the camera buffer update
    write_register( CONTROL_DATA_REGISTER, CONTROL__PAUSE_BUFFER_UPDATE );

    //  read status register (first return byte should be ignored)
    read_register( STATUS_REGISTER );

    //  wait until the status register become 0x51(ready to transfer data)
    while ( 0x51 != read_register( STATUS_REGISTER ) )
        ;
}

void MARY_CAMERA::close_transfer( void )
{
    //  send command to resume the camera buffer update
    write_register( CONTROL_DATA_REGISTER, CONTROL__RESUME_BUFFER_UPDATE );

    //  read status register (first return byte should be ignored)
    read_register( STATUS_REGISTER );

    //  wait until the status register become 0x50(camera updating the buffer)
    while ( 0x50 != read_register( STATUS_REGISTER ) )
        ;
}

void MARY_CAMERA::set_address( int address )
{
    //  set memory address (3 bytes)

    write_register( MEMORY_ADDR_LOW__REGISTER, (address >>  0) & 0xFF );
    write_register( MEMORY_ADDR_MID__REGISTER, (address >>  8) & 0xFF );
    write_register( MEMORY_ADDR_HIGH_REGISTER, (address >> 16) & 0xFF );
}

void MARY_CAMERA::write_register( char reg, char value )
{
    //  camera register write

    send_spi( COMMAND_WRITE | reg );    //  send command and register number
    send_spi( value );                  //  send register value
}

int MARY_CAMERA::read_register( char reg )
{
    //  camera register read
    //  returning current data in SPI buffer (data returned by previous command)

    return ( send_spi( COMMAND_READ | reg | ((reg == CAMERA_DATA_REGISTER) ? COMMAND_ADDR_INCREMENT : 0x00) ) );
}

int MARY_CAMERA::send_spi( char data )
{
    int     tmp;

    //  SPI access

    _cs     = 0;
    tmp     = _spi.write( data );
    _cs     = 1;

    return ( tmp );
}
