/*
 *   NXP LM75B temperature sensor test
 *   www.nxp.com/pip/LM75B_2.html
 *   
 *   Expecting to use the pins 9 and 10 for I2C bus
 *   these pins should be pulled-up properly. 
 *
 *   The temperature read out will be shown on terminal on the PC screen. 
 *   
 *   In this demo code, two LM75B devices can be driven. 
 *   These two devices should have different I2C address setting 
 *   using its address pins (LM75B's A0 to A2 (pins 5 to 7)). 
 *   One LM75B should have all those pins tied to GND. 
 *   And another should have the pin A0(pin7) pulled-up. 
 *
 *   From the software, those devices can be accessed by I2C addresses 
 *   "0x90" and "0x92". 
 *   It will not be as "0x90" and "0x91" because the address has 
 *   7 bit only and stuffed to left. So the "A0" setting become 0xX2. 
 *   The LSB does not care because it will be set by I2C libraly when 
 *   it transfer the data for read and write. 
 */

#include "mbed.h"
//#include "SDFileSystem.h"

//  LM75B IIC address
#define    LM75B_ADDR 0x90

//  LM75B registers
#define    Conf        0x01
#define    Temp        0x00
#define    Tos         0x03
#define    Thyst       0x02


DigitalOut    led[]            = { LED4, LED3, LED2, LED1 };
Serial        pc(USBTX, USBRX); // tx, rx

I2C           i2c( p9, p10 );        // sda, scl


void   iic_write( char addr, char reg, char data );
char   iic_read( char addr, char reg );
short  iic_read_short( char addr, char reg );
void   iic_error( void );

void   init_temp_sensor( char dev_num );
float  get_temp( char dev_num );

int main() {
    int    i;
    
    init_temp_sensor( 0 );
    init_temp_sensor( 1 );

    while(1) {
        pc.printf( "  (%d)  temp[0]=%6.3f, temp[1]=%6.3f(degree-C)\n", i++, get_temp( 0 ), get_temp( 1 ) );
        wait( 1 );
    }
}

void init_temp_sensor( char dev_num )
{
    dev_num    <<= 1;
    
    iic_write( LM75B_ADDR | dev_num, Conf, 0x00 );    //    configuration
    pc.printf( "sensor[%d] : Conf  register read out = 0x%02X\n", dev_num, iic_read( LM75B_ADDR | dev_num, Conf ) );
    pc.printf( "sensor[%d] : Tos   register read out = 0x%04X\n", dev_num, iic_read_short( LM75B_ADDR | dev_num, Tos ) );
    pc.printf( "sensor[%d] : Thyst register read out = 0x%04X\n", dev_num, iic_read_short( LM75B_ADDR | dev_num, Thyst ) );
}


float get_temp( char dev_num )
{
    dev_num    <<= 1;

    return ( (float)(iic_read_short( LM75B_ADDR | dev_num, Temp )) / 256.0 );
}

void iic_write( char addr, char reg, char data )
{
    char cmd[2];
    
    cmd[ 0 ]    = reg;
    cmd[ 1 ]    = data;
    
    if ( i2c.write( addr, cmd, 2) )
        iic_error();        
}

char iic_read( char addr, char reg )
{
    char cmd;
    int nack;

    cmd    = reg;
   
    nack     = i2c.write( addr, &cmd, 1); // Send command string
    nack    |= i2c.read( addr, &cmd, 1); // Send command string
    
    if ( nack )
        iic_error();        
    
    return ( cmd );
}


short iic_read_short( char addr, char reg )
{
    char cmd[ 2 ];
    int nack;

    cmd[ 0 ]    = reg;
   
    nack     = i2c.write( addr, cmd, 1); // Send command string
    nack    |= i2c.read( addr, cmd, 2); // Send command string
    
    if ( nack )
        iic_error();        

    return ( ((short)cmd[ 0 ]) << 8 | cmd[ 1 ] );
}


void iic_error( void )
{
    pc.printf( "I2C error\n" );

    for ( int i = 0; i < 4; i++ )
        led[ i ]    = i & 0x01;
    
    wait ( 0.2 );
    
    for ( int i = 0; i < 4; i++ )
        led[ i ]    = !(i & 0x01);
        
    wait ( 0.2 );
}

