/** LED Cube 4x4x4 control library
 *
 *  @class   LED_Cube444
 *  @author  Tedd OKANO
 *  @version 0.51(08-Dec-2010)
 *   
 *      This is a library for a demo code of mbeduino + 4x4x4 LED Cube shield
 * 
 *      mbeduino              = http://mbed.org/users/okini3939/notebook/mbeduino/  (Japanese)
 *      4x4x4 LED Cube shield = http://www.galileo-7.com/?pid=20015630  (Japanese)
 *
 *      Released under the MIT License: http://mbed.org/license/mit
 *
 *      revision 0.5   14-Oct-2010   1st release
 *      revision 0.51  08-Dec-2010   Document reformatted
 */


#ifndef _LED_Cube
#define _LED_Cube

#include "mbed.h"
#include "mbeduino_shield.h"

typedef     unsigned short  U16;    //  bit length should be same or more than SR_BIT_LENGTH

#define     CUBE_SIZE       4                       //  n of nodes on one edge
#define     SR_BIT_LENGTH   (CUBE_SIZE * CUBE_SIZE) //  n of bits on one layer

#define     SR_DATA_OUT     ARD_D2  //  data line to the shiftregister
#define     SR_CLCK_OUT     ARD_D3  //  clock line to the shiftregister
#define     CATHODE0        ARD_D11 //  cathode control line for bottom layer
#define     CATHODE1        ARD_D10 //  cathode control line for 2nd layer from bottom
#define     CATHODE2        ARD_D9  //  cathode control line for 3nd layer from bottom
#define     CATHODE3        ARD_D8  //  cathode control line for top layer

#define     DISPLAY_REFRESH_RETE    0.002

/** LED_Cube444 control class
 *
 *  Example:
 *  @code
 *  #include "mbed.h"
 *  #include "LED_Cube444.h"
 *   
 *  LED_Cube444 cube;
 *  
 *  int main() {
 *  
 *      cube.set_bits( 0, 0xA5A5 );
 *      cube.set_bits( 1, 0x5A5A );
 *      cube.set_bits( 2, 0xA5A5 );
 *      cube.set_bits( 3, 0x5A5A );
 *      wait( 0.5 );
 *  
 *      cube.set_bits( 0, 0x5A5A );
 *      cube.set_bits( 1, 0xA5A5 );
 *      cube.set_bits( 2, 0x5A5A );
 *      cube.set_bits( 3, 0xA5A5 );
 *      wait( 0.5 );
 *  
 *      cube.clear();
 *  
 *      int v   = 1;
 *  
 *      while ( 1 ) {
 *  
 *          for ( int x = 0; x < 4; x++ ) {
 *              for ( int y = 0; y < 4; y++ ) {
 *                  for ( int z = 0; z < 4; z++ ) {
 *                      cube.set_bit( x, y, z, v );
 *                      wait( 0.01 );
 *                  }
 *              }
 *          }
 * 
 *         v   = !v;
 *     }
 * }
 *  @endcode
 */


class LED_Cube444 {
public:

	/** Class constructor for LED_Cube444
	 *  
	 *  This will create the instance and start periodical routine to display class internal buffer content to LED cube
	 */

    LED_Cube444();
	
	/** Set bits into cube
	 *  
	 *  The 16 bit word array will be copied into class internal buffer. 
	 *  It will be displayed at next update timing by periodical routine
	 * 
	 *  @param    v[4]     whole cube bits can be represented in this array. each U16 data represents the bits on each planes. 
	 */

    void set_bits( U16 v[4] );
	
	/** Set bits into cube
	 *  
	 *  The 16 bit word will be copied into specified layer of class internal buffer. 
	 *  It will be displayed at next update timing by periodical routine
	 * 
	 *  @param    v        Bit pattern for a layer. Overwrites the buffer contents 
	 */

    void set_bits( int layer, U16 v );
	
	/** Set bit into cube
	 *  
	 *  Set a specified bit in the buffer.  
	 *  It will be displayed at next update timing by periodical routine
	 * 
	 *  @param    x        Coodinate: x
	 *  @param    y        Coodinate: y 
	 *  @param    z        Coodinate: z 
	 *  @param    v        Value for the bit (0 or 1) 
	 */

    void set_bit( int x, int y, int z, U16 v );
	
	/** Clear
	 *  
	 *  Turn-OFF all LEDs
	 */

    void clear( void );
	
	/** All on
	 *  
	 *  Turn-ON all LEDs
	 */

    void all_on( void );
	
	/** Set bits into cube
	 *  
	 *  The 16 bit word will be copied into all layers of class internal buffer. 
	 *  It will be displayed at next update timing by periodical routine
	 * 
	 *  @param    v        Bit pattern for layers. Overwrites the buffer contents 
	 */

    void set_all_words( int v );
	
	/** Setting for synchronous display update
	 *  
	 *  If the "synchronous display" option is set (default) the display 
	 *  (main) buffer will be updated each timing if just before layer 0 update. 
	 *  This machanism has been made to avoid flicker when main routine update 
	 *  buffer so frequently. 
	 *  To implement this mechanism, this class has 2 buffers. 
	 *  One is called main buffer and another is temporary buffer. 
	 *  All API calls that makes modifications of the buffer content will affect to the temp buffer. 
	 *  The display will be done with main buffer contents. So the timing of update can be controled by buffer copying. 
	 *  If the "synchronous display" option is cleard, the temporary buffer will be used for display (not tested). 
	 * 
	 *  @param    v        TRUE for set, FALSE for clear 
	 */

    void set_synchronous_draw( int v );

private:
    DigitalOut  sr_data;
    DigitalOut  sr_clk;
    BusOut      cathode;
    Ticker      t;
    U16         main_buffer[ CUBE_SIZE ];
    U16         temp_buffer[ CUBE_SIZE ];
    int         syncronous;

	/** Displaying
	 *  
	 *  This function will be called by Ticker which set by this class' constructor. 
	 *  It displays just one layer by single call. 
	 */

    void display( void );
	
	/** Data for serial register
	 *  
	 *  Drives the pins to send serial data to serial register on the shield 
	 *
	 *  @param    v        Bit pattern for a layer. 
	 */

    void set_serialregister( U16 v );
	
	/** Array copy function
	 *  
	 *  Just copies the array. Loop is unrolled because it;s not big array.  
	 *
	 *  @param    src      Pointer to source array.
	 *  @param    trg      Pointer to target array.
	 */

    void buffer_copy( U16 *src, U16 *trg );
};

#endif  // _LED_Cube
