/** LED Cube 4x4x4 control library
 *
 *  @class   LED_Cube444
 *  @author  Tedd OKANO
 *  @version 0.51(08-Dec-2010)
 *   
 *      This is a library for a demo code of mbeduino + 4x4x4 LED Cube shield
 * 
 *      mbeduino              = http://mbed.org/users/okini3939/notebook/mbeduino/  (Japanese)
 *      4x4x4 LED Cube shield = http://www.galileo-7.com/?pid=20015630  (Japanese)
 *
 *      Released under the MIT License: http://mbed.org/license/mit
 *
 *      revision 0.5   14-Oct-2010   1st release
 *      revision 0.51  08-Dec-2010   Document reformatted
 */


#include "LED_Cube444.h"

/** Class constructor for LED_Cube444
 *  
 *  This will create the instance and start periodical routine to display class internal buffer content to LED cube
 */

LED_Cube444::LED_Cube444() : sr_data(SR_DATA_OUT), sr_clk(SR_CLCK_OUT), cathode( CATHODE0, CATHODE1, CATHODE2, CATHODE3 ), syncronous( 1 ) {
    cathode = 0x0;
    t.attach( this, &LED_Cube444::display, DISPLAY_REFRESH_RETE );
}

/** Set bits into cube
 *  
 *  The 16 bit word array will be copied into class internal buffer. 
 *  It will be displayed at next update timing by periodical routine
 * 
 *  @param    v[4]     whole cube bits can be represented in this array. each U16 data represents the bits on each planes. 
 */

void LED_Cube444::set_bits( U16 v[4] ) {
    buffer_copy( v, temp_buffer );
}

/** Set bits into cube
 *  
 *  The 16 bit word will be copied into specified layer of class internal buffer. 
 *  It will be displayed at next update timing by periodical routine
 * 
 *  @param    v        Bit pattern for a layer. Overwrites the buffer contents 
 */

void LED_Cube444::set_bits( int layer, U16 v ) {
    temp_buffer[ layer ]    = v;
}

/** Set bit into cube
 *  
 *  Set a specified bit in the buffer.  
 *  It will be displayed at next update timing by periodical routine
 * 
 *  @param    x        Coodinate: x
 *  @param    y        Coodinate: y 
 *  @param    z        Coodinate: z 
 *  @param    v        Value for the bit (0 or 1) 
 */

void LED_Cube444::set_bit( int x, int y, int z, U16 v ) {
    if ( v )
        temp_buffer[ z ]    |= 0x1 << ((y << 2) + x);
    else
        temp_buffer[ z ]    &= ~(0x1 << ((y << 2) + x));
}

/** Clear
 *  
 *  Tuen-off all LEDs
 */

void LED_Cube444::clear( void ) {
    set_all_words( 0x0000 );
}

/** All on
 *  
 *  Tuen-on all LEDs
 */
 
void LED_Cube444::all_on( void ) {
    set_all_words( 0xFFFF );
}

/** Set bits into cube
 *  
 *  The 16 bit word will be copied into all layers of class internal buffer. 
 *  It will be displayed at next update timing by periodical routine
 * 
 *  @param    v        Bit pattern for layers. Overwrites the buffer contents 
 */

void LED_Cube444::set_all_words( int v ) {
    temp_buffer[ 0 ]    = v;
    temp_buffer[ 1 ]    = v;
    temp_buffer[ 2 ]    = v;
    temp_buffer[ 3 ]    = v;
}

/** Setting for synchronous display update
 *  
 *  If the "synchronous display" option is set (default) the display 
 *  (main) buffer will be updated each timing if just before layer 0 update. 
 *  This machanism has been made to avoid flicker when main routine update 
 *  buffer so frequently. 
 *  To implement this mechanism, this class has 2 buffers. 
 *  One is called main buffer and another is temporary buffer. 
 *  All API calls that makes modifications of the buffer content will affect to the temp buffer. 
 *  The display will be done with main buffer contents. So the timing of update can be controled by buffer copying. 
 *  If the "synchronous display" option is cleard, the temporary buffer will be used for display (not tested). 
 * 
 *  @param    v        TRUE for set, FALSE for clear 
 */

void LED_Cube444::set_synchronous_draw( int v ) {
    syncronous  = v;
}

/** Displaying
 *  
 *  This function will be called by Ticker which set by this class' constructor. 
 *  It displays just one layer by single call. 
 */

void LED_Cube444::display( void ) {
    static int  layer = 0;
    U16         *the_buffer;

    if ( !layer & syncronous )
        buffer_copy(temp_buffer, main_buffer);

    if ( syncronous )
        the_buffer  = main_buffer;
    else
        the_buffer  = temp_buffer;

    cathode = 0x0;
    set_serialregister( the_buffer[ layer ] );
    cathode = 0x1 << layer;

    layer++;
    layer   &= 0x3;
}

/** Data for serial register
 *  
 *  Drives the pins to send serial data to serial register on the shield 
 *
 *  @param    v        Bit pattern for a layer. 
 */

void LED_Cube444::set_serialregister( U16 v ) {
    for ( int i = 0; i < SR_BIT_LENGTH; i++ ) {
        sr_data = ((v >> i) & 0x1);
        sr_clk  = 0;
        sr_clk  = 1;
    }
}

/** Array copy function
 *  
 *  Just copies the array. Loop is unrolled because it;s not big array.  
 *
 *  @param    src      Pointer to source array.
 *  @param    trg      Pointer to target array.
 */

void LED_Cube444::buffer_copy( U16 *src, U16 *trg ) {
    trg[ 0 ]    = src[ 0 ];
    trg[ 1 ]    = src[ 1 ];
    trg[ 2 ]    = src[ 2 ];
    trg[ 3 ]    = src[ 3 ];
}
