/** Clock demo with stepper motor + NTP
 *
 *  Simple mbed analog clock demo with stepper motor.
 *  Time information will be get from NTP
 *  This code based on
 *  http://mbed.org/users/nucho/programs/SimpleLCDClock/5zi4y
 *  http://mbed.org/users/okano/notebook/nxp_pcf2127a-demo-code/
 *
 *  2010-Nov-20  @tedd_okano
 *  Released under the MIT License: http://mbed.org/license/mit
 */

#include "mbed.h"
#include "EthernetNetIf.h"
#include "NtpClient.h"
#include "TextLCD.h"
#include "StepperMotor.h"


EthernetNetIf   eth;
NtpClient       ntp;
TextLCD         lcd( p14, p15, p16, p17, p18, p19 ); // rs, rw, d0, d1, d2, d3
Ticker          updater;
StepperMotor    m_s( p21, p22, p23, p30 );
StepperMotor    m_m( p24, p25, p26, p30 );
StepperMotor    m_h( p27, p28, p29, p30 );
AnalogIn        ain( p20 );

int     tz_index            = 4;
int     update_stop_lcd     = false;
int     update_stop_hands   = false;
int     resync              = false;

#define RESYNC_INTERVAL     1  //  minutes

typedef    struct _tz    {
    char    *name;
    float   offset;
}
tz;

//  http://wwp.greenwichmeantime.com/info/timezone.htm

tz  timezone[ 8 ]  = {
    { "London",          0 },
    { "Amsterdam",       1 },
    { "India",           5.5 },
    { "Singapore",       8 },
    { "Tokyo",           9 },
    { "Hawaii",        -10 },
    { "San Francisco",  -8 },
    { "New York",       -5 },
};

void clock_update( void );
void show_clock( int h, int m, int s );
int access_NTP( void );

int main() {
    m_h.go_angle( 180 );
    m_m.go_angle( 180 );
    m_s.go_angle( 180 );

    lcd.cls();
    lcd.printf( "Ethernet setting" );

    EthernetErr ethErr = eth.setup();

    if (ethErr) {
        lcd.printf("Error %d in setup.\n", ethErr);
        return -1;
    }

    if ( access_NTP() ) {
        m_h.go_angle( 0 );
        m_m.go_angle( 0 );
        m_s.go_angle( 0 );

        return -1;
    }

    while ( m_h.distance() | m_m.distance() | m_s.distance() )
        wait( 0.1 );

    m_h.set_home_position();
    m_m.set_home_position();
    m_s.set_home_position();

    lcd.cls();

    updater.attach(&clock_update, 1.0);

    int     tz_i    = 0;
    int     tz_i_p  = -1;
    int     unstable = 50;

    while (1) {

        if ( resync ) {
            update_stop_hands   = true;

            m_h.go_angle( 0 );
            m_m.go_angle( 0 );
            m_s.go_angle( 0 );

            if ( access_NTP() ) {
                return -1;
            }

            update_stop_hands   = false;
            resync              = false;
            tz_i_p              = -1;
        }
        tz_i    = (int)( ain * 7.99 );

        if ( tz_i != tz_i_p ) {
            update_stop_lcd = true;
            lcd.locate( 0, 1 );
            lcd.printf( "                " );
            lcd.locate( 0, 1 );
            lcd.printf( "%s", timezone[ tz_i ].name );
            update_stop_lcd = false;

            tz_i_p      = tz_i;
            unstable    = 50;
        } else {
            unstable    = unstable ? unstable-- : 0;
            tz_index    = unstable ? tz_i : tz_index;
        }
        wait( 0.1 );
    }
}

void clock_update( void ) {
    struct tm   dt, *dtp;
    time_t      t;
    char        s[ 30 ];
    dtp = &dt;

    t       = time( NULL ) + (int)(timezone[ tz_index ].offset * 3600.0);
    dtp     = localtime( &t );

    if ( !(dtp->tm_min % RESYNC_INTERVAL) && !(dtp->tm_sec) )
        resync  = true;

    if ( !update_stop_hands )
        show_clock( dtp->tm_hour, dtp->tm_min, dtp->tm_sec );

    if ( update_stop_lcd )
        return;

    strftime( s, 20, "%H:%M:%S", dtp );
    lcd.locate( 0, 0 );
    lcd.printf( "%s", s );

#if 0
    strftime( s, 20, "%Y/%b/%d(%a)", dtp );
    lcd.locate( 0, 1 );
    lcd.printf( "%s", s );
#endif
}

void show_clock( int h, int m, int s ) {
    printf( "%02d:%02d:%02d\r\n",h,m,s );

    h   = h % 12;
    m_s.go_angle( -((float)s * 360.0) / 60.0 );
    m_m.go_angle( -((float)m * 360.0) / 60.0 );
    m_h.go_angle( -(((float)h + ((float)m / 60.0)) * 360.0) / 12.0 );
}

int access_NTP( void ) {
    lcd.cls();
    lcd.printf( "accessing NTP.. " );

    lcd.locate( 0, 0 );
    lcd.printf("Setup OK");

    lcd.printf("Timer setting...\n");
    Host server(IpAddr(), 123, "ntp.jst.mfeed.ad.jp");
    ntp.setTime(server);

    lcd.cls();

    return 0;
}

