/*
 *  I2C/SPI digital pressure sensor "LPS331AP" "LPS25H" library for SPI mode.
 *
 *  Copyright(c) -2016 Toru OHTSUKA, 
 *  Copyright(c) -2013 unos@NYAMFG, 
 *  Released under the MIT License: http://mbed.org/license/mit
 *
 *  revision: see LPS331_SPI.h.
 */

#include "LPS331_SPI.h"

LPS331_SPI::LPS331_SPI(PinName mosi, PinName miso, PinName sclk, PinName cs_pin):_spi(mosi, miso, sclk), _cs(cs_pin)
{
    _spi.format(8,0);
    _spi.frequency(1000000);

    _ctrlreg1 = 0x20;
}

LPS331_SPI::~LPS331_SPI()
{
}

char LPS331_SPI::whoami()
{
    return _read(0x0f);
}

bool LPS331_SPI::isLPS331()
{
    if(whoami() == 0xbb) { return true; }
    
    return false;
}

bool LPS331_SPI::isLPS25H()
{
    if(whoami() == 0xbd) { return true; }
    
    return false;
}

void LPS331_SPI::setResolution(char pressure_avg, char temp_avg)
{
    _write(0x10, ((temp_avg & 0x07) << 4) | (pressure_avg & 0x0f));
}

void LPS331_SPI::setActive(bool is_active)
{
    if(is_active) {
        _ctrlreg1 |= 0x80;
    } else {
        _ctrlreg1 &= ~0x80;
    }

    _write(0x20, _ctrlreg1);
}

void LPS331_SPI::setDataRate(char datarate)
{
    datarate &= 0x07;
    
    _ctrlreg1 &= ~(0x07 << 4);
    _ctrlreg1 |= datarate << 4;
    
    _write(0x20, _ctrlreg1);
}

void LPS331_SPI::quickStart()
{
    // initialize
    setResolution(LPS331_SPI_PRESSURE_AVG_1, LPS331_SPI_TEMP_AVG_1); // To prevent delays, not use average.
    setDataRate(LPS331_SPI_DATARATE_25HZ);
    setActive(true);
}
    
float LPS331_SPI::getPressure()
{
    char data[3];
    float pressure = 0;

    _read_multibyte(0x28, data, 3);
    
    pressure  = data[0];
    pressure += data[1] << 8;
    pressure += data[2] << 16;
    pressure /= 4096.0;
    
    return pressure;
}

float LPS331_SPI::getTemperature()
{
    char data[2];
    short temp = 0;
    
    _read_multibyte(0x2b, data, 2);    
    
    temp  = data[0];
    temp |= data[1] << 8;
    
    return (float)(42.5 + temp / 480.0);
}


void LPS331_SPI::_write(char subaddress, char data)
{
    _cs = 0;
    _spi.write(subaddress);
    _spi.write(data);
    _cs = 1;
}

char LPS331_SPI::_read(char subaddress)
{
    char result = 0;
    
    _cs = 0;
    _spi.write(subaddress | 0x80);
    result = _spi.write(0);
    _cs = 1;

    return result;
}

void LPS331_SPI::_read_multibyte(char startsubaddress, char* data, char count)
{
    _cs = 0;
    _spi.write(startsubaddress | 0x80 | 0x40);

    for(int i = 0; i < count; i++) {
        data[i] = _spi.write(0);
    }
    _cs = 1;

}
