/** @file NJU6063.h
* @author k.og
* @version 0.1
* @date 26-Dec-2015
* @brief mbed library to use a NJU6063 RGB LED driver
* http://www.njr.com/semicon/products/NJU6063.html
* interface: I2C
*/
#ifndef MBED_NJU6063_H
#define MBED_NJU6063_H
#
#include "mbed.h"
#define NJU6063_SLAVE 0x40
#define NJU6063_INIT 0x00
#define NJU6063_ILED 0x01
#define NJU6063_PWM1 0x02
#define NJU6063_PWM2 0x03
#define NJU6063_PWM3 0x04
#define NJU6063_DIMM 0x05
#define NJU6063_STON 0x06
#define NJU6063_START 0x07
#define NJU6063_DCHK 0x0b
#define NJU6063_MADRES 0x0d


#define NJU6063_EN 0x01
#define NJU6063_EXT 0x02
#define NJU6063_FDX1 0x00
#define NJU6063_FDX2 0x04
#define NJU6063_FDX4 0x08
#define NJU6063_FDX8 0x0C
#define NJU6063_FC1000KHZ 0x00
#define NJU6063_FC1300KHZ 0x10
#define NJU6063_FC800KHZ 0x20
#define NJU6063_FC2200KHZ 0x30
#define NJU6063_DMGP 0x80
#define NJU6063_DO1 0x40
#define NJU6063_ILED1OFF 0x00
#define NJU6063_ILED1X1  0x03
#define NJU6063_ILED1X05 0x02
#define NJU6063_ILED1X025 0x01
#define NJU6063_ILED2OFF 0x00
#define NJU6063_ILED2X1  0x03
#define NJU6063_ILED2X05 0x02
#define NJU6063_ILED2X025 0x01
#define NJU6063_ILED3OFF 0x00
#define NJU6063_ILED3X1  0x03
#define NJU6063_ILED3X05 0x02
#define NJU6063_ILED3X025 0x01
#define NJU6063_ALL 0xff

#define NJU6063_DEFAULTINIT NJU6063_FC800KHZ|NJU6063_FDX8|NJU6063_EN
#define NJU6063_WAIT wait_us(500*3)
#define NJU6063_WAIT_MS wait_ms(200)

class NJU6063
{
private:
    I2C _i2c;
    //I2C_TypeDef* _myI2c;
    //I2C_HandleTypeDef _myI2cH;
    DigitalOut _rst;
    int ack;
public:
    /**
    * @param sda I2C-bus SDA pin
    * @param scl I2C-bus SCL pin
    * @param rst Digital pin
    */
    NJU6063( PinName sda, PinName scl, PinName rst);
    /** Destructor of NJU6063
    */
    ~NJU6063() {};
    /** NJU6063 Reset
    */
    void reset(void);
    /** NJU6063 device addres set
    * @param n Max number of device. 1 - 254
    */
    uint8_t set_multi_device(uint8_t n=0xfe);
    /** NJU6063 Initial setting
    * @param chip_addr  target chip address
    * @param d          data
    *   7   6   5   4   3   2   1   0
    *  DM  DO FC1 FD0 FD1 FD0 EXT  EN
    * DM DO1 Function 0: Multi device control, 1: GPO
    * DO The bit output to DO1
    * FC OSC Frequency 0:1MHz, 1:1.3MHz, 2:2.2MHz, 3:0.8MHz
    * FD divider 0:1/256, 1:1/512, 2:1/1024, 3:1/2048
    * EXT OSC select 0:INT OSC, 1:EXT CLOCK
    * EN 0:Disable, 1:Enable
    */
    void init(uint8_t chip_addr, uint8_t d);
    /** Set LED current
    * @param chip_addr  target chip address
    * @param d1         led1 current set
    * @param d2         led2 current set
    * @param d3         led3 current set
    * 0: LED off, 1: 0.25*ILED, 2: 0.5*ILED, 3: 1*ILED
    */
    void set_iled(uint8_t chip_addr, uint8_t d1, uint8_t d2, uint8_t d3);
    /** Set PWM Duty
    * @param chip_addr  target chip address
    * @param d1         led1 pwm  0 - 0x7f (127/128)
    * @param d2         led2 pwm  0 - 0x7f (127/128)
    * @param d3         led3 pwm  0 - 0x7f (127/128)
    * @param loop       no of loop & no of step
    * @param son        led3 current set bit0:LED1, bit1:LED2, bit3:LED3
    * 
    * STEP[6:4] 2^n , 0:1, 1:2, 2:4, 3:8, 4:16, 5:32, 6:1, 7:1
    * LOOP[3:0] 4*(n+1) , 0:4 ... 15:64
    */
    void set_pwm(uint8_t chip_addr, uint8_t d1, uint8_t d2, uint8_t d3, uint8_t loop=0x5f, uint8_t son=0x00);
    /** Dimming Start
    * @param chip_addr  target chip address
    */
    void dim_start(uint8_t chip_addr);
    /** Dimming stop
    */
    void dim_stop(uint8_t chip_addr);
    /** Dimming check
    */
    void check_dim(void);
};

#endif // MBED_NJU6063_H