/*
 * StateMachine.cpp
 * Copyright (c) 2020, ZHAW
 * All rights reserved.
 */

#include <cmath>
#include "StateMachine.h"

using namespace std;

const float StateMachine::PERIOD = 0.01f;                   // period of task in [s]
const float StateMachine::DISTANCE_THRESHOLD = 0.25f;        // minimum allowed distance to obstacle in [m]
const float StateMachine::TRANSLATIONAL_VELOCITY = 1.0f;    // translational velocity in [m/s]
const float StateMachine::ROTATIONAL_VELOCITY = 1.5f;       // rotational velocity in [rad/s]

/**
 * Creates and initializes a state machine object.
 */
StateMachine::StateMachine(Controller& controller, DigitalOut& enableMotorDriver, DigitalOut& led0, DigitalOut& led1, DigitalOut& led2, DigitalOut& led3, DigitalOut& led4, DigitalOut& led5, DigitalIn& button, IRSensor& irSensor0, IRSensor& irSensor1, IRSensor& irSensor2, IRSensor& irSensor3, IRSensor& irSensor4, IRSensor& irSensor5) : controller(controller), enableMotorDriver(enableMotorDriver), led0(led0), led1(led1), led2(led2), led3(led3), led4(led4), led5(led5), button(button), irSensor0(irSensor0), irSensor1(irSensor1), irSensor2(irSensor2), irSensor3(irSensor3), irSensor4(irSensor4), irSensor5(irSensor5) {
    
    enableMotorDriver = 0;
    state = ROBOT_OFF;
    buttonNow = button;
    buttonBefore = buttonNow;
    
    ticker.attach(callback(this, &StateMachine::run), PERIOD);
}

/**
 * Deletes the state machine object and releases all allocated resources.
 */
StateMachine::~StateMachine() {
    
    ticker.detach();
}

/**
 * Gets the actual state of this state machine.
 * @return the actual state as an int constant.
 */
int StateMachine::getState() {
    
    return state;
}

/**
 * This method is called periodically by the ticker object and implements the
 * logic of the state machine.
 */
void StateMachine::run() {
    
    // set the leds based on distance measurements
    
    led0 = irSensor0 < DISTANCE_THRESHOLD;
    led1 = irSensor1 < DISTANCE_THRESHOLD;
    led2 = irSensor2 < DISTANCE_THRESHOLD;
    led3 = irSensor3 < DISTANCE_THRESHOLD;
    led4 = irSensor4 < DISTANCE_THRESHOLD;
    led5 = irSensor5 < DISTANCE_THRESHOLD;
    
    // implementation of the state machine
    
    switch (state) {
        
        case ROBOT_OFF:
            
            buttonNow = button;
            
            if (buttonNow && !buttonBefore) {   // detect button rising edge
                
                enableMotorDriver = 1;
                
                controller.setTranslationalVelocity(TRANSLATIONAL_VELOCITY);
                controller.setRotationalVelocity(0.0f);
                
                state = MOVE_FORWARD;
            }
            
            buttonBefore = buttonNow;
            
            break;
            
        case MOVE_FORWARD:
            
            if (!led2 && !led3 && !led4) {
                controller.setTranslationalVelocity(TRANSLATIONAL_VELOCITY);
                controller.setRotationalVelocity(0.0f);
                }
            else if(led2 || (led2&&led3)){
                state = TURN_LEFT;
                }
            else if(led4 || (led3&&led4)){
                state = TURN_RIGHT;
                }
            else if(led2 && led3 && led4){
               state = SLOWING_DOWN;
               }
                
                
            // bitte implementieren!
            
            break;
            
        case TURN_LEFT:
       
            
           
            if(led4 || led3 || (led4&&led3)){
                controller.setTranslationalVelocity(0.0f);
                controller.setRotationalVelocity(-ROTATIONAL_VELOCITY);
                }
         
            state = MOVE_FORWARD;
            // bitte implementieren!
            
            break;
            
        case TURN_RIGHT:
        
            if(led3 || led2 || (led2&&led3)){
                controller.setTranslationalVelocity(0.0f);
                controller.setRotationalVelocity(ROTATIONAL_VELOCITY);
                }
            
            state = MOVE_FORWARD;
            // bitte implementieren!
            
            break;
            
        case SLOWING_DOWN:
            if (controller.getActualTranslationalVelocity()>0.0f){
                    controller.setTranslationalVelocity(0.0f);
            } else if (controller.getActualTranslationalVelocity()<= 0.0001f){
                enableMotorDriver = 0;
                }
            state = ROBOT_OFF;
            
            // bitte implementieren!
            
            break;
            
        default:
            
            state = ROBOT_OFF;
    }
}

