#include "LT_I2C.h"

/* mbed headers */
#include <mbed.h>
#include <I2C.h>

I2C *i2c_object = NULL;

int lt_i2c_init(int sda, int scl) {
    if (i2c_object == NULL) {
        i2c_object = new I2C((PinName) sda, (PinName) scl);
        return LT_I2C_INIT_FINE;
    }
    else {
        return LT_I2C_INIT_QUIT_FIRST;
    }
}

int lt_i2c_quit() {
    delete i2c_object;
    return LT_I2C_SUCCESS;
}

int lt_i2c_init_attach(void *p_i2c_object) {
    if (i2c_object == NULL) {
        i2c_object = (I2C *) p_i2c_object;
        return LT_I2C_INIT_FINE;
    }
    else {
        return LT_I2C_INIT_QUIT_FIRST;
    }
}

void *lt_i2c_get_i2c_object() {
    return i2c_object;
}

int lt_i2c_start() {
    i2c_object->start();
    return LT_I2C_SUCCESS;
}

int lt_i2c_stop() {
    i2c_object->stop();
    return LT_I2C_SUCCESS;
}

uint8_t lt_i2c_write(uint8_t data) {
    return 1-i2c_object->write(data);
}

uint8_t lt_i2c_read(int8_t ack) {
    return (uint8_t) (i2c_object->read(1-ack)); // act is different in mbed
}

int8_t lt_i2c_write_byte_data(
    uint8_t address,
    uint8_t command,
    uint8_t value) {
    int8_t ret = 0;
    if (lt_i2c_start()!=0) //I2C START
    {
        return(1);      //Stop and return 0 if START fail
    }
    ret |= lt_i2c_write((address<<1)|LT_I2C_WRITE_BIT);
        // Write 7 bit address with W bit
    ret|= lt_i2c_write(command);   // Set register to be read to command
    ret|= lt_i2c_write(value);
    lt_i2c_stop(); // I2C STOP
    if (ret!=0)     //If there was a NAK return 1
    {
        return(1);
    }
    return(0);      // Return success
}

int8_t lt_i2c_write_word_data(
    uint8_t address,
    uint8_t command,
    uint16_t value) {
    int8_t ret=0;
    union
    {
        uint8_t b[2];
        uint16_t w;
    } data;
    data.w = value;
    if (lt_i2c_start()!=0) //I2C START
    {
        return(1);      //Stop and return 0 if START fail
    }
    ret |= lt_i2c_write((address<<1)|LT_I2C_WRITE_BIT);
        // Write 7 bit address with W bit
    ret|= lt_i2c_write(command);   // Set register to be read to command
    ret|= lt_i2c_write(data.b[1]); //Write MSB
    ret|= lt_i2c_write(data.b[0]); //Write LSB;
    lt_i2c_stop(); // I2C STOP
    if (ret!=0) //If there was a NAK return 1
    {
       return(1);
    }
    return(0);
}

uint8_t lt_i2c_write_block_data(
    uint8_t address,
    uint8_t command,
    uint8_t length,
    uint8_t *values) {
    int8_t i = length-1;
    int8_t ret = 0;
    if (lt_i2c_start()!=0) //I2C START
        return(1);      //Stop and return 0 if START fail
    ret |= lt_i2c_write((address<<1)|LT_I2C_WRITE_BIT);
        // Write 7 bit address with W bit
    ret|= lt_i2c_write(command);   // Set register to be read to command
    while (i>=0)
    {
        ret|= lt_i2c_write(values[i]); //Write Value
        i--;
    }
    lt_i2c_stop(); // I2C STOP
    if (ret!=0)
    {
        return(1);
    }
    else
    {
        return(0);  // Success!
    }
}

uint8_t lt_i2c_read_byte_data(
    uint8_t address,
    uint8_t command,
    uint8_t *value) {
    int8_t ret = 0;
    if (lt_i2c_start() != 0) // I2C START
    {
        return (1); // Stop and return 0 if START fail
    }
    ret |= lt_i2c_write((address << 1) | LT_I2C_WRITE_BIT); // Write 7 bit address with W bit
    if (ret != 0) // If NACK return 1
    {
        lt_i2c_stop(); // I2C STOP
        return (1);
    }
    ret |= lt_i2c_write(command); // Set register to be read to command
    if (ret != 0) // If NACK return 1
    {
        lt_i2c_stop(); // I2C STOP
        return (1);
    }
    if (lt_i2c_start() != 0) // I2C repeated START
    {
        lt_i2c_stop(); // Attempt to issue I2C STOP
        return (1); // Stop and return 0 if START fail
    }
    ret |= lt_i2c_write((address << 1) | LT_I2C_READ_BIT); // Write 7 bit address with R bit
    *value = lt_i2c_read(LT_I2C_WITH_NACK); // Read byte from buffer with NAK
    lt_i2c_stop(); // I2C STOP
    if (ret != 0) // If there was a NACK return 1
    {
        return (1);
    }
    return (0); // Return success
}

uint8_t lt_i2c_read_word_data(
    uint8_t address,
    uint8_t command,
    uint16_t *value) {
    int8_t ret = 0;
    union {
        uint8_t b[2];
        uint16_t w;
    } data;
    if (lt_i2c_start() != 0) //I2C START
    {
        return (1); //Stop and return 0 if START fail
    }
    ret |= lt_i2c_write((address << 1) | LT_I2C_WRITE_BIT); // Write 7 bit address with W bit
    if (ret != 0) //If NACK return 1
    {
        lt_i2c_stop(); //I2C STOP
        return (1);
    }
    ret |= lt_i2c_write(command); // Set register to be read to command
    if (ret != 0) //If NACK return 1
    {
        lt_i2c_stop(); //I2C STOP
        return (1);
    }
    if (lt_i2c_start() != 0) //I2C START
    {
        lt_i2c_stop(); //Attempt to issue I2C STOP
        return (1); //Stop and return 0 if START fail
    }
    ret |= lt_i2c_write((address << 1) | LT_I2C_READ_BIT); // Write 7 bit address with R bit
    data.b[1] = lt_i2c_read(LT_I2C_WITH_ACK); // Read MSB from buffer
    data.b[0] = lt_i2c_read(LT_I2C_WITH_NACK); // Read LSB from buffer
    lt_i2c_stop(); //I2C STOP
    *value = data.w;
    if (ret != 0) //If NAK
    {
        return (1); //return 1
    }
    return (0); // Return success
}

uint8_t lt_i2c_read_block_data(
    uint8_t address,
    uint8_t command,
    uint8_t length,
    uint8_t *values) {
    uint8_t i = (length - 1);
    int8_t ret = 0;
    if (lt_i2c_start() != 0) //I2C START
    {
        return (1); //Stop and return 0 if START fail
    }
    ret |= lt_i2c_write((address << 1) | LT_I2C_WRITE_BIT); //Write 7-bit address with W bit
    if (ret != 0) //If NACK return 1
    {
        lt_i2c_stop(); //I2C STOP
        return (1);
    }
    ret |= lt_i2c_write(command); //Write 8 bit command word
    if (ret != 0) //If NACK return 1
    {
        lt_i2c_stop(); //I2C STOP
        return (1);
    }
    if (lt_i2c_start() != 0) //I2C repeated START
    {
        lt_i2c_stop(); //Attempt to issue I2C STOP
        return (1); //Stop and return 0 if START fail
    }
    ret |= lt_i2c_write((address << 1) | LT_I2C_READ_BIT); //Write 7-bit address with R bit
    if (ret != 0) //If NACK return 1
    {
        lt_i2c_stop(); //I2C STOP
        return (1);
    }
    while (i > 0) //Begin read loop
    {
        values[i] = lt_i2c_read(LT_I2C_WITH_ACK); //Read from bus with ACK
        i--;
    }
    values[0] = lt_i2c_read(LT_I2C_WITH_NACK); //Read from bus with NACK for the last one;
    lt_i2c_stop(); //I2C STOP    
    return (0); // Success!
}
