/**
********************************************************************************
* @file         bluetooth.h
* @brief        Header file containing API for bluetooth.c
*
********************************************************************************
* @author       TTP
* @copyright (c) 2017 TTP (The Technology Partnership) plc
*
********************************************************************************
*/

#ifndef BLUETOOTH_H
#define BLUETOOTH_H

//-------- Includes ----------------------------------------------------------//

//-------- Defines -----------------------------------------------------------//

#define SEND_BLE_RESPONSE               0
#define DO_NOT_SEND_BLE_RESPONSE        1

#define UBLOX_BLE_NAME_BYTES            24
#define UBLOX_BLE_FW_VERSION_LEN_BYTES  9

#define UBLOX_DEFAULT_1_BAUD_RATE       115200
#define UBLOX_DEFAULT_2_BAUD_RATE       230400 
#define UBLOX_MICRO_BAUD_RATE           230400  //115200 //230400

//-------- Constants & enums -------------------------------------------------//

/**
* @brief        Enumeration of the different modes of the Bluetooth state machine
*/
typedef enum
{
  UBLOX_IN_CMD_MODE             = 0,    ///<0
  UBLOX_IN_DATA_MODE            = 1,    ///<1
  UBLOX_IN_EDM_MODE             = 2,
  UBLOX_IN_UPGRADE_MODE         = 3,
  
} ublox_modes_e;

typedef enum
{
  //NC = Not Connected
  //C = Connected
  //A = Active
  //NA = Not Active
  //E = Error
  //NE = No Error
  
  BLE_LINK_A            = 0x01,
  BLE_LINK_NA           = 0x00,
  
  BLE_PEER_C            = 0x02,
  BLE_PEER_NC           = 0x00, 
  
  WIFI_LINK_A           = 0x04,
  WIFI_LINK_NA          = 0x00,          
  
  WIFI_STATION_C        = 0x08,
  WIFI_STATION_NC       = 0x00,
  
  CLOUD_E               = 0x10,
  CLOUD_NE              = 0x00,
  
  CAP_E                 = 0x20,
  CAP_NE                = 0x00
  
} comms_led_state_e;

/**
* @brief        Enumeration of the User to Base Unit BLE primitives. 
*               Primitives here imply the first byte of a message that is transmitted
*/
typedef enum
{
  START_PCR                     = 'a',          ///<a - 0x61 - 97(d)
  STOP_PCR                      = 'b',          ///<b - 0x62 - 98(d)
  GET_BU_SOFTWARE_DETAILS       = 'c',          ///<c - 0x63 - 99(d)
  GET_CURRENT_PCR_STATE         = 'd',          ///<d - 0x64 - 100(d)
  GET_HEALTH_SENSOR_READINGS    = 'e',          ///<e - 0x65 - 101(d)
  HANDLE_PICTURES               = 'f',          ///<f - 0x66 - 102(d)
  SET_PCR_LED_INTENSITY         = 'o',          ///<o - 0x6F - 111(d) 
  RESET_MICRO                   = '*',          ///<* - 0x2a - 42(d)
  BLE_HANDSHAKE                 = '@',          ///<@ - 0x40 - 100(d)
  TOGGLE_BU_SAFETY_CHECKS       = 'z',          ///<z - 0x7a - 122(d) 
  EPOCH_TIMESTAMP               = '%',          ///<% - 0x25 - 37(d)
  ODIN_CHIP_CONTROL                = '/'   ///</ - 0X2F - 47(d)
  
} user_to_bu_ble_primitives_e;

/**
* @brief        Enumeration of the Base Unit BLE to User primitives. 
*               Primitives here imply the first byte of a message that is transmitted
*/
typedef enum
{
  PCR_STARTED                   = 'A',          ///<A - 0x41 - 65(d)
  BASE_UNIT_DETAILS             = 'B',          ///<B - 0x42 - 66(d)
  PCR_CURRENT_STATE             = 'C',          ///<C - 0x43 - 67(d)
  TRIGGER_CAMERA                = 'D',          ///<D - 0x44 - 68(d)
  SENSOR_READINGS               = 'E',          ///<E - 0x45 - 69(d)
  TIMESTAMP_RESPONSE            = 'F',          ///<F - 0x46 - 70(d)
  WIFI_CONFIG_RESP              = 'G',          ///<G - 0x47
  
  START_OF_BLE_MESSAGE          = 'X',          ///<X - 0x58 - 88(d)
  
  BASE_UNIT_NOTIFICATION        = 'Y',          ///<Y - 0x59 - 89(d)
  BASE_UNIT_ERROR               = 'Z',          ///<Z - 0x5A - 90(d)
  
} bu_to_user_ble_primitives_e;

//-------- Structs & typedefs ------------------------------------------------//


//-------- Global variables --------------------------------------------------//


//-------- Global function prototypes ----------------------------------------//

extern void initialise_ublox_module(void);
extern void reset_ublox_module(void);
extern UART_HandleTypeDef* get_uart3_address(void);

extern uint8_t* get_uart3_tx_buffer(void);
extern void MX_UART3_Init(void);
extern void MX_UART3_DeInit(void);
extern void change_uart3_baud_rate(uint32_t baud_rate);

extern uint8_t is_ublox_module_configured(void);

extern HAL_StatusTypeDef write_bytes_to_ublox_uart3(uint8_t *p_buffer, uint16_t len_bytes);
extern void process_uart3_interrupt_rx_data(void);
extern void process_ublox_uart3_rx_msgs(void);

extern void transmit_picture_req_to_ble_user(uint8_t cycle_no, uint8_t cam_pic_id);
extern void transmit_base_unit_error(uint8_t system_error);
extern void transmit_base_unit_notification(uint8_t system_notification);

extern void process_start_pcr_req(uint8_t output_choice);
extern void process_stop_pcr_req(uint8_t output_choice);
extern void get_bu_software_details(uint8_t output_choice);
extern void get_current_pcr_state(uint8_t output_choice);
extern void transmit_health_sensor_readings(void);

extern void transmit_ublox_config_msg(void);

extern void read_one_byte_from_uart3(void);

extern void change_ublox_current_mode(uint8_t mode);

extern void reset_ublox_rx_buffer(void);

extern void process_ublox_at_event_pkt(const uint8_t* rx_data, uint16_t len_bytes);

extern void set_wifi_link_conn_status(uint8_t status);

extern uint8_t is_wifi_link_connected(void);

extern void set_ble_chan_conn_status(uint8_t status);

extern uint8_t is_ble_chan_connected(void);

extern void set_ble_peer_status(uint8_t status);

extern void set_ble_chan_id(uint8_t chan_id);

extern uint8_t get_ble_chan_id(void);

extern void set_ble_peer_id(uint8_t peer_id);

extern uint8_t get_ble_peer_id(void);

extern uint8_t is_ble_peer_connected(void);

extern uint8_t is_wifi_chan_connected(void);

extern uint8_t is_wifi_peer_connected(void);

extern void set_wifi_chan_id(uint8_t chan_id);

extern uint8_t get_wifi_chan_id(void);

extern void set_wifi_peer_id(uint8_t peer_id);

extern uint8_t get_wifi_peer_id(void);

extern void set_wifi_chan_conn_status(uint8_t status);

extern void set_wifi_peer_status(uint8_t status);

extern void process_ublox_data_event_pkt(const uint8_t* rx_data, uint16_t len_bytes);

extern void send_message_over_ble(const uint8_t* p_message, uint8_t len_bytes);

extern uint8_t update_ublox_firmware(void);

extern uint8_t configure_wifi_link(void);

extern uint8_t activate_wifi_link(void);

extern uint8_t deactivate_wifi_link(void);

extern uint8_t store_wifi_ssid(const char *ssid);

extern uint8_t store_wifi_pwd(const char *pwd);

extern uint8_t write_wifi_details_to_sd(void);

extern uint8_t store_web_addr(const char *web_addr);

extern uint8_t write_web_addr_to_sd(void);

extern const uint8_t* get_ublox_fw_ver_bytes(void);

extern void set_comms_indicator_leds();

extern void set_cloud_conn_error_status(uint8_t status);

extern void set_capsule_conn_error_status(uint8_t status);

extern void set_wifi_link_active_status(uint8_t status);

extern void read_wifi_details_from_sd(void);

extern void read_cloud_details_from_sd(void);

extern void print_wifi_details(void);

extern void print_cloud_details(void);

//----------------------------------------------------------------------------//

#endif /* BLUETOOTH_H */

