/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#ifdef OBSOLETE_MAIN
#include <events/mbed_events.h>
#include <mbed.h>
#include "ble/BLE.h"
#include "LEDService.h"
#include "ble/services/UARTService.h"

UARTService *uart;

DigitalOut alivenessLED(LED1, 0);
DigitalOut actuatedLED(LED2, 0);
RawSerial pc(USBTX, USBRX); // tx, rx
const static char     DEVICE_NAME[] = "BLE-UART";
static const uint16_t uuid16_list[] = {LEDService::LED_SERVICE_UUID};
#define BUFFER_LEN 256
#define TX_BUFFER_LEN 4*256
#define RX_BUFFER_LEN 4*256
char buffer[BUFFER_LEN];
uint8_t TxBuffer[TX_BUFFER_LEN];
uint8_t RxBuffer[RX_BUFFER_LEN];
static EventQueue eventQueue(/* event count */ 10 * EVENTS_EVENT_SIZE);

LEDService *ledServicePtr;

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    (void) params;
    BLE::Instance().gap().startAdvertising();
}

void blinkCallback(void)
{
    alivenessLED = !alivenessLED; /* Do blinky on LED1 to indicate system aliveness. */
}

void EchoBleUartReceived()
{
    uart->writeString(buffer);
    uart->writeString("\n");    //flushes uart output buffer and sends data    
}


/**
 * This callback allows the LEDService to receive updates to the ledState Characteristic.
 *
 * @param[in] params
 *     Information about the characterisitc being updated.
 */
void onDataWrittenCallback(const GattWriteCallbackParams *params) {
    if ((params->handle == ledServicePtr->getValueHandle()) && (params->len == 1)) {
        actuatedLED = *(params->data);
    }
    else if ((uart != NULL) && (params->handle == uart->getTXCharacteristicHandle())) {
        uint16_t bytesRead = params->len;
        
        pc.printf("received %u bytes\n\r ", bytesRead);
        
        if(bytesRead >= 255){
            pc.printf("Overflow command %u n\r ", bytesRead);
            bytesRead = 255;
        }
        
        unsigned index = 0;
        for (; index < bytesRead; index++) {
            buffer[index] = params->data[index];
        }
        
        buffer[index++] = 0;
        
        pc.printf("Data : %s ",buffer);
        pc.printf("\r\n");
        eventQueue.call(EchoBleUartReceived);

    }
}


/**
 * This function is called when the ble initialization process has failled
 */
void onBleInitError(BLE &ble, ble_error_t error)
{
    /* Initialization error handling should go here */
}

void printMacAddress()
{
    /* Print out device MAC address to the console*/
    Gap::AddressType_t addr_type;
    Gap::Address_t address;
    BLE::Instance().gap().getAddress(&addr_type, address);
    pc.printf("DEVICE MAC ADDRESS: ");
    for (int i = 5; i >= 1; i--){
        pc.printf("%02x:", address[i]);
    }
    pc.printf("%02x\r\n", address[0]);
}

/**
 * Callback triggered when the ble initialization process has finished
 */
void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE&        ble   = params->ble;
    ble_error_t error = params->error;
    if (error != BLE_ERROR_NONE) {
        /* In case of error, forward the error handling to onBleInitError */
        onBleInitError(ble, error);
        return;
    }

    /* Ensure that it is the default instance of BLE */
    if(ble.getInstanceID() != BLE::DEFAULT_INSTANCE) {
        return;
    }

    ble.gap().onDisconnection(disconnectionCallback);
    ble.gattServer().onDataWritten(onDataWrittenCallback);

    bool initialValueForLEDCharacteristic = false;
    ledServicePtr = new LEDService(ble, initialValueForLEDCharacteristic);
    /* Setup primary service */
    uart = new UARTService(ble);

    /* setup security */
    error = ble.securityManager().init();
    if(error != BLE_ERROR_NONE)
    {
        pc.printf("\nBLE Security manager initialization failed!!\n");
    }
    /* setup advertising */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    //ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    /* set up the services that can be discovered */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,(const uint8_t *)UARTServiceUUID_reversed, sizeof(UARTServiceUUID_reversed));
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(1000); /* 1000ms. */
    ble.gap().startAdvertising();

    printMacAddress();
}

void scheduleBleEventsProcessing(BLE::OnEventsToProcessCallbackContext* context) {
    BLE &ble = BLE::Instance();
    eventQueue.call(Callback<void()>(&ble, &BLE::processEvents));
}

static int uartExpectedRcvCount = 0;
static int uartCharRcvCount = 0;
static bool UartBusy = false;
int WriteUartBytes(const uint8_t * txBuffer, size_t bufSize, int txLen)
{
    if(txLen > bufSize)
    {
        txLen = bufSize;
    }
    for(int i=0;i<txLen;i++)
    {
        pc.putc(txBuffer[i]);
    }
    // return number of bytes written to UART
    return txLen;
}

void UartRxcallback_ex() {
    if(uartCharRcvCount >= uartExpectedRcvCount)
    {
        int x = pc.getc();
        return;
    }
    if(uartCharRcvCount == 0)
    {
        pc.printf("\nFirst Call to UART attach callback!!\n");
    }
    // Note: you need to actually read from the serial to clear the RX interrupt
    RxBuffer[uartCharRcvCount] = (uint8_t) pc.getc();
    uartCharRcvCount++;
    if(uartCharRcvCount >= uartExpectedRcvCount)
    {
        alivenessLED = !alivenessLED; /* Do blinky on LED1 to indicate system aliveness. */
        pc.printf("\nNumber of Received Bytes = %d\n\n", uartCharRcvCount);
        pc.printf("--- Writing back received bytes --- \n");
        int n;
        n = WriteUartBytes(RxBuffer, TX_BUFFER_LEN, uartCharRcvCount);
        UartBusy = false;
    }
}

void BackGndUartRead(uint8_t * rxBuffer, size_t bufSize, int rxLen)
{
    UartBusy = true;
    pc.printf("Setting up background UART read -  rxLen = %d\n", rxLen);
    uartCharRcvCount = 0;
    if(rxLen > bufSize)
    {
        rxLen = bufSize;
    }
    uartExpectedRcvCount = rxLen;
    pc.attach(&UartRxcallback_ex);
    pc.printf("\nBackground UART read setup completed\n\n");
    //for(int i=0;i<rxLen;i++)
    //{
    //    rxBuffer[i] = (uint8_t) pc.getc();
    //}
    // return number of bytes written to UART
    //return rxLen;
}

int ReadUartBytes(uint8_t * rxBuffer, size_t bufSize, int rxLen, bool echo)
{
    UartBusy = true;
    if(rxLen > bufSize)
    {
        rxLen = bufSize;
    }
    for(int i=0;i<rxLen;i++)
    {
        rxBuffer[i] = (uint8_t) pc.getc();
        if(echo)pc.putc(rxBuffer[i]);
    }
    UartBusy = false;
    //return number of bytes written to UART
    return rxLen;
}


void checkUartReceive()
{
    //pc.printf("Hello World!\n\r");
    char cbuf[100];
    int rxCnt=0;
    while(pc.readable()) {
        //pc.printf("uartCharRcvCount = %d\n\r", uartCharRcvCount++);
        cbuf[rxCnt++] = pc.getc();
        //pc.putc(pc.getc() + 1); // echo input back to terminal
    }
    cbuf[rxCnt] = NULL;
    if(rxCnt > 0)
    {
        pc.printf("received %d chars\n", rxCnt);
        pc.printf("%s\n", cbuf);
    }
    
}
uint64_t lastTime = 0;
uint64_t now = 0;
uint32_t callCount = 0;
void HelloUart()
{
    //if(UartBusy)return;
    // 64-bit time doesn't wrap for half a billion years, at least
    lastTime = now;
    now = Kernel::get_ms_count();
    callCount++;
    pc.printf("\nHello : %d secs elapsed : CallCount = %d \n", uint32_t(now - lastTime), callCount);
}

Thread t;
int main()
{
    pc.baud(115200);
    eventQueue.call_every(500, blinkCallback);
    eventQueue.call_every(60000, HelloUart);
    //eventQueue.call_every(1000, checkUartReceive);

    BLE &ble = BLE::Instance();
    ble.onEventsToProcess(scheduleBleEventsProcessing);
    ble.init(bleInitComplete);
    for(int i=0;i<255;i++)
    {
        pc.putc(i);
    }
    int n;
    //ReadUartBytes(RxBuffer, RX_BUFFER_LEN, 4);    
    pc.printf("\n\n\nEnter # of expected bytes: ");
    n = ReadUartBytes(RxBuffer, RX_BUFFER_LEN, 4, true);
    int rxLen = (int) 100*(RxBuffer[0]-'0') + 10*(RxBuffer[1]-'0') + (RxBuffer[0]-'0');
    pc.printf("\n\nExpected # of Received Bytes = %d\n", rxLen);
    BackGndUartRead(RxBuffer, RX_BUFFER_LEN, rxLen);
    //pc.printf("--- Writing back received data --- \n\n");
    //n = WriteUartBytes(RxBuffer, TX_BUFFER_LEN, rxLen);
    //pc.write("\n\ntesting Serial Write\n", 40); //, checkUartReceive, SERIAL_EVENT_TX_COMPLETE);
    //eventQueue.dispatch_forever();
    t.start(callback(&eventQueue, &EventQueue::dispatch_forever));
    return 0;
}
#endif