/**
******************************************************************************
* @file         debug.c
* @brief        Source file containing logic responsible for:
* @details
*
******************************************************************************
* @author       TTP
* @copyright (c) 2017 TTP (The Technology Partnership) plc
*
******************************************************************************
*/

//-------- Includes ----------------------------------------------------------//

#include "debug.h"
#include <mbed.h>
//#include "clocks.h"

//-------- Defines -----------------------------------------------------------//

#define FILE_CODE       "dbgm"
#define ERROR       "err"

#define DBG_RX_BUFFER_SIZE_BYTES        (1024)
#define DBG_RX_BUFFER_MASK              (DBG_RX_BUFFER_SIZE_BYTES - 1)

#define DBG_TX_BUFFER_SIZE_BYTES        (1024) 

#if (DBG_RX_BUFFER_SIZE_BYTES & DBG_RX_BUFFER_MASK)
    #error Rx buffer size is not power of 2
#endif

//-------- Constants & enums -------------------------------------------------//


//-------- Structs & typedefs ------------------------------------------------//

//-------- Global Variables --------------------------------------------------//

/**
*@brief         Global array of length ::DBG_TX_BUFFER_SIZE_BYTES to store
*               the current debug print statement of the firmware. 
*/
static char g_dbg_buffer[DBG_TX_BUFFER_SIZE_BYTES];

//-------- Static Variables --------------------------------------------------//

static const char dbg_tx_terminate_string[] = {'\n', '\r'}; ///<Array of characters representing the terminate sequence of a debug transmit string

static uint8_t current_debug_level; ///<Variable keeping track of the current debug level

static char fw_ver_string[REL_FW_VER_STRING_LEN_BYTES + SVN_FW_VER_STRING_LEN_BYTES];
static uint8_t fw_ver_bytes[5];

static data_mode_e dbg_data_mode;

//Rx buffer to be used for both ASCII and binary mode
static uint8_t dbg_rx_buffer[DBG_RX_BUFFER_SIZE_BYTES];
static volatile uint16_t dbg_rx_pos;
static volatile uint32_t dbg_rx_count;
static volatile uint16_t dbg_rx_cmd_count;

//static char g_dbg_buffer[DBG_TX_BUFFER_SIZE_BYTES];

//-------- Static function prototypes ----------------------------------------//

static void set_fw_ver_string(void);
static void set_fw_ver_bytes(void);
static char get_rx_byte_from_buffer(void);
static const char* debug_level_to_string(uint8_t debug_level);

//-------- Function implementations ------------------------------------------//
#ifdef BOX_MICRO_CODE
static inline void disable_dbg_rx_intrup(void)
{
  __HAL_PCD_DISABLE(&hpcd_USB_OTG_FS);
}

static inline void enable_dbg_rx_intrup(void)
{
  __HAL_PCD_ENABLE(&hpcd_USB_OTG_FS);
}
#endif
/**
* @brief        Starting function to supervise the
*               - Initialisation of UART2 responsible for acting as the debug port for the firmware
*               - Initialisation of the UART2 transmit and receive buffers
*               - Initial read from UART2
*
* @param        None 
* @retval       None 
*/
#define DEBUG_ENABLED
void initialise_debug(void)
{   
  memset(g_dbg_buffer, 0, sizeof(g_dbg_buffer));
  
  //. Set the version strings
  set_fw_ver_string();
  set_fw_ver_bytes();
#ifdef DEBUG_ENABLED  
  current_debug_level = (LOG | ERR | TXT | DBG); //NONE; // 
#else
  current_debug_level = NONE; // 
#endif
  dbg_data_mode = DATA_ASCII;
  dbg_rx_pos = 0;
  dbg_rx_count = 0;
  dbg_rx_cmd_count = 0;
}

/**
* @brief
* @param
* @param
* @retval
*/
const char* get_dbg_tx_terminate_string(void)
{
  if (dbg_data_mode == DATA_ASCII)
  { 
    return dbg_tx_terminate_string;
  }
  else
  {  
    return NULL;
  }
}

/**
* @brief
* @param
* @param
* @retval
*/
uint8_t get_dbg_tx_terminate_string_len(void)
{
  if (dbg_data_mode == DATA_ASCII)
  {  
    return sizeof(dbg_tx_terminate_string);
  }
  else
  {  
    return 0;
  }
}

static const char* debug_level_to_string(uint8_t debug_level)
{
  switch(debug_level)
  {
  case LOG:
    return "LOG";
      
  case ERR:
    return "ERR";
    
  case DBG:
    return "DBG";
    
  case TXT:
    return "TXT";
      
  default:
    return "INV"; //Invalid
  }
}


//void dbg_printf(uint8_t debug_level, ...)
//{
//    dbg_print(FILE_CODE, __LINE__, debug_level, ...);
//}
void dbg_print(const char *file_code, uint16_t line_number, 
                uint8_t debug_level, const char *text, ...)
{
  //const char *file_code, uint16_t line_number,
                
  if (current_debug_level & debug_level)
  {
    //Re-init the gloabl debug buffer  
    memset(g_dbg_buffer, 0, sizeof(g_dbg_buffer));
    uint16_t len_bytes = 0;
    
    if (debug_level != TXT)
    {
      //sprintf(g_dbg_buffer, "|%s|%10u|", debug_level_to_string(debug_level), Kernel::get_ms_count());
      sprintf(g_dbg_buffer, "|%s|%s|%4d|%10u|", debug_level_to_string(debug_level), file_code, line_number, Kernel::get_ms_count());
      len_bytes = strlen(g_dbg_buffer);
    }
    
    va_list args;
    va_start(args, text); 
    vsnprintf(&g_dbg_buffer[len_bytes], (sizeof(g_dbg_buffer) - len_bytes - 2), text, args);
    printf("%s", g_dbg_buffer);
    va_end(args);
    
    len_bytes = strlen(g_dbg_buffer);
    g_dbg_buffer[len_bytes++] = '\r';
    g_dbg_buffer[len_bytes++] = '\n';
#ifdef WRITE_TO_FILE_ENABLED   
    if (debug_level == LOG || debug_level == ERROR)
    {
      write_to_file(SYS_LOG, (const uint8_t *)g_dbg_buffer, len_bytes);
    }
    
    if (dbg_data_mode == DATA_ASCII)
    {
      transmit_bytes_over_usb((uint8_t *)g_dbg_buffer, len_bytes);    
    }
#endif
  }
}

/**
    Function used to send data during firmware update (in binary mode)
*/
#ifdef WRITE_TO_FILE_ENABLED  
void debug_send_data(uint8_t *pData, uint16_t length)
{
  transmit_bytes_over_usb(pData, length);
}
#endif

/**
* @brief        Function to set the debug level. Currently only 2 debug levels
*               are provided as mentioned in ::debug_log_level_e.
*               LOG is constantly enabled
*
* @param        (uint8_t) debug_level  
* @retval       None
*/
void set_debug_level(uint8_t debug_level)
{
  current_debug_level = (LOG | ERR | TXT | debug_level);
  dbg_printf(LOG, "Current debug level set to %02x", current_debug_level);
}

/**
* @brief        Function to get the current debug level set. 
*
* @param        None 
* @retval       (uint8_t) current_debug_level
*/
uint8_t get_debug_level(void)
{
  return current_debug_level;
}

/**
* @brief        Function to set the micro firmware version 
*               Firmware version follows the following format: 
*               major.minor.patch.svn_revision
*               
*               If 'svnversion' is not installed on the machine, the svn
*               revision is returned null. In that instance, the svn revision
*               is set to n.a
*
* @param        None 
* @retval       None
*/
static void set_fw_ver_string(void)
{
  memset(fw_ver_string, 0, sizeof(fw_ver_string));
  
  memcpy(fw_ver_string, REL_FW_VER_STRING, REL_FW_VER_STRING_LEN_BYTES);
  
  uint8_t rel_fw_string_len_bytes = strlen(REL_FW_VER_STRING);
  
  fw_ver_string[rel_fw_string_len_bytes] = '.';
#ifdef BOX_MICRO_CODE   
  if (strlen(SVN_FW_VERSION_STRING) == 0)
  {
    //svnversion is not installed on the machine
    memcpy(&fw_ver_string[rel_fw_string_len_bytes + 1], "n.a", 4);
  }
  else
  {
    memcpy(&fw_ver_string[rel_fw_string_len_bytes + 1], SVN_FW_VERSION_STRING, SVN_FW_VER_STRING_LEN_BYTES);
  }
#endif
}

/**
* @brief        Function to get the firmware version 
*
* @param        None 
* @retval       (const char *) fw_ver_string
*/
const char* get_fw_ver_string(void)
{
  return fw_ver_string;
}

static void set_fw_ver_bytes(void)
{
  memset(fw_ver_bytes, 0, sizeof(fw_ver_bytes));
  
  uint16_t val = 0;
  sscanf(fw_ver_string, "%hhx.%hhx.%hhx.%hx", &fw_ver_bytes[0], &fw_ver_bytes[1], &fw_ver_bytes[2], &val);
  
  fw_ver_bytes[3] = (val >> 8) & 0xff;
  fw_ver_bytes[4] = val & 0xff;
}

const uint8_t* get_fw_ver_bytes(void)
{
  return fw_ver_bytes;
}

void debug_setDataMode(data_mode_e mode)
{  
  dbg_data_mode = mode;
}

void print_debug_hex(const uint8_t* p_buffer, uint16_t len)
{
  //print out the rxvd EDM bytes  
  
  uint16_t quo = len / 8;
  uint16_t rem = len % 8;
  
  uint16_t i = 0;
  uint16_t j = 0;
  for (i = 0; i < quo; i++)
  { 
    dbg_printf(LOG, "[%3d]: %02x %02x %02x %02x %02x %02x %02x %02x", j,
                    p_buffer[j + 0], p_buffer[j + 1], p_buffer[j + 2], p_buffer[j + 3],
                    p_buffer[j + 4], p_buffer[j + 5], p_buffer[j + 6], p_buffer[j + 7]);
    j += 8;
  }
  
  if (rem == 7)
  {
    dbg_printf(LOG, "[%3d]: %02x %02x %02x %02x %02x %02x %02x", j,
                    p_buffer[j + 0], p_buffer[j + 1], p_buffer[j + 2], p_buffer[j + 3],
                    p_buffer[j + 4], p_buffer[j + 5], p_buffer[j + 6]);      
  }
  else if (rem == 6)
  { 
    dbg_printf(LOG, "[%3d]: %02x %02x %02x %02x %02x %02x", j,
                    p_buffer[j + 0], p_buffer[j + 1], p_buffer[j + 2], p_buffer[j + 3],
                    p_buffer[j + 4], p_buffer[j + 5]);     
  }
  else if (rem == 5)
  {
    dbg_printf(LOG, "[%3d]: %02x %02x %02x %02x %02x", j,
                    p_buffer[j + 0], p_buffer[j + 1], p_buffer[j + 2], p_buffer[j + 3],
                    p_buffer[j + 4]);      
  }
  else if (rem == 4)
  {
    dbg_printf(LOG, "[%3d]: %02x %02x %02x %02x", j,
                    p_buffer[j + 0], p_buffer[j + 1], p_buffer[j + 2], p_buffer[j + 3]);   
  }
  else if (rem == 3)
  {
    dbg_printf(LOG, "[%3d]: %02x %02x %02x", j,
                    p_buffer[j + 0], p_buffer[j + 1], p_buffer[j + 2]);      
  }
  else if (rem == 2)
  {
    dbg_printf(LOG, "[%3d]: %02x %02x", j, p_buffer[j + 0], p_buffer[j + 1]);     
  }
  else if (rem == 1)
  {
    dbg_printf(LOG, "[%3d]: %02x", j, p_buffer[j]);    
  }  

  dbg_printf(LOG, ""); 
}
