#ifndef __COMMON_TYPES_H__
#define __COMMON_TYPES_H__
#include <mbed.h>
#include <string>
#include "ble/BLE.h"
#include "SecurityManager.h"
#include "http_request.h"
#include "common_config.h"
#define BLE_MAX_DEVICE_NAME_LEN 10
#define MAX_SSID_LEN 32
#define MAX_PASSKEY_LEN 32
#define EDM_START_BYTE       0xAA
#define EDM_STOP_BYTE        0x55
#define EDM_HDR_LEN          6
#define MAX_EDM_PAYLOAD_LEN  (4096-1)
#define MIN_EDM_PAYLOAD_LEN (2)
#define MAX_HTTP_HEADER_LINES
#define MAX_URL_LEN          100

/** ble configuration structure
*/
typedef struct  {
    char      deviceName[BLE_MAX_DEVICE_NAME_LEN]; /* BLE Device Name */
    uint16_t  advInterval; /* advertising interval in msecs */
    uint16_t  advTimeout;  /* advertising timeout in secs */
    //Passkey_t pairingKey;  /* pairing Key */
    uint8_t   pairingKey[6];  /* pairing Key */
} ble_config_t;

/** ble configuration structure
*/
typedef struct  {
    char             ssid[MAX_SSID_LEN]; /* WiFi SSID */
    char             pass[MAX_PASSKEY_LEN]; /* WiFi Passkey */
    nsapi_security_t security;  /* WiFi security */
} wifi_config_t;

/** ble configuration structure
*/
typedef struct  {
    wifi_config_t wifi_config; /* wifi configuration */
    ble_config_t ble_config;   /* ble configuration */
} app_config_t;


typedef enum
{
  WIFI_CONFIG_RESET,
  WIFI_CONFIG_STORE,
  WIFI_CONFIG_LOAD,
  WIFI_CONFIG_ACTIVATE,
  WIFI_CONFIG_DEACTIVATE
}action_id_t;


typedef enum
{
  WIFI_CMD_NONE,
  WIFI_CMD_CONFIG,
  WIFI_CMD_INTERNET_CONFIG,
  WIFI_CMD_SCAN,
  WIFI_CMD_DETAILED_SCAN,
  WIFI_CMD_CONNECT,
  WIFI_CMD_DISCONNECT,
  WIFI_CMD_NETWORK_STATUS,
  WIFI_CMD_WIFI_STATUS,
  WIFI_CMD_SEND_HTTPS_REQ,
  WIFI_CMD_SEND_HTTP_REQ
}wifi_cmd_t;

typedef enum
{
  AT_RESP_NONE,
  AT_SCAN_RESP,
  AT_DETAILED_SCAN_RESP,
  AT_CONNECT_RESP,
  AT_DISCONNECT_RESP,
  WIFI_RECONNECT_INFO,
  WIFI_DISCONNECT_INFO,
  AT_CONFIG_RESP,
  AT_INTERNET_CONFIG_RESP,
  AT_NETWORK_STATUS_RESP,
  AT_WIFI_STATUS_RESP,
  AT_HTTPS_RESP,
  AT_HTTPS_RESP_DOWNLOAD,
  AT_HTTP_RESP,
  AT_HTTP_RESP_DOWNLOAD,
  AT_COMMAND_FAILED
}at_cmd_resp_t;

typedef enum edm_msg_id
{
  CONNECT_MSG_ID    = 0x0010,
  DISCONNECT_MSG_ID = 0x0020,
  DATA_MSG_ID       = 0x0030,
  AT_MSG_ID         = 0x0040   
}edm_msg_id_t;


typedef enum edm_msg_type
{
  EVENT_MSG_TYPE        = 0x01,
  INDICATION_MSG_TYPE   = 0x02,
  RESPONSE_MSG_TYPE     = 0x03,
  REQUEST_MSG_TYPE      = 0x04,
  CONFIRMATION_MSG_TYPE = 0x05,
  COMMAND_MSG_TYPE      = 0x06
}edm_msg_type_t;

typedef enum channel_id
{
  WIFI_CHANNEL          = 0x00,
  BLE_CHANNEL           = 0x01,
  NO_CHANNEL            = 0xFF
}channel_id_t;



typedef enum at_data_mode
{
  AT_CMD_DATA_MODE,
  AT_STD_DATA_MODE,
  AT_EXT_DATA_MODE
}at_data_mode_t;

typedef enum edm_payload_id
{
  CONNECT_EVENT    = 0x0011,
  DISCONNECT_EVENT = 0x0021,
  DATA_EVENT       = 0x0031,
  DATA_COMMAND     = 0x0036,
  AT_REQUEST       = 0x0044,
  AT_CONFIRMATION  = 0x0045,
  AT_EVENT         = 0x0041
}edm_payload_id_t;

typedef enum conn_scheme
{
  INVALID_SCHEME,
  ALWAYS_CONNECTED,
  EXTERNAL_CONNECT
}conn_scheme_t;


typedef struct {
    wifi_cmd_t    wifi_cmd;   /* wifi command */
    uint8_t       padding[4-sizeof(wifi_cmd_t)];   /* padding */
} wifi_cmd_message_t;

typedef struct {
    at_cmd_resp_t    at_resp;   /* AT response */
} at_resp_message_t;

typedef struct {
    wifi_cmd_t    wifi_cmd;              /* wifi data command */
    uint8_t       padding[4-sizeof(wifi_cmd_t)];   /* padding */
    int           dataLen;              /* size of data in buffer */
    uint8_t       buffer[TX_BUFFER_LEN]; /* buffer length     */
} wifi_data_msg_t;

typedef struct {
    at_cmd_resp_t at_resp;              /* AT response */
    int           dataLen;              /* size of data in buffer */
    uint8_t       buffer[TX_BUFFER_LEN]; /* buffer length     */
} at_data_msg_t;

typedef struct {
    http_method   method;       /* POST/GET etc... */
    char *        request_URI;  /* request URI/path */
    char *        http_version; /* http verstion HTTP/1.1 */
    char *        hostName;     /* host name */
    char *        AcceptVal;    /* types accepted */
    char *        contentType;  /* content type */
    char *        contentLen;   /* content length   */
    uint8_t *     body;         /* body     */
    uint8_t       buffer[TX_BUFFER_LEN - 8*sizeof(char *) 
                                     - sizeof(http_method)]; // buffer   
} http_request_t;

typedef struct {
    uint8_t       startByte;   /* start Byte 0xAA */
    uint16_t      payloadLen;  /* payload Length */
    uint16_t      payloadID;   /* payload identifier     */
    channel_id_t  channel_id;   /* channel_id_t*/
} edm_header_t;

typedef struct {
    string        contentType;  /* content type */
    int           bodyLen;      /* body length     */
    uint8_t       body[900];    /* body     */
} http_response_t;

typedef struct {
    uint8_t       peer_id          ;  /* peer id */
    conn_scheme_t connectionScheme;   /* connection scheme     */
    char          url[MAX_URL_LEN];   /* url string     */
} internet_config_t;

#endif  // __COMMON_TYPES_H__