#include "WiFiManager.h"
#include "common_config.h"


WiFiManager::WiFiManager(wifi_config_t wifi_config, WiFiInterface *wifi, 
                         events::EventQueue &event_queue, 
                         MemoryPool<wifi_cmd_message_t, 16> *aT2WiFimPool, 
                         Queue<wifi_cmd_message_t, 16> *aT2WiFiCmdQueue, 
                         MemoryPool<at_resp_message_t, 16> *wiFi2ATmPool, 
                         Queue<at_resp_message_t, 16> *wiFi2ATCmdQueue, 
                         MemoryPool<wifi_data_msg_t, PQDSZ> *aT2WiFiDatamPool, 
                         Queue<wifi_data_msg_t, PQDSZ> *aT2WiFiDataQueue, 
                         MemoryPool<at_data_msg_t, PQDSZ> *wiFi2ATDatamPool, 
                         Queue<at_data_msg_t, PQDSZ> *wiFi2ATDataQueue) 
:
     wifi_config(wifi_config),
     network(wifi),
    _event_queue(event_queue),
    _aT2WiFimPool(aT2WiFimPool),
    _aT2WiFiCmdQueue(aT2WiFiCmdQueue),
    
    _wiFi2ATmPool(wiFi2ATmPool),
    _wiFi2ATCmdQueue(wiFi2ATCmdQueue),
    
    _aT2WiFiDatamPool(aT2WiFiDatamPool),
    _aT2WiFiDataQueue(aT2WiFiDataQueue),
    
    _wiFi2ATDatamPool(wiFi2ATDatamPool),
    _wiFi2ATDataQueue(wiFi2ATDataQueue)

{
 lastScanCount = 0;
 wifiCmd = WIFI_CMD_NONE;
 internet_config.connectionScheme = ALWAYS_CONNECTED; // set default connection scheme
 is_connected = false;
 chunkNum = 0;
 socket = NULL;
 https_connection_active = false;
}

WiFiManager::~WiFiManager()
{
}
  
bool WiFiManager::queueATresponse(at_cmd_resp_t resp){
#ifndef USE_MALLOC_FOR_COMMAND_MEMORY_POOL
    at_resp_message_t *atResp  = _wiFi2ATmPool->alloc();
#else
    at_resp_message_t *atResp  = (at_resp_message_t *) malloc(sizeof(at_resp_message_t));
#endif
    if(atResp == NULL) return false; // queue full;
    atResp->at_resp            = resp;
    _wiFi2ATCmdQueue->put(atResp);
    return true;
}


bool WiFiManager::queueWiFiDataResponse(at_data_msg_t at_resp){
    at_data_msg_t *atData = _wiFi2ATDatamPool->alloc();
    if(atData == NULL) return false; // queue full;
    atData->at_resp        = at_resp.at_resp;
    atData->dataLen        = at_resp.dataLen;
    memcpy(atData->buffer, at_resp.buffer, at_resp.dataLen);
    _wiFi2ATDataQueue->put(atData);
    printf("[WIFI MAN] queued data size = %d : at_resp = %d\n", at_resp.dataLen, at_resp.at_resp);
    return true;
}



void WiFiManager::runMain(){
    nsapi_error_t error;
    bool httpReqResult;
    printf("\r\n [WIFI MAN]  Thread Id = %X\r\n", (uint32_t)ThisThread::get_id());
    while(true){
        dequeueWiFiCommands();
        dequeueATdataResponse();
        switch(wifiCmd){
            case WIFI_CMD_NONE:
                // IDLE STATE
                break;
            case WIFI_CMD_SCAN:
                error = scanNetworks();
                wifiCmd = WIFI_CMD_NONE;
                queueATresponse(AT_SCAN_RESP);
                break;
            case WIFI_CMD_DETAILED_SCAN:
            {
                nsapi_size_or_error_t cnt_err;
                cnt_err = getAvailableAPs(lastScanCount);
                wifiCmd = WIFI_CMD_NONE;
                queueATresponse(AT_DETAILED_SCAN_RESP);
                break;
            }
            case WIFI_CMD_CONNECT:
            {
                error = connect();
                int secCount = 0;
                while(secCount++ < WIFI_CONNECT_TIMEOUT_SECS || is_connected==false){
                    wait(1); // wait 1 sec
                }
                wifiCmd = WIFI_CMD_NONE;
                if(is_connected==false){
                    printf("[WIFI MAN] +++ WIFI CONNECTION TIMEOUT +++ \r\n");
                    //queueATresponse(AT_COMMAND_FAILED);
                    responseString = (char *) malloc(100);
                    sprintf(responseString, "\r\n+UUTIMEOUT\r\n");
                    sendATresponseString(AT_COMMAND_FAILED);
                }
                else {
                    sendATresponseString(AT_CONNECT_RESP);
                }
                break;
            }
            case WIFI_CMD_DISCONNECT:
                error = disconnect();
                wifiCmd = WIFI_CMD_NONE;
                queueATresponse(AT_DISCONNECT_RESP);
                break;
            case WIFI_CMD_CONFIG:
                set_WIFI_CONFIG();
                wifiCmd = WIFI_CMD_NONE;
                queueATresponse(AT_CONFIG_RESP);
                break;
            case WIFI_CMD_INTERNET_CONFIG:
                set_internet_config();
                queueATresponse(AT_INTERNET_CONFIG_RESP);
                wifiCmd = WIFI_CMD_NONE;
                break;
            case WIFI_CMD_NETWORK_STATUS:
                getNetworkStatus();
                sendATresponseString(AT_NETWORK_STATUS_RESP);
                wifiCmd = WIFI_CMD_NONE;
                break;
            case WIFI_CMD_WIFI_STATUS:
                getWiFiStatus();
                sendATresponseString(AT_WIFI_STATUS_RESP);
                wifiCmd = WIFI_CMD_NONE;
                break;
            case WIFI_CMD_SEND_HTTPS_REQ:
                printf("before call to send http request \n");
                print_memory_info();
                httpReqResult = createHttpsRequest();
                if(httpReqResult == false)
                {
                    sendATresponseString(AT_COMMAND_FAILED);
                }
                printf("after call to send http request \n");
                print_memory_info();
                wifiCmd = WIFI_CMD_NONE;
                break;
            case WIFI_CMD_SEND_HTTP_REQ:
                break;
            default:
                break;
        }
        wait_ms(100); // 
    }
    
}


void  WiFiManager::sendATresponseString(at_cmd_resp_t at_cmd)
{
    int strLen = strlen(responseString) + 1;
    at_data_resp = new at_data_msg_t;
    // create message pointer for response header generation
    char * msgPtr = (char *)at_data_resp->buffer;
    // set string length 
    at_data_resp->dataLen = strLen;
    memcpy(at_data_resp->buffer, responseString, strLen);
    free(responseString);
    // package and send on wifi data queue    
    at_data_resp->at_resp = at_cmd;
    bool queueResult = true;
    int wait_count = 0;
    do
    {
        if(!queueResult){
            wait_count++;
            printf("ATCMD Queue full waiting %d ms so far...\n", wait_count*10);
            wait_ms(10);
        }
        queueResult = queueWiFiDataResponse(*at_data_resp);
    }while(queueResult == false);
    delete at_data_resp;
}                                            

bool  WiFiManager::dequeueWiFiCommands(){
    if(wifiCmd != WIFI_CMD_NONE) return false; // busy
    osEvent evt = _aT2WiFiCmdQueue->get(0);
    if(evt.status == osEventMessage){
        wifi_cmd_message_t *cmd = (wifi_cmd_message_t*)evt.value.p;
        setNextCommand(cmd->wifi_cmd);
#ifndef USE_MALLOC_FOR_COMMAND_MEMORY_POOL
        _aT2WiFimPool->free(cmd);
        cmd = NULL;
#else
        free(cmd);
        cmd = NULL;
#endif
    }
    return true;
}


bool WiFiManager::dequeueATdataResponse(){
    if(wifiCmd != WIFI_CMD_NONE) return false; // busy
    osEvent evt = _aT2WiFiDataQueue->get(0);
    if(evt.status == osEventMessage){
        data_msg = (wifi_data_msg_t*)evt.value.p;
        setNextCommand(data_msg->wifi_cmd);
        //_wiFi2ATDatamPool->free(data_msg);
    }
    return true;
}


bool WiFiManager::setNextCommand(wifi_cmd_t cmd)
{
    printf("\n [WIFI-MAN] About to set next WiFi manager command to %d\n", cmd);
    if(wifiCmd == WIFI_CMD_NONE){
        wifiCmd = cmd;
        return true; // success
    }
    printf("\n [WIFI-MAN] Busy : current state = %d \n", wifiCmd);
    return false; // wiFiManager busy
}

const char * WiFiManager::sec2str(nsapi_security_t sec)
{
    switch (sec) {
        case NSAPI_SECURITY_NONE:
            return "None";
        case NSAPI_SECURITY_WEP:
            return "WEP";
        case NSAPI_SECURITY_WPA:
            return "WPA";
        case NSAPI_SECURITY_WPA2:
            return "WPA2";
        case NSAPI_SECURITY_WPA_WPA2:
            return "WPA/WPA2";
        case NSAPI_SECURITY_UNKNOWN:
        default:
            return "Unknown";
    }
}


nsapi_size_or_error_t WiFiManager::scanNetworks()
{
    nsapi_error_t error;
    printf("\n [WIFI-MAN] About to start scan for WiFi networks\n");
    lastScanCount = network->scan(NULL, 0);
    printf("\n [WIFI-MAN] Scan for WiFi networks completed - \n");
    return lastScanCount;
}


//nsapi_size_or_error_t WiFiManager::getAvailableAPs(WiFiAccessPoint * res, 
//                                                   nsapi_size_t ncount)
nsapi_size_or_error_t WiFiManager::getAvailableAPs(nsapi_size_t ncount)
{
    WiFiAccessPoint       *ap;
    nsapi_size_or_error_t count;
    count = ncount;
    //count = wiFiManager->scanNetworks();
    if (count <= 0) {
        //_smutex.lock();
        printf("[WIFI-MAN] scan() failed with return value: %d\n", count);
        //_smutex.unlock();
        return 0;
    }
    /* Limit number of network arbitrary to 15 */
    count = count < 15 ? count : 15;
    ap = new WiFiAccessPoint[count];
    count = network->scan(ap, count);
    if (count <= 0) {
        printf("[WIFI-MAN] scan() failed with return value: %d\n", count);
        return 0;
    }

    for (int i = 0; i < count; i++) {
        printf("[WIFI-MAN]: %s secured: %s BSSID: %hhX:%hhX:%hhX:%hhx:%hhx:%hhx RSSI: %hhd Ch: %hhd\n", ap[i].get_ssid(),
               sec2str(ap[i].get_security()), ap[i].get_bssid()[0], ap[i].get_bssid()[1], ap[i].get_bssid()[2],
               ap[i].get_bssid()[3], ap[i].get_bssid()[4], ap[i].get_bssid()[5], ap[i].get_rssi(), ap[i].get_channel());
    }
    printf("[WIFI-MAN]  %d networks available.\n", count);

    delete[] ap;
    return count;
}


void WiFiManager::set_WIFI_CONFIG()
{
    wifi_config_t *wifi_cfg= (wifi_config_t *) data_msg->buffer;
    if(wifi_cfg->ssid[0] != NULL)set_WIFI_SSID(wifi_cfg->ssid);
    if(wifi_cfg->pass[0] != NULL)set_WIFI_PASSWORD(wifi_cfg->pass);
    if(wifi_cfg->security != NSAPI_SECURITY_UNKNOWN)set_WIFI_SECURITY(wifi_cfg->security);
    free_DataMsg();
}

void WiFiManager::set_WIFI_SSID(char * wifi_ssid)
{
    strcpy(wifi_config.ssid, wifi_ssid);
    printf("[WIFI-MAN]  wifi_ssid set to %s\n", wifi_config.ssid);
    https_connection_active = false; // reset whenever any of the security credentials change
    delete socket;
}


void WiFiManager::set_WIFI_PASSWORD(char * wifi_pass)
{
    strcpy(wifi_config.pass, wifi_pass);
    printf("[WIFI-MAN]  wifi_pass set to %s\n", "****************");
    https_connection_active = false; // reset whenever any of the security credentials change
    delete socket;
}


void WiFiManager::set_WIFI_SECURITY(nsapi_security_t wifi_security)
{
    wifi_config.security = wifi_security;
    printf("[WIFI-MAN]  wifi_security set to %s\n", sec2str(wifi_config.security));
    https_connection_active = false; // reset whenever any of the security credentials change
    delete socket;
}



void WiFiManager::set_internet_config()
{
    internet_config_t *internet_cfg  = (internet_config_t *) data_msg->buffer;
    internet_config.peer_id          = internet_cfg->peer_id;
    strncpy(internet_config.url,internet_cfg->url, strlen(internet_cfg->url)+1);
    internet_config.connectionScheme = internet_cfg->connectionScheme;
    free_DataMsg();
    printf("[WIFI MAN] Internet configuration setup completed\n"); 
    printf("peer_id = %1d, url = %s, connScheme = %1d\n", internet_config.peer_id, 
                                                      internet_config.url, 
                                                      internet_config.connectionScheme);
    https_connection_active = false; // reset whenever any of the security credentials change
    delete socket;
}



void WiFiManager::getNetworkStatus(){
    
    responseString = (char *) malloc(MAX_RESPONSE_STRING_LEN);
    net_stat_id_t status_id;
    char * nextStrPtr = responseString;
    for(int i=0; i< NumNetworkStatus;i++){
        status_id = netStatusIds[i]; // get current status id
        switch(status_id){
            case IF_HW_ADDRESS:
                sprintf(nextStrPtr, "\r\n%s%d,%d,%s\r\n", NETWORK_STATUS, 
                                                              WIFI_CHANNEL,
                                                              status_id,
                                                              network->get_mac_address());
                break;
            case NETWORK_IF_STATUS:
                sprintf(nextStrPtr, "\r\n%s%d,%d, %d\r\n", NETWORK_STATUS, 
                                                           WIFI_CHANNEL,
                                                           status_id,
                                                           (uint8_t)is_connected);
                break;
            case INTERFACE_TYPE:
                sprintf(nextStrPtr, "\r\n%s%d,%d,%d\r\n", NETWORK_STATUS, 
                                                              WIFI_CHANNEL,
                                                              status_id,
                                                              WIFI_STATION);
                break;
            case IPv4_ADDRESS:
                sprintf(nextStrPtr, "\r\n%s%d,%d,%s\r\n", NETWORK_STATUS, 
                                                              WIFI_CHANNEL,
                                                              status_id,
                                                              network->get_ip_address());
                break;
            case SUBNET_MASK:
                sprintf(nextStrPtr, "\r\n%s%d,%d,%s\r\n", NETWORK_STATUS, 
                                                              WIFI_CHANNEL,
                                                              status_id,
                                                              network->get_netmask());
                break;
            case GATEWAY_ADDRESS:
                sprintf(nextStrPtr, "\r\n%s%d,%d,%s\r\n", NETWORK_STATUS, 
                                                              WIFI_CHANNEL,
                                                              status_id,
                                                              network->get_gateway());
                break;
            case PRIMARY_DNS_SERVER:
                sprintf(nextStrPtr, "\r\n%s%d,%d,%s\r\n", NETWORK_STATUS, 
                                                              WIFI_CHANNEL,
                                                              status_id,
                                                              DEFAULT_DNS_ADDRESS);
                break;
            case SECONDARY_DNS_SERVER:
                sprintf(nextStrPtr, "\r\n%s%d,%d,%s\r\n", NETWORK_STATUS, 
                                                              WIFI_CHANNEL,
                                                              status_id,
                                                              DEFAULT_DNS_ADDRESS);
                break;
            case IPv6_ADDRESS:
                sprintf(nextStrPtr, "\r\n%s%d,%d,::\r\n", NETWORK_STATUS, 
                                                              WIFI_CHANNEL,
                                                              status_id);
                break;
            default:
                sprintf(nextStrPtr, "\r\n%s,::\r\n", NETWORK_STATUS);
                break;
        }
        nextStrPtr += strlen(nextStrPtr) +1; // progress to end of current string
    }
}



void WiFiManager::getWiFiStatus(){
    
    responseString = (char *) malloc(MAX_RESPONSE_STRING_LEN);
    wifi_stat_id_t status_id;
    char * nextStrPtr = responseString;
    for(int i=0; i< NumWiFiStatus;i++){
        status_id = wifiStatusIds[i]; // get current status id
        switch(status_id){
            case WIFI_SSID:
                sprintf(nextStrPtr, "\r\n%s%d,%s\r\n", WIFI_NETWORK_STATUS, 
                                                       status_id,
                                                       wifi_config.ssid);
                break;
            case WIFI_BSSID:
                sprintf(nextStrPtr, "\r\n%s%d,%s\r\n", WIFI_NETWORK_STATUS, 
                                                       status_id,
                                                       network->get_mac_address());
                break;
            case WIFI__CURRENT_CHANNEL:
                sprintf(nextStrPtr, "\r\n%s%d,%d\r\n", WIFI_NETWORK_STATUS, 
                                                       status_id,
                                                       DEFAULT_WIFI_CHANNEL);
                break;
            case WIFI_STA_STATUS:
                sprintf(nextStrPtr, "\r\n%s%d,%d\r\n", WIFI_NETWORK_STATUS, 
                                                       status_id,
                                                       (uint8_t)is_connected);
                break;
            case WIFI_RSSI:
                sprintf(nextStrPtr, "\r\n%s%d,%d\r\n", WIFI_NETWORK_STATUS, 
                                                       status_id,
                                                       network->get_rssi());
                break;
            default:
                sprintf(nextStrPtr, "\r\n%s,::\r\n", WIFI_NETWORK_STATUS);
                break;
        }
        nextStrPtr += strlen(nextStrPtr) +1; // progress to end of current string
    }
}


void WiFiManager::free_DataMsg()
{
    // free memory after processing
    _aT2WiFiDatamPool->free(data_msg);
    data_msg = NULL;
}



void WiFiManager::status_callback(nsapi_event_t status, intptr_t param)
{
    printf("[WIFI-MAN] about call callback... \r\n");
    _event_queue.call_in(50, this, &WiFiManager::status_callback_event, status, param);
    //status_callback_event(status, param);
}
void WiFiManager::status_callback_event(nsapi_event_t status, intptr_t param)
{
    //if (status == NSAPI_EVENT_CONNECTION_STATUS_CHANGE) {
    //}
    switch(param) {
        case NSAPI_STATUS_LOCAL_UP:
            printf("[WIFI-MAN] Local IP address set!\r\n");
            printf("[WIFI-MAN] IP address: %s\n", network->get_ip_address());
            break;
        case NSAPI_STATUS_GLOBAL_UP:
            printf("Global IP address set!\r\n");
            printf("[WIFI-MAN] IP address: %s\n", network->get_ip_address());
            printf("[WIFI-MAN] Connected to the network %s\n", wifi_config.ssid);
            responseString = (char *) malloc(MAX_RESPONSE_STRING_LEN); 
            sprintf(responseString, "\r\n%s%d,%s,%d\r\n", WIFI_LINK_ENABLED, 
                                                          WIFI_CHANNEL,
                                                          network->get_mac_address(),
                                                          DEFAULT_WIFI_CHANNEL);
            
            is_connected = true;
            break;
        case NSAPI_STATUS_DISCONNECTED:
            printf("No connection to network!\r\n");
            printf("\n [WIFI-MAN] No connection to network!\n");
            is_connected = false;
            //queueATresponse(AT_DISCONNECT_RESP);
            // attempt reconnection if always connected scheme is set
            if(internet_config.connectionScheme == ALWAYS_CONNECTED)
            {
                nsapi_error_t error;
                error = connect();
                queueATresponse(WIFI_RECONNECT_INFO);
            }
            break;
        case NSAPI_STATUS_CONNECTING:
            printf("Connecting to network!\r\n");
            break;
        default:
            printf("Not supported");
            break;
    }
}



   //     NSAPI_STATUS_LOCAL_UP           = 0,        /*!< local IP address set */
   //     NSAPI_STATUS_GLOBAL_UP          = 1,        /*!< global IP address set */
   //     NSAPI_STATUS_DISCONNECTED       = 2,        /*!< no connection to network */
   //     NSAPI_STATUS_CONNECTING         = 3,        /*!< connecting to network */
   //     NSAPI_STATUS_ERROR_UNSUPPORTED  = NSAPI_ERROR_UNSUPPORTED   
   
nsapi_error_t WiFiManager::connect()
{
    nsapi_error_t error;
    printf("\n [WIFI-MAN] About to connect to WiFi network\n");
    network->attach(callback(this, &WiFiManager::status_callback));
    error = network->set_blocking(false);
    if(error)
    {
        printf("\n [WIFI-MAN] Could not set non-blocking mode for Wifi -- aborting!! - \n");
        return error;
    }
    printf("[WIFI-MAN] Connecting to network ssid = %s passwd = %s  security = %s \r\n", 
                                                    wifi_config.ssid, 
                                                    "****************", 
                                                    sec2str(wifi_config.security));
    error = network->connect(wifi_config.ssid,
                     wifi_config.pass,
                     wifi_config.security);
    printf("[WIFI-MAN] network->connect called. error = %d\r\n", error);
    return error;
}


nsapi_error_t WiFiManager::disconnect()
{
    nsapi_error_t error;
    error = network->disconnect();
    return error;
}

#define MIX_HDR_AND_BODY
void WiFiManager::sendResponseDownloadData(at_cmd_resp_t at_cmd, const uint8_t * buf, int bufLen)
{    

    //printf("before call to new at_data_msg_t \n");
    //print_memory_info();
    at_data_resp = new at_data_msg_t;
    //printf("after call to new at_data_msg_t \n");
    //print_memory_info();
    at_data_resp->at_resp = at_cmd;
    size_t bufSize = sizeof(at_data_resp->buffer);
    int pos = 0;
    at_data_resp->dataLen = 0;
    bool queueResult = true;
    int hdrLen = 0;
    int wait_count = 0;
    do {
        if(!queueResult){
            wait_count++;
            printf("[WIFI-MAN] ATCMD Queue full waiting %d ms so far...\n", wait_count*10);
            wait_ms(10);
        }
        else {            
            if(http_response_hdr_sent == false && chunkNum==1){ // only do this for first chunk
                bool status = copyResponseHdr2Queue();  
                if(status == true){
                    printf("[WIFI-MAN] Http Response header copied to response buffer [bytes = %d] \r\n",at_data_resp->dataLen);
                    hdrLen =  at_data_resp->dataLen;
                    http_response_hdr_sent = true;  
                } 
                else {     
                    printf("[WIFI-MAN] Http Response header copy failed\r\n");
                }
            }
            int cpyLen = (bufLen - pos) > bufSize? bufSize : (bufLen - pos) ;
            printf("[WIFI-MAN] Http Response body [bytes = %d] \r\n",cpyLen);
            at_data_resp->dataLen += cpyLen;
            memcpy(&at_data_resp->buffer[hdrLen], &buf[pos], cpyLen);
            printf("[WIFI-MAN] Http Response header and body copied to response buffer [bytes = %d] \r\n",at_data_resp->dataLen);
        }
        queueResult = queueWiFiDataResponse(*at_data_resp);
        if(queueResult){ 
            pos+= at_data_resp->dataLen;
            at_data_resp->dataLen = 0;
            hdrLen = 0; 
        }
    }while(queueResult ==  false || pos < bufLen);
    printf("[WIFI-MAN] response data queued - deleting data memory\r\n");
    delete at_data_resp;
}

bool WiFiManager::copyResponseHdr2Queue()
{
    int numChars = 0;
    // create message pointer for response header generation
    char * msgPtr = (char *)at_data_resp->buffer;
    // do status line
    printf("before getting HTTP status line\n");
    numChars = sprintf(msgPtr, "HTTP/1.1 %d %s\r\n", http_response->get_status_code(), 
                                                     http_response->get_status_message().c_str());
    msgPtr += numChars;
    printf("before getting HTTP headers length\n");

    int hdrsLen = http_response->get_headers_length();
    printf("after getting HTTP headers length = %d\n", hdrsLen);
    //print_memory_info();
    if(hdrsLen <= 1)
    {
        printf("copy failed: Header Line = [%s]", msgPtr);
        return false;
    }
    char * hdrField;
    char * hdrValue;
    for (size_t ix = 0; ix < hdrsLen; ix++) {
        int sLen = http_response->get_headers_fields()[ix]->size()+1;
        hdrField = new char [sLen];
        std::strcpy (hdrField, http_response->get_headers_fields()[ix]->c_str());
        hdrValue = new char [sLen];
        std::strcpy (hdrValue, http_response->get_headers_values()[ix]->c_str());
        numChars = sprintf(msgPtr, "%s: %s\r\n", hdrField, hdrValue);
        delete hdrField;
        delete hdrValue;
        msgPtr += numChars;
    }
    numChars = sprintf(msgPtr, "\r\n");
    msgPtr += numChars;
    // print out generated header
    printf("[WiFi MAN] generated response header:\n");
    printf("%s\r\n", (char *)at_data_resp->buffer);
    // calculate header length 
    at_data_resp->dataLen = (msgPtr - (char *)at_data_resp->buffer);
    return true;
}

void WiFiManager::return_response(HttpResponse* res) {
    
    at_data_resp = new at_data_msg_t;
    int numChars = 0;
    // create message pointer for response header generation
    char * msgPtr = (char *)at_data_resp->buffer;
    // do status line
    numChars = sprintf(msgPtr, "HTTP/1.1 %d %s\r\n", res->get_status_code(), res->get_status_message().c_str());
    msgPtr += numChars;
    for (size_t ix = 0; ix < res->get_headers_length(); ix++) {
        numChars = sprintf(msgPtr, "%s: %s\r\n", 
                           res->get_headers_fields()[ix]->c_str(), 
                           res->get_headers_values()[ix]->c_str());
        msgPtr += numChars;
    }
    numChars = sprintf(msgPtr, "\r\n\r\n");
    msgPtr += numChars;
    // print out generated header
    printf("[WiFi MAN] generated response header:\n");
    printf("%s\r\n", (char *)at_data_resp->buffer);
    // calculate header length 
    at_data_resp->dataLen = (msgPtr - (char *)at_data_resp->buffer);
    
    // package and send on wifi data queue    
    at_data_resp->at_resp = AT_HTTPS_RESP;
    bool queueResult = true;
    int wait_count = 0;
    do
    {
        if(!queueResult){
            wait_count++;
            printf("ATCMD Queue full waiting %d ms so far...\n", wait_count*10);
            wait_ms(10);
        }
        queueResult = queueWiFiDataResponse(*at_data_resp);
    }while(queueResult == false);
    delete at_data_resp;
}


void WiFiManager::printBufferInHex(uint8_t *buf, int pLen)
{
    for(int i =0;i<pLen;i++){
        if(i%8==0) printf("\n[%3d]",i);
        printf("%02x ", buf[i]);
    }
    printf("\n");
}

//#define TRY_PRINTF

void WiFiManager::body_callback(const char *at, uint32_t length) {    
    printf("\n Chunked response: Chunk %d : Total Bytes = %d\n", chunkNum , length);
    chunkNum++;
    sendResponseDownloadData(AT_HTTPS_RESP_DOWNLOAD, (uint8_t *)at, length);
}


bool WiFiManager::createTLSconnection(const char * hostName)
{
#ifdef ENABLE_MBED_TRACE
    mbed_trace_init();
#endif
    socket = new TLSSocket();

    nsapi_error_t r;
    // make sure to check the return values for the calls below (should return NSAPI_ERROR_OK)
    r = socket->open(network);
    if(r != NSAPI_ERROR_OK)
    { 
        printf("TLS open failed!!\n");
        return false;
    }
    printf("TLS open passed!!\n");
    r = socket->set_root_ca_cert(SSL_CA_PEM);
    if(r != NSAPI_ERROR_OK)
    { 
        socket->close();
        printf("TLS set_root_ca_cert failed!!\n");
        return false;
    }
    printf("TLS set_root_ca_cert passed!!\n");
    r = socket->connect(hostName, 443);
    if(r != NSAPI_ERROR_OK)
    { 
        char errstr[100];
        mbedtls_strerror(r, errstr, 100);
        printf("TLS connect failed for hostname %s -- ERROR = %s !!\n", hostName, errstr);
        socket->close();
        return false;
    }
    printf("TLS connection successful for https site :  %s\n", hostName);
    return true;
}
#define TESTING_HTTPS

//#define DONT_USE_TLS_SOCKET
bool WiFiManager::createHttpsRequest()
{
    // reset chunk #;
    chunkNum = 0;
#ifdef MIX_HDR_AND_BODY
    http_response_hdr_sent = false;
#else
    http_response_hdr_sent = true;
#endif    
    printf("\n[WIFI MAN] Http Request received:\n");
    http_req_cfg = (http_request_t *) data_msg->buffer;
    printf("\n[WIFI MAN] uri = %s\n", http_req_cfg->request_URI);
    printf("\n[WIFI MAN] internet cfg url = %s\n", internet_config.url);
    char full_url[100];
    char host[60] ;
    strncpy(full_url,internet_config.url, strlen(internet_config.url)+1);
    strncpy(host,http_req_cfg->hostName, strlen(http_req_cfg->hostName)+1);
    strncat(full_url, http_req_cfg->request_URI, strlen(http_req_cfg->request_URI)+1);
    printf("\n[WIFI MAN] server url+uri = %s\n", full_url);
    printf("\n[WIFI MAN] Host = %s\n", http_req_cfg->hostName);
    printf("\n[WIFI MAN] Accept = %s\n", http_req_cfg->AcceptVal);
    printf("\n[WIFI MAN] Content-Type = %s\n", http_req_cfg->contentType);
    printf("\n[WIFI MAN] contentLenstr = %s\n", http_req_cfg->contentLen);
                                           
    int bodyLen;
    sscanf(http_req_cfg->contentLen, "%d", &bodyLen);
    printf("contenLenstr = %s bodyLen = %d\n", http_req_cfg->contentLen, bodyLen);
    
    if(bodyLen > 10){
        printf("\n [WIFI MAN] Message Body:\n");
        printBufferInHex(http_req_cfg->body, bodyLen);
    }
    if(strstr(internet_config.url, "http:")!=NULL) // http request
    {
        http_request = new HttpRequest(network,  
                                       http_req_cfg->method, 
                                       full_url,
                                       callback(this, &WiFiManager::body_callback));
        setHttpHeader("Host", http_req_cfg->hostName);
        setHttpHeader("Accept", http_req_cfg->AcceptVal);
        printf("http_req_cfg->method = %d\n", http_req_cfg->method);
        if(http_req_cfg->method == HTTP_GET){
            printf("HTTP_GET -- ignoring body\n");
            //setHttpHeader("Content-Type", http_req_cfg->contentType);
            //setHttpHeader("Content-Length", http_req_cfg->contentLen);
            http_response = http_request->send(NULL, 0);
        }
        else{
            setHttpHeader("Content-Type", http_req_cfg->contentType);
            setHttpHeader("Content-Length", http_req_cfg->contentLen);
            http_response = http_request->send(http_req_cfg->body, bodyLen);
        }
        free_DataMsg();
        if (!http_response) {
            char buf[100];
            mbedtls_strerror(http_request->get_error(), buf, 100);
            printf("HttpRequest failed (error code %s)\n", buf);
            //printf("HttpsRequest failed (error code %d)\n", https_request->get_error());
            delete http_request; // free the memory
            return false;
        }
        delete http_request; // free the memory
        printf("\n----- HTTP POST response -----\n");
    }
    else
    {
#ifndef DONT_USE_TLS_SOCKET
        if(https_connection_active == false){
            bool tlsResult;
            tlsResult = createTLSconnection(host);
#ifdef ENABLE_MBED_TRACE
            mbed_trace_free(); // free trace memory
#endif
            if(tlsResult == false){
                 delete socket;
                 printf("TLS Socket connection failed - deleting data msg\r\n");
                 free_DataMsg();
                 printf("data msg deleted \r\n");
                 return false;
            }
            //printf("[create https] TLS connection successful for https site :  %s\n", host);
            //printf("after call to createTLSconnection \n");
            //print_memory_info();
        }        
        // Pass in `socket`, instead of `network` as first argument, and omit the `SSL_CA_PEM` argument
        //HttpsRequest* get_req = new HttpsRequest(socket, HTTP_GET, "https://httpbin.org/status/418");
        //_wmutex.lock();
        https_request = new HttpsRequest(socket, 
                                         http_req_cfg->method, 
                                         full_url,
                                         callback(this, &WiFiManager::body_callback));
#else
        https_request = new HttpsRequest(network, 
                                         SSL_CA_PEM,
                                         http_req_cfg->method, 
                                         full_url,
                                         callback(this, &WiFiManager::body_callback));
#endif        
#ifdef TESTING_HTTPS
        printf("http_req_cfg->method = %d\n", http_req_cfg->method);
        if(http_req_cfg->method == HTTP_GET){
            printf("HTTP_GET -- ignoring body\n");
            setHttpsHeader("Host", http_req_cfg->hostName);
            setHttpsHeader("Accept", http_req_cfg->AcceptVal);
            //setHttpHeader("Content-Type", http_req_cfg->contentType);
            //setHttpHeader("Content-Length", http_req_cfg->contentLen);
            http_response = https_request->send(NULL, 0);
        }
        else{
            printf("about to setup https headers\r\n");
            setHttpsHeader("Host", http_req_cfg->hostName);
            setHttpsHeader("Accept", http_req_cfg->AcceptVal);
            setHttpsHeader("Content-Type", http_req_cfg->contentType);
            setHttpsHeader("Content-Length", http_req_cfg->contentLen);
            printf("https headers setup - about to send request\r\n");
            mbedtls_ssl_context* tlsContext ;
            tlsContext = socket->get_ssl_context();
            mbedtls_ssl_config * tlsConfig;
            tlsConfig = socket->get_ssl_config();
            if(tlsContext != NULL)
            {
                printf("current TLS tlsContext is not null [%d] \r\n", tlsContext->state);
            }
            else
            {
                printf("TLS Context is NULL \r\n");
            }
            //_wmutex.lock();
            http_response = https_request->send(http_req_cfg->body, bodyLen);
            //_wmutex.unlock();
        }
#else
        setHttpsHeader("Host", http_req_cfg->hostName);
        setHttpsHeader("Accept", http_req_cfg->AcceptVal);
        setHttpsHeader("Content-Type", http_req_cfg->contentType);
        setHttpsHeader("Content-Length", http_req_cfg->contentLen);
        http_response = https_request->send(http_req_cfg->body, bodyLen);
#endif        
        
        //_wmutex.unlock();
        //free_DataMsg();
        if (!http_response) {
            char buf[100];
            mbedtls_strerror(https_request->get_error(), buf, 100);
            printf("HttpsRequest failed (error code %s)\n", buf);
            delete https_request; // free the memory
            https_request = NULL;
            https_connection_active = false; // reset true whenever connection fails
            socket->close();
            delete socket;
            socket = NULL;
            free_DataMsg();
            return false;
        }
        https_connection_active = true; // set true whenever connection succeeds
        printf("\n----- HTTPS POST response -----\r\n");
    }
    if(http_response != NULL){

#ifndef MIX_HDR_AND_BODY
        return_response(http_response);
#else
    if(http_response_hdr_sent == false){ // if it failed to add to first chunk send separately
        return_response(http_response);
    }
#endif    
        //delete http_response; // free the response memory
        //http_response = NULL;
        //printf("deleted http_response\r\n");
    }
    free_DataMsg();
    delete https_request; // free the request & response memory
    printf("deleted https_request\r\n");
    https_request = NULL;
    return true;
}

void WiFiManager::createHttpRequest(http_method method,
                                    const char* url,
                                    Callback<void(const char *at, uint32_t length)> body_callback)
{
    http_request = new HttpRequest(network, 
                                   method, url, body_callback);;
}

void WiFiManager::setHttpHeader(string key, string value)
{
    http_request->set_header(key, value);
}

void WiFiManager::setHttpsHeader(string key, string value)
{
    https_request->set_header(key, value);
}

void WiFiManager::sendHttpsRequest(const char * body, int bodyLen)
{
}

void WiFiManager::sendHttpRequest(const char * body, int bodyLen)
{
}

