#include "ATCmdManager.h"
#include "common_config.h"
#include "common_types.h"
#include "http_request.h"
#include "mbed_memory_status.h"
//#include "mbed_memory_status.h"

ATCmdManager::ATCmdManager(PinName tx, PinName rx, SMDevicePeripheral *blePeripheral, 
                           events::EventQueue &event_queue, WiFiManager *wifi, 
                           MemoryPool<wifi_cmd_message_t, 16> *aT2WiFimPool, 
                           Queue<wifi_cmd_message_t, 16>      *aT2WiFiCmdQueue, 
                           MemoryPool<at_resp_message_t, 16>  *wiFi2ATmPool, 
                           Queue<at_resp_message_t, 16>       *wiFi2ATCmdQueue, 
                           MemoryPool<wifi_data_msg_t, PQDSZ>     *aT2WiFiDatamPool, 
                           Queue<wifi_data_msg_t, PQDSZ>          *aT2WiFiDataQueue, 
                           MemoryPool<at_data_msg_t, PQDSZ>       *wiFi2ATDatamPool, 
                           Queue<at_data_msg_t, PQDSZ>            *wiFi2ATDataQueue, 
                           bool debug)
    :
    _serial(tx, rx, DEFAULT_BAUD_RATE),
    blePeripheral(blePeripheral),
    _event_queue(event_queue),
    wiFiManager(wiFiManager),
    _aT2WiFimPool(aT2WiFimPool),
    _aT2WiFiCmdQueue(aT2WiFiCmdQueue),
    
    _wiFi2ATmPool(wiFi2ATmPool),
    _wiFi2ATCmdQueue(wiFi2ATCmdQueue),
    
    _aT2WiFiDatamPool(aT2WiFiDatamPool),
    _aT2WiFiDataQueue(aT2WiFiDataQueue),
    
    _wiFi2ATDatamPool(wiFi2ATDatamPool),
    _wiFi2ATDataQueue(wiFi2ATDataQueue),
    
    _parser(&_serial)
    

{ 
// constructor
    _serial.set_baud(DEFAULT_BAUD_RATE);
    _parser.debug_on(debug);
    _parser.set_delimiter("\r\n");
    _parser.send("+STARTUP");
    _parser.oob("AT\n", callback(this, &ATCmdManager::_oob_ok_hdlr));
    _parser.oob("ATE0", callback(this, &ATCmdManager::_oob_echo_off));
    _parser.oob("ATE1", callback(this, &ATCmdManager::_oob_echo_on));
    _parser.oob("AT+UMRS", callback(this, &ATCmdManager::_oob_uart_setup));
    
    _parser.oob("ATO", callback(this, &ATCmdManager::_oob_data_mode));
    _parser.oob("AT+UMLA", callback(this, &ATCmdManager::_oob_get_mac_addr));
    _parser.oob("AT+UBTLE?", callback(this, &ATCmdManager::_oob_get_ble_role));
    _parser.oob("AT+UBTLE=2", callback(this, &ATCmdManager::_oob_ena_ble_peri));
    _parser.oob("AT+CPWROFF", callback(this, &ATCmdManager::_oob_reboot));
    _parser.oob("AT+CGMR", callback(this, &ATCmdManager::_oob_get_fw_ver));
    _parser.oob("AT+UWSCAN", callback(this, &ATCmdManager::_oob_scanWiFiNetworks));
    _parser.oob("AT+UWSCA=", callback(this, &ATCmdManager::_oob_WiFiStationConfigAction));
    //_parser.oob("AT+UWSCD=", callback(this, &ATCmdManager::_oob_disconnectWiFiNetwork));
    _parser.oob("AT+UDDRP", callback(this, &ATCmdManager::_oob_setupInternetConnection));
    _parser.oob("AT+UWSC=0,2", callback(this, &ATCmdManager::_oob_setWiFiSSID));
    _parser.oob("AT+UWSC=0,8", callback(this, &ATCmdManager::_oob_setWiFiPWD));
    _parser.oob("AT+UWSC=0,5", callback(this, &ATCmdManager::_oob_setWiFiSecurity));
    _parser.oob("AT+UNSTAT=", callback(this, &ATCmdManager::_oob_getNetworkStatus));
    _parser.oob("AT+UWSSTAT", callback(this, &ATCmdManager::_oob_WiFiNetworkStatus));
    
    //_parser.oob("AT+UWSC=0,5", callback(this, &ATCmdManager::_oob_sendHttpMessage));
 
    printf("\n --- ATCmdManager constructor completed ---\n");
    at_resp = AT_RESP_NONE;
    dataMode = AT_CMD_DATA_MODE;
    debug_flag = 0;
    _event_queue.call_in(10, &print_heap_and_isr_stack_info);
    //print_heap_and_isr_stack_info();
    _event_queue.call_every(3600000,&print_memory_info);
}


void ATCmdManager::runMain(){
    printf("\r\n [ATCMD MAN]  Thread Id = %X\r\n", (uint32_t)ThisThread::get_id());
    while(true){
        _process_oob(UBLOX_ODIN_W2_RECV_TIMEOUT, true);
        wait_ms(MAIN_LOOP_WAIT_TIME_MS); // allow BTLE/WiFi some time 
        processResponses();
    }
    
}

void ATCmdManager::processResponses(){
        dequeueATresponse();
        dequeueWiFidataResponse();
        if(debug_flag>0){
            //debug_flag--;
            printf(" back to ProcessResponses in Main loop... at_resp = %d\r\n", at_resp);
        }
        char * respStr;
        //if(at_resp > AT_COMMAND_FAILED)
        //return;
        switch(at_resp){
            case AT_RESP_NONE:
                if(debug_flag>0){
                    debug_flag--;
                    printf(" In IDLE state \r\n");
                }
                // IDLE response state
                break;
            case AT_SCAN_RESP:
                // AT_SCAN_RESP response state
                printf("\n [ATCMD MAN] WIFI SCAN RESPONSE RECEIVED!!\r\n");
                //_smutex.lock();
                //sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
                //_smutex.unlock();
                respStr = (char *) resp_data->buffer;
                sendAtConfirmationFreeMpool(respStr);
                at_resp = AT_RESP_NONE;
                break;
            case AT_DETAILED_SCAN_RESP:
                // AT_DETAILED_SCAN_RESP response state
                printf("\n [ATCMD MAN] WIFI DETAILED SCAN RESPONSE RECEIVED!!\r\n");
                //_smutex.lock();
                //sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
                //_smutex.unlock();
                respStr = (char *) resp_data->buffer;
                sendAtConfirmationFreeMpool(respStr);
                at_resp = AT_RESP_NONE;
                break;
            case AT_CONNECT_RESP:  
                // AT_CONNECT_RESP response state 
                printf("\n [ATCMD MAN] WIFI CONNECT RESPONSE RECEIVED!!\r\n");
                //_smutex.lock();
                //sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
                respStr = (char *) resp_data->buffer;
                sendAtConfirmationFreeMpool(respStr);
                //_smutex.unlock();
                at_resp = AT_RESP_NONE;
                break;
            case AT_DISCONNECT_RESP:  
                // AT_DISCONNECT_RESP response state 
                //////_smutex.lock();
                printf("\n [ATCMD MAN] WIFI DISCONNECT RESPONSE RECEIVED!!\r\n");
                //_smutex.lock();
                //sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
                //_smutex.unlock();
                respStr = (char *) resp_data->buffer;
                sendAtConfirmationFreeMpool(respStr);
                at_resp = AT_RESP_NONE;
                break;
            case AT_CONFIG_RESP:  
                // AT_CONFIG_RESP response state 
                //////_smutex.lock();
                printf("\n [ATCMD MAN] WIFI CONFIG RESPONSE RECEIVED!!\r\n");
                //_smutex.lock();
                sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
                //_smutex.unlock();
                at_resp = AT_RESP_NONE;
                break;
            case AT_NETWORK_STATUS_RESP:  
                // AT_CONFIG_RESP response state 
                //////_smutex.lock();
                printf("\n [ATCMD MAN] NETWORK STATUS RESPONSE RECEIVED!!\r\n");
                respStr = (char *) resp_data->buffer;
                sendAtConfirmationFreeMpool(respStr);
                at_resp = AT_RESP_NONE;
                break;
            case AT_WIFI_STATUS_RESP:  
                // AT_CONFIG_RESP response state 
                //////_smutex.lock();
                printf("\n [ATCMD MAN] WIFI RESPONSE RECEIVED!!\r\n");
                respStr = (char *) resp_data->buffer;
                sendAtConfirmationFreeMpool(respStr);
                at_resp = AT_RESP_NONE;
                break;
            case AT_INTERNET_CONFIG_RESP:  
                // AT_CONFIG_RESP response state 
                printf("\n [ATCMD MAN] WIFI INTERNET_CONFIG RESPONSE RECEIVED!!\r\n");
                //_smutex.lock();
                sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
                //_smutex.unlock();
                //respStr = (char *) resp_data->buffer;
                //sendAtConfirmationFreeMpool(respStr);
                at_resp = AT_RESP_NONE;
                break;
            case AT_HTTPS_RESP:  
                // AT_HTTP_RESP response state 
                printf("\n [ATCMD MAN] WIFI HTTPS RESPONSE RECEIVED!!\r\n");
                return_response();
                //_smutex.lock();
                sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
                //_smutex.unlock();
                at_resp = AT_RESP_NONE;
                break;
            case AT_HTTPS_RESP_DOWNLOAD:  
                // AT_HTTPS_RESP_DOWNLOAD response state 
                printf("\n [ATCMD MAN] WIFI HTTPS DOWNLOAD RESPONSE RECEIVED!!\r\n");
                return_response(true); // set download paramter to true
                //_smutex.lock();
                sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
                //_smutex.unlock();
                at_resp = AT_RESP_NONE;
                break;
            case AT_HTTP_RESP:  
                // AT_HTTP_RESP response state 
                printf("\n [ATCMD MAN] WIFI HTTP RESPONSE RECEIVED!!\r\n");
                return_response(); 
                //_smutex.lock();
                sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
                //_smutex.unlock();
                at_resp = AT_RESP_NONE;
                break;
            case AT_HTTP_RESP_DOWNLOAD:  
                // AT_HTTP_RESP response state 
                printf("\n [ATCMD MAN] WIFI HTTP RESPONSE RECEIVED!!\r\n");
                return_response(true); // set download paramter to true
                //_smutex.lock();
                //sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
                //_smutex.unlock();
                at_resp = AT_RESP_NONE;
                break;
            case AT_COMMAND_FAILED:  
            {
                // AT_HTTP_RESP response state 
                printf("\n [ATCMD MAN] WIFI COMMAND FAILED!!\r\n");
                //_smutex.lock();
                //_parser.send("\r\nNACK\r\n");
                //_smutex.unlock();
                respStr = (char *) resp_data->buffer;
                sendAtConfirmationFreeMpool(respStr);
                at_resp = AT_RESP_NONE;
                break;
            }
            //default:
                // UNKNOWN response state
                //_smutex.lock();
                //printf("\n [ATCMD MAN] UNKNOWN RESPONSE RECEIVED!!\r\n");
                //sendAtConfirmationFreeMpool(ERROR_RESP);   //_parser.send(OK_RESP);
                //_smutex.unlock();
                //at_resp = AT_RESP_NONE;
                //break;
        }
}


void ATCmdManager::sendAtConfirmationFreeMpool(const char *buf)
{
    sendAtConfirmation(buf);
    _wiFi2ATDatamPool->free(resp_data);
    resp_data = NULL;
}


void ATCmdManager::printBufferInHex(uint8_t *buf, int pLen)
{
    for(int i =0;i<pLen;i++){
        if(i%8==0) printf("\n[%3d]",i);
        printf("%02x ", buf[i]);
    }
    printf("\n");
}

bool ATCmdManager::validate(edm_header_t edm_header)
{
    if(edm_header.startByte != EDM_START_BYTE)return false; // unexpected start byte found!
    if(edm_header.payloadID != CONNECT_EVENT &&
       edm_header.payloadID != DISCONNECT_EVENT &&
       edm_header.payloadID != DATA_EVENT &&
       edm_header.payloadID != DATA_COMMAND &&
       edm_header.payloadID != AT_REQUEST &&
       edm_header.payloadID != AT_CONFIRMATION &&
       edm_header.payloadID != AT_EVENT 
       )return false; // unexpected payload ID found!
    if(edm_header.payloadLen > MAX_EDM_PAYLOAD_LEN ||
       edm_header.payloadLen < MIN_EDM_PAYLOAD_LEN 
       ) return false; // unexpected length received!
    if(edm_header.channel_id != WIFI_CHANNEL &&
       edm_header.channel_id != BLE_CHANNEL 
       ) return false; // unexpected channel_id received!
    return true;
}

http_method ATCmdManager::str2HttpMethod(const char * methodStr)
{
    http_method _http_method;
    if(strstr(methodStr, "POST")!= NULL){
        _http_method = HTTP_POST;
    }
    else if(strstr(methodStr, "GET")!= NULL){
        _http_method = HTTP_GET;
    }
    else{
        _http_method = HTTP_DELETE; // unsupported method - set to HTTP_DELETE
    }
    return _http_method;
}
bool ATCmdManager::createHttpRequest()
{
    http_request_t *http_req; // = new http_request_t;
    wifi_data_msg_t data_msg;
    http_req = (http_request_t *)data_msg.buffer;
    char s1[32];
    char s2[32];
    char s3[32];
    int len = 0;
    int n;
    char * strPtr = (char *)rx_buf_ptr;
    char * p;
    char * p2 = strstr(strPtr, "\r\n\r\n");
    char * nxtPtr = strPtr;
    char * outPtr;
    char * bodyPtr = p2+4;
    printf("\nstrPtr address= %x",strPtr);
    printf("\np2 address= %x", p2);
    for(int i = 0; i < 5; i++){
        if(i == 0)// firstline scan method uri and http_ver
        {
            n = sscanf(nxtPtr,"%s %s %s", s1, s2, s3);
            if(n!=3) return false; // error in input abort
            printf("\nmethod = %s\nuri = %s\nhttp_ver = %s",s1, s2, s3 );
            //http_req.method = str2HttpMethod(s1.c_str());
            http_req->method = str2HttpMethod(s1);
            http_req->request_URI = (char *) http_req->buffer; // point 1st string to start of buffer
            len = strlen(s2)+1;
            strncpy(http_req->request_URI, s2, len);
            http_req->http_version = http_req->request_URI + len; // point 2nd string to after 1st
            len = strlen(s3)+1;
            strncpy(http_req->http_version, s3, len);
            printf("\nhttp_request 1st line:\n method = %d\nuri = %s\nhttp_ver = %s",http_req->method, 
                                                                      http_req->request_URI, 
                                                                      http_req->http_version );
            printf("\nhttp_request str ptrs\nuri = %X\nhttp_ver = %X",http_req->request_URI, 
                                                                      http_req->http_version );
            outPtr = http_req->http_version + len; // point output buffer ptr to after 2nd string
        }
        else{ // scan header pairs
            //n = sscanf(nxtPtr,"%s %s", &s1[0], &s2[0]);
            n = sscanf(nxtPtr,"%s %s", s1, s2);
            if(n!=2) return false; // error in input abort
            p = strstr(s1,":");
            *p = NULL;
            printf("\nname = %s value = %s",s1, s2);
            if(strstr(s1, "Host")!= NULL){
                http_req->hostName = outPtr;
                len = strlen(s2)+1;                
                strncpy(outPtr, s2, len);
                printf("\nname = %s value = %s",s1, outPtr);
                outPtr += len; // point output buffer ptr to after current string
            }
            else if(strstr(s1, "Accept")!= NULL){
                http_req->AcceptVal = outPtr;
                len = strlen(s2)+1;                
                strncpy(outPtr, s2, len);
                printf("\nname = %s value = %s",s1, outPtr);
                outPtr += len; // point output buffer ptr to after current string
            }
            else if(strstr(s1, "Content-Type")!= NULL){
                http_req->contentType = outPtr;
                len = strlen(s2)+1;                
                strncpy(outPtr, s2, len);
                printf("\nname = %s value = %s",s1, outPtr);
                outPtr += len; // point output buffer ptr to after current string
            }
            else if(strstr(s1, "Content-Length")!= NULL){
                //http_req.contentLen.assign(s2);
                http_req->contentLen = outPtr;
                len = strlen(s2)+1;                
                strncpy(outPtr, s2, len);
                printf("\nname = %s value = %s",s1, outPtr);
                outPtr += len; // point output buffer ptr to after current string
            }
            
            printf("\noutPtr = %X len = %d\n", outPtr, len);
        }
        nxtPtr = strstr(nxtPtr, "\r\n")+2; // goto next line
        if(nxtPtr >= p2) break;
    }
    // print header from http_req_struct
    printf("\nhttp request header: \n %s\n", http_req->buffer);
    int bodyLen = edm_hdr.payloadLen -(p2+7-strPtr);
    printf("\nLen = %d\n", bodyLen);
    http_req->body = (uint8_t *) outPtr;
    memcpy(outPtr, bodyPtr, bodyLen);
    if(bodyLen > 10){
        printf("\n Message Body:\n");
        printBufferInHex(http_req->body, bodyLen);
    }
    outPtr += bodyLen; // move output pointer to end output (header + body)
    // package and send on wifi data queue
    data_msg.wifi_cmd = WIFI_CMD_SEND_HTTPS_REQ;
    printf("\nsizeof(http_req) on population = %d\n", sizeof(*http_req));
    data_msg.dataLen = (uint8_t *)outPtr - http_req->buffer;//sizeof(http_request_t);
    printf("\ndata_msg.dataLen = %d\n", data_msg.dataLen);
    // queue next data request
    queueWiFiDataRequest(data_msg);
    return true;
}
// OOB processing
void ATCmdManager::_process_oob(uint32_t timeout, bool all){
    set_timeout(timeout);
    int cnt = 0;
    int start;
    //channel_id_t chan_id;
    if(dataMode == AT_EXT_DATA_MODE)
    {
        int n;
        //if(cnt++ % 10 == 0)printf("In EDM mode\n");
        uint8_t edm[EDM_HDR_LEN];
        char cmdType[16];
        // Poll for edm packets
        do{
            n = _parser.read((char *)edm, EDM_HDR_LEN-1);
            edm_hdr.startByte = edm[0];
            edm_hdr.payloadLen = edm[1]*256 + edm[2];
            edm_hdr.payloadID = edm[3]*256 + edm[4];
            edm_hdr.channel_id = (channel_id_t) 0;
            start = Kernel::get_ms_count();
            if(n == -1) break; // break if it times out
            int pT;
            if(edm_hdr.payloadID < AT_REQUEST) //
            {
                n += _parser.read((char *)edm, 1);
                edm_hdr.channel_id = (channel_id_t) edm[0];
                pT = 0;
                strcpy(cmdType, "DATA COMMAND");
            }
            else
            {
                pT = 1;
                strcpy(cmdType, "AT REQUEST");
            }
            //printf("%d bytes read! : type = %s\n", n, cmdType2str(edm_hdr.payloadID));
            printf("%d bytes read! : CMD type = %s\n", n, cmdType);
            if(n==(EDM_HDR_LEN-pT))
            printf("Start = %d, payloadID = %d len = %d chan_id = %d\n", edm_hdr.startByte, 
                                                                         edm_hdr.payloadID, 
                                                                         edm_hdr.payloadLen,
                                                                         edm_hdr.channel_id);
            if(n == (EDM_HDR_LEN-pT) && validate(edm_hdr)) // if AT command use process oob to decode
            {
                if(edm_hdr.payloadID == AT_REQUEST)
                {
                    //_parser.process_oob();
                    while (_parser.process_oob() && all) {
                        if(debug_flag>0){
                            printf("finished queuing WIFI CONNECT CMD -- back to process_oob loop \r\n");
                        }
                    }
                    break;
                }
                else
                {
                    int pLen = edm_hdr.payloadLen-2-pT;
                    rx_buf_ptr = (uint8_t *) malloc (pLen); // we already read 2 bytes from payload but expect 1 stop byte
                    rx_buf_ptr[pLen-1] = 0x00; // clear last byte so the readback value is as expected
                    n = _parser.read((char *)rx_buf_ptr, pLen);
                    if(n == -1) break; // timeout!
                    printf("%d bytes read - expected %d!\n", n, pLen);
                    printBufferInHex(rx_buf_ptr, pLen);
                    printf("rx_buf_ptr[pLen-1] = %0x\n",rx_buf_ptr[pLen-1]);
                    if(rx_buf_ptr[pLen-1] != EDM_STOP_BYTE) {
                        _smutex.lock();
                        _parser.send("ERR");
                        _smutex.unlock();
                        free(rx_buf_ptr); // make sure to free buffer
                        rx_buf_ptr = NULL;
                        break; // exit if stop byte not found - possible data corruption!
                    }
                    switch(edm_hdr.payloadID)
                    {
                        case CONNECT_EVENT:
                            printf("Connection Event received!\n");
                            break;
                        case DISCONNECT_EVENT:
                            printf("DISCONNECT_EVENT received!\n");
                            break;
                        case DATA_EVENT:
                            printf("DATA_EVENT received!\n");
                            break;
                        case DATA_COMMAND:
                        {
                            printf("DATA_COMMAND received!\n");
                            sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
                            if(createHttpRequest() == true){
                                //_smutex.lock();
                                //sendAtConfirmation(OK_RESP);   //sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
                                //_smutex.unlock();
                            }
                            else{
                                _smutex.lock();
                                _parser.send("NACK");
                                _smutex.unlock();
                            }
                            free(rx_buf_ptr);
                            rx_buf_ptr = NULL;
                            int stop = Kernel::get_ms_count();
                            printf("\n Time Elapsed = %d\n", stop-start);
                            break;
                        }
                        case AT_REQUEST:
                            printf("AT_REQUEST received!\n");
                            break;
                        case AT_CONFIRMATION:
                            printf("AT_CONFIRMATION received!\n");
                            break;
                        case AT_EVENT:
                            printf("AT_EVENT received!\n");
                            break;
                        default:
                            printf("UNKNOWN MESSAGE received!\n");
                            break;
                    }
                }
                //_parser.process_oob();
            }
            else // incorrect # of bytes received abort!!
            {
                break;
            }
        }while (all);  // continue to process until timeout
    }
    else
    {
        // Poll for inbound packets
        static int msgCnt = 0;
        while (_parser.process_oob() && all) {
                        printf("inbound message found... msgCnt = %d \r\n", ++msgCnt);
                        if(debug_flag>0){
                            printf("finished queuing WIFI CONNECT CMD -- back to process_oob loop \r\n");
                        }
        }
    }        
    set_timeout();
}


// OOB message handlers
void ATCmdManager::_oob_startup_hdlr(){
}

void ATCmdManager::_oob_ok_hdlr(){
    //_smutex.lock();
    sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
    //_smutex.unlock();
}


void ATCmdManager::_oob_bleRole_hdlr(){
}


void ATCmdManager::_oob_wifiMode_err(){
}


void ATCmdManager::_oob_conn_already(){
}


void ATCmdManager::_oob_err(){
}

void ATCmdManager::_oob_get_fw_ver()
{
}


void ATCmdManager::_oob_uart_setup(){
    int uOpts[NUM_UART_OPTIONS];
    //if(_parser.recv("=%d,%d,%d,%d,%d,%d", &uOpts[0], &uOpts[1], &uOpts[2], &uOpts[3], &uOpts[4], &uOpts[5])) {
    if(_parser.scanf("=%d,%d,%d,%d,%d,%d", &uOpts[0], &uOpts[1], &uOpts[2], &uOpts[3], &uOpts[4], &uOpts[5]) >0) {
        printf("\nATCmdParser: Uart Options=%d,%d,%d,%d,%d,%d\n", uOpts[0], uOpts[1], uOpts[2], uOpts[3], uOpts[4], uOpts[5]);
       //AT+UMRS=230400,2,8,1,1,1
        printf("\n Changing Baud Rate to %d\n", uOpts[0]);

        sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
        wait(0.1);
        _serial.set_baud(uOpts[0]);
        wait(0.1);
        printf("\n Baud Rate now %d\n", uOpts[0]);

    } else {
        printf("\nATCmdParser: Retrieving Uart Options failed");
    }
}

void ATCmdManager::set_timeout(uint32_t timeout_ms)
{
    _parser.set_timeout(timeout_ms);
}


void ATCmdManager::_oob_echo_off()
{
    //_smutex.lock();
    printf("\n Received ATEO OOB command!!\n");
    printf("\n turning echo OFF!!\n");
    _parser.debug_on(false);
    sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
    //_smutex.unlock();
}


void ATCmdManager::_oob_echo_on()
{
    //_smutex.lock();
    printf("\n Received ATE1 OOB command!!\n");
    printf("\n turning echo ON!!\n");
    _parser.debug_on(true);
    sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
    //_smutex.unlock();
}


void ATCmdManager::_oob_data_mode(){
    //_smutex.lock();
    printf("\n Received EDM mode command!!\n");
    if(_parser.scanf("%d", &dataMode) >0) {
        printf("\nATCmdParser: Data mode=%d\n", dataMode);
        sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
        switch(dataMode)
        {
            case 0:
                printf("\nATCmdParser: Command Mode request received\n");
                dataMode = AT_CMD_DATA_MODE;
                break;
            case 1:
                printf("\nATCmdParser: Data Mode request received\n");
                dataMode = AT_STD_DATA_MODE;
                break;
            case 2:
                printf("\nATCmdParser: Extended data Mode request received\n");
                dataMode = AT_EXT_DATA_MODE;
                _event_queue.call_in(10, &print_heap_and_isr_stack_info);
                //print_heap_and_isr_stack_info();
                break;
            default:
                printf("\nATCmdParser: ERROR - UNKNOWN DATA MODE RECEIVED!!! \n");
                break;
        }
    } else {
        printf("\nATCmdParser: Retrieving Uart Options failed\n");
    }
    //_smutex.unlock();
}

void ATCmdManager::_oob_get_mac_addr(){
    int bleOrWifi;
    //_smutex.lock();
    if(_parser.scanf("=%d", &bleOrWifi) >0) {
        switch(bleOrWifi)
        {
            case 1:
                printf("\nATCmdParser: BLE MAC Address request received");
                break;
            case 2:
                printf("\nATCmdParser: WiFi MAC Address request received");
                break;
            default:
                printf("\nATCmdParser: ERROR - UNKNOWN MAC ADDRESS REQUEST RECEIVED!!! \n");
                break;
        }
    } else {
        printf("\nATCmdParser: Retrieving Uart Options failed");
    }
    sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
    //_smutex.unlock();
}

void ATCmdManager::_oob_get_ble_role(){
    //_smutex.lock();
    printf("\n Received get BLE role command!!\n");
    sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
    //_smutex.unlock();
}

void ATCmdManager::_oob_ena_ble_peri(){
    //_smutex.lock();
    printf("\n Received enable BLE Peripheral command!!\n");
    sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
    //_smutex.unlock();
}

void ATCmdManager::_oob_reboot(){
    //_smutex.lock();
    printf("\n Received reboot command!!\n");
    sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
    _parser.send("System Resetting....\n");
    system_reset();
    //_smutex.unlock();
}

const char * ATCmdManager::sec2str(nsapi_security_t sec)
{
    switch (sec) {
        case NSAPI_SECURITY_NONE:
            return "None";
        case NSAPI_SECURITY_WEP:
            return "WEP";
        case NSAPI_SECURITY_WPA:
            return "WPA";
        case NSAPI_SECURITY_WPA2:
            return "WPA2";
        case NSAPI_SECURITY_WPA_WPA2:
            return "WPA/WPA2";
        case NSAPI_SECURITY_UNKNOWN:
        default:
            return "Unknown";
    }
}

bool ATCmdManager::setNextResponse(at_cmd_resp_t resp)
{
     if(at_resp == AT_RESP_NONE){
        at_resp = resp;
        return true; // success
    }
    return false; // wiFiManager busy
}


void ATCmdManager::_oob_scanWiFiNetworks(){
    //_smutex.lock();
    printf("\n Received scanWiFiNetworks command!!\n");
    sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
    //_smutex.unlock();
    wifi_cmd_t cmd = WIFI_CMD_SCAN;
    // queue next command
    queueWiFiCommand(cmd);
    return;    
}



void ATCmdManager::_oob_WiFiStationConfigAction()
{
    int if_id; // interface id for request
    int aId; // interface id for request
    _parser.recv("%d,%d", &if_id, &aId);
    //_smutex.lock();
    printf("\n Received WiFi Configuration Action command %d %d!!\r\n", if_id, aId);
    sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
    if(if_id == WIFI_CONFIG_ID && aId <5){
        wifi_cmd_t cmd;
        action_id_t action_id = (action_id_t) aId;
        switch(action_id){
            case WIFI_CONFIG_RESET:
                break;
            case WIFI_CONFIG_STORE:
                break;
            case WIFI_CONFIG_LOAD:
                break;
            case WIFI_CONFIG_ACTIVATE:
                cmd = WIFI_CMD_CONNECT;
                printf("\n About to Queue wifi cmd = %d!!\n", cmd);
                //_smutex.unlock();
                // queue next command
                queueWiFiCommand(cmd);
                break;
            case WIFI_CONFIG_DEACTIVATE:
                cmd = WIFI_CMD_DISCONNECT;
                printf("\n About to Queue wifi cmd = %d!!\n", cmd);
                //_smutex.unlock();
                // queue next command
                queueWiFiCommand(cmd);
                break;
            default:
                break;
        }
    }
    if(debug_flag>0){
        printf("finished queuing WIFI CONNECT CMD \r\n");
    }
    return;
}


void ATCmdManager::_oob_disconnectWiFiNetwork()
{
    //_smutex.lock();
    printf("\n Received WiFi Disconnect command!!\n");
    sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
    wifi_cmd_t cmd = WIFI_CMD_DISCONNECT;
    //_smutex.unlock();
    // queue next command
    queueWiFiCommand(cmd);
    return;
}

void ATCmdManager::_oob_setupInternetConnection()
{
    char str[MAX_URL_LEN];
    char url[MAX_URL_LEN];
    int n;
    internet_config_t internet_config;
    //_smutex.lock();
    printf("sizeof internet_config_t = %d\n", sizeof(internet_config_t));
    n = ReadBytes((uint8_t *)str, MAX_URL_LEN);
    str[n]=NULL;
    printf("\n read string = %s , n = %d\n", str, n);
    //n = sscanf(str, "=%1d,%199[^,],%1d", &(uint8_t)internet_config.peer_id, 
    n = sscanf(str, "=%1d,%99[^,],%1d", &internet_config.peer_id, 
                                         url, //internet_config.url,    
                                        &internet_config.connectionScheme);
    char *p = strstr(url,"\""); 
    if(p!=NULL)
    {
        strncpy(internet_config.url, &p[1], strlen(url));
        p = strstr(internet_config.url,"\"");
        *p = NULL;
    }
    else
    {
        strncpy(internet_config.url, url, strlen(url)+1);
    }
    printf("\n read string = %s , n = %d -- strlen(url) = %d\n", internet_config.url, n, strlen(internet_config.url));
    if(n>0) 
    {
        printf("peer_id = %1d, url = %s, connScheme = %1d\n", internet_config.peer_id, 
                                                              internet_config.url, 
                                                              internet_config.connectionScheme);
        // package and send on wifi data queue
        wifi_data_msg_t data_msg; 
        data_msg.wifi_cmd = WIFI_CMD_INTERNET_CONFIG;
        data_msg.dataLen = sizeof(internet_config_t); // + strlen(internet_config.url);
        printf("\n url size = %d url str = %s\n",strlen(internet_config.url), internet_config.url );
        memcpy(data_msg.buffer,&internet_config, data_msg.dataLen);
        sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
        // queue next data request
        queueWiFiDataRequest(data_msg);
        print_memory_info();
    } else {
        printf("\n[ATCMD MAN]: internet configuration failed %d fields parsed \r\n", n);
        _parser.send("NAK\n");
    }
    //_smutex.unlock();
}

wifi_config_t ATCmdManager::init_wifi_config()
{ 
    wifi_config_t wifi_cfg;
    wifi_cfg.ssid[0]  = NULL;
    wifi_cfg.pass[0]  = NULL;
    wifi_cfg.security = NSAPI_SECURITY_UNKNOWN;
    return wifi_cfg;    
}
/*  read ASCII characters into buffer and null terminate */
int ATCmdManager::readStringBytes(uint8_t *buf, int maxBytes)
{ 
    int c;
    int sptr = 0;
    int quoteCnt = 0;
    for(int i=0;i<maxBytes;i++){
        c = _parser.getc();
        if(c== '"')quoteCnt++;
        if(c==-1 || quoteCnt==2){
            buf[sptr] = NULL; // null terminate if string
            return i;
        }
        if(c != ',' && c!= '"'){
            buf[sptr++] = (uint8_t) c;
        }
    }
    return maxBytes;
}

int ATCmdManager::ReadBytes(uint8_t *buf, int maxBytes)
{ 
    int c;
    int sptr = 0;
    for(int i=0;i<maxBytes;i++){
        c = _parser.getc();
        if(c==-1){
            return i;
        }
        buf[sptr++] = (uint8_t) c;
    }
    return maxBytes;
}

void ATCmdManager::_oob_setWiFiSSID()
{
    int n;
    wifi_config_t wifi_cfg = init_wifi_config();
    //_smutex.lock();
    n = readStringBytes((uint8_t *)wifi_cfg.ssid, 32);
    printf("[ATCMD MAN]: number of bytes read = %d\n", n);
    if(n>0) 
    {
        printf("[ATCMD MAN]: wifi_cfg.ssid = %s\n", wifi_cfg.ssid);
        // package and send on wifi data queue
        wifi_data_msg_t data_msg;
        data_msg.wifi_cmd = WIFI_CMD_CONFIG;
        data_msg.dataLen = sizeof(wifi_config_t);
        memcpy(data_msg.buffer,&wifi_cfg, data_msg.dataLen);
        sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
        queueWiFiDataRequest(data_msg);
    } else {
        printf("\n[ATCMD MAN]: wifi configuration failed  \n");
        _parser.send("NAK\n");
    }
    //_smutex.unlock();
}

void ATCmdManager::_oob_setWiFiPWD()
{
    int n;
    wifi_config_t wifi_cfg = init_wifi_config();
    //_smutex.lock();
    //n = _parser.scanf("%31[^\r\n]", wifi_cfg.pass);
    n = readStringBytes((uint8_t *)wifi_cfg.pass, 32);
    if(n>0) 
    {
        printf("ATCMD MAN]: wifi_cfg.pass = %s\n", "****************");
        // package and send on wifi data queue
        wifi_data_msg_t data_msg;
        data_msg.wifi_cmd = WIFI_CMD_CONFIG;
        data_msg.dataLen = sizeof(wifi_config_t);
        memcpy(data_msg.buffer,&wifi_cfg, data_msg.dataLen);
        sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
        queueWiFiDataRequest(data_msg);
    } else {
        printf("\n[ATCMD MAN]: wifi configuration failed  \n");
        _parser.send("NAK\n");
    }
    //_smutex.unlock();
}

void ATCmdManager::_oob_setWiFiSecurity()
{
    int n;
    wifi_config_t wifi_cfg = init_wifi_config();
    _smutex.lock();
    n = _parser.scanf(",%d", &wifi_cfg.security);
    _smutex.unlock();
    if(n>0) 
    {
        printf("ATCMD MAN]: wifi_cfg.security = %s\n", sec2str(wifi_cfg.security));
        // package and send on wifi data queue
        wifi_data_msg_t data_msg;
        data_msg.wifi_cmd = WIFI_CMD_CONFIG;
        data_msg.dataLen = sizeof(wifi_config_t);
        memcpy(data_msg.buffer,&wifi_cfg, data_msg.dataLen);
        sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
        queueWiFiDataRequest(data_msg);
    } else {
        printf("\n[ATCMD MAN]: wifi configuration failed  \n");
        _smutex.lock();
        _parser.send("NAK\n");
        _smutex.unlock();
    }
    //_smutex.unlock();
}



void ATCmdManager::_oob_getNetworkStatus()
{
    uint8_t if_id; // interface id for request
    _parser.scanf(",%d", &if_id);
    //_smutex.lock();
    printf("\n Received Get Network Status command!!\n");
    sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
    if(if_id == WIFI_INTERFACE_ID){
        wifi_cmd_t cmd = WIFI_CMD_NETWORK_STATUS;
        printf("\n About to Queue wifi cmd = %d!!\n", cmd);
        //_smutex.unlock();
        // queue next command
        queueWiFiCommand(cmd);
    }
    return;
}


void ATCmdManager::_oob_WiFiNetworkStatus()
{
    //_smutex.lock();
    printf("\n Received Get WiFi Network Status command!!\n");
    sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
    wifi_cmd_t cmd = WIFI_CMD_WIFI_STATUS;
    printf("\n About to Queue wifi cmd = %d!!\n", cmd);
    //_smutex.unlock();
    // queue next command
    queueWiFiCommand(cmd);
    return;
}

void ATCmdManager::_oob_sendHttpMessage()
{
    int n;
    http_request_t http_post_request;
    //_smutex.lock();
    /*
    n = _parser.scanf(",%d", &wifi_cfg.security);
    if(n>0) 
    {
        printf("ATCMD MAN]: wifi_cfg.security = %s\n", sec2str(wifi_cfg.security));
        // package and send on wifi data queue
        wifi_data_msg_t data_msg;
        data_msg.wifi_cmd = WIFI_CMD_CONFIG;
        data_msg.dataLen = sizeof(wifi_config_t);
        memcpy(data_msg.buffer,&wifi_cfg, data_msg.dataLen);
        queueWiFiDataRequest(data_msg);
        sendAtConfirmation(OK_RESP);   //_parser.send(OK_RESP);
    } else {
        printf("\n[ATCMD MAN]: wifi configuration failed  \n");
        _parser.send("NAK\n");
    }
    */
    //_smutex.unlock();

}

bool  ATCmdManager::queueWiFiCommand(wifi_cmd_t cmd){
    printf("[ATCMD MAN] about to be queued with wifi_cmd = %d\n", cmd);
#ifndef USE_MALLOC_FOR_COMMAND_MEMORY_POOL
    wifi_cmd_message_t *wifiCmd = _aT2WiFimPool->alloc();
    if(wifiCmd == NULL){
        printf("[ATCMD MAN] queued memory allocation failed\n");
        return false;
    }
#else
    wifi_cmd_message_t *wifiCmd = (wifi_cmd_message_t *) malloc(sizeof(wifi_cmd_message_t));
    if(wifiCmd == NULL){
        printf("[ATCMD MAN] try malloc() : queued memory allocation failed\n");
        //_event_queue.call_in(10, &print_memory_info);
        _event_queue.call_in(10, &print_heap_and_isr_stack_info);
        //print_memory_info();
        //print_heap_and_isr_stack_info();
        return false;
    }
#endif
    wifiCmd->wifi_cmd            = cmd;
    _aT2WiFiCmdQueue->put(wifiCmd);
    printf("[ATCMD MAN] queued wifi_cmd = %d\n", wifiCmd->wifi_cmd);
    if((int)cmd == 5)
    debug_flag = 3;
    return true;
}

bool  ATCmdManager::dequeueATresponse(){
    if(debug_flag>0){
        debug_flag--;
        printf("Called dequeueATresponse after WIFI CONNECT CMD \r\n");
    }
    if(at_resp != AT_RESP_NONE) return false; // busy
    osEvent evt = _wiFi2ATCmdQueue->get(0);
    if(evt.status == osEventMessage){
        at_resp_message_t *resp = (at_resp_message_t*)evt.value.p;
        setNextResponse(resp->at_resp);
#ifndef USE_MALLOC_FOR_COMMAND_MEMORY_POOL
        _wiFi2ATmPool->free(resp);
        resp = NULL;
#else
        free(resp);
        resp = NULL;
#endif
    }
    return true;
}

bool  ATCmdManager::queueWiFiDataRequest(wifi_data_msg_t data_req){
    //print_memory_info();
    _event_queue.call_in(10, &print_heap_and_isr_stack_info);
    //print_heap_and_isr_stack_info();
    wifi_data_msg_t *wifiData = _aT2WiFiDatamPool->alloc();
    wifiData->wifi_cmd        = data_req.wifi_cmd;
    wifiData->dataLen        = data_req.dataLen;
    memcpy(wifiData->buffer, data_req.buffer, data_req.dataLen);
    _aT2WiFiDataQueue->put(wifiData);
    printf("[ATCMD MAN] queued data size = %d : wifi_cmd = %d\n", data_req.dataLen, data_req.wifi_cmd);
    //free(&data_req);
    return true;
}

bool  ATCmdManager::dequeueWiFidataResponse(){
    if(debug_flag>0){
        debug_flag--;
        printf("Called dequeueWiFidataResponse after WIFI CONNECT CMD \r\n");
    }
    if(at_resp != AT_RESP_NONE) return false; // busy
    osEvent evt = _wiFi2ATDataQueue->get(0);
    if(evt.status == osEventMessage){
        resp_data = (at_data_msg_t*)evt.value.p;
        setNextResponse(resp_data->at_resp);
        _event_queue.call_in(10, &print_heap_and_isr_stack_info);
        //print_heap_and_isr_stack_info();
    }
    return true;
}


void ATCmdManager::sendAtConfirmation(const char *buf)
{
    _smutex.lock();
    switch(dataMode){
        case AT_CMD_DATA_MODE:
            _parser.send("%s", buf);
            break;
        case AT_STD_DATA_MODE:
            _parser.send("%s", buf);
            break;
        case AT_EXT_DATA_MODE:
        {
            int len =  strlen(buf);
            outputEDMdata((const uint8_t *) buf, len, AT_MSG_ID, CONFIRMATION_MSG_TYPE, NO_CHANNEL);
            break;
        }
        default:
            _parser.send("%s", buf);
            break;
    }
    //wait_ms(1);
    _smutex.unlock();
}
void ATCmdManager::sendAtEvent(const char *buf)
{
    _smutex.lock();
    switch(dataMode){
        case AT_CMD_DATA_MODE:
            _parser.send(buf);
            break;
        case AT_STD_DATA_MODE:
            _parser.send(buf);
            break;
        case AT_EXT_DATA_MODE:
            int len =  strlen(buf);
            outputEDMdata((const uint8_t *) buf, len, AT_MSG_ID, EVENT_MSG_TYPE, NO_CHANNEL);
            break;
        //default:
            //_parser.send(buf);
            //break;
    }
    _smutex.unlock();
}

void ATCmdManager::outputEDMdata(const uint8_t *buf, int pLen, 
                                 edm_msg_id_t identifier, edm_msg_type_t type,
                                 channel_id_t channel_id)
{
    int epLen = pLen + 2; // edm payload length = data length + 2
    if(channel_id != NO_CHANNEL)
        epLen += 1;
    _smutex.lock();
    // send EDM Message start byte
    _parser.putc(EDM_START_BYTE);
    // send EDM Message length
    _parser.putc(epLen>>8);
    _parser.putc(epLen%256);
    // send EDM Identifier + Type
    _parser.putc(identifier>>8);
    _parser.putc(identifier%256 | type);
    // send channel id if valid
    if(channel_id != NO_CHANNEL)
        _parser.putc(channel_id);
    // send the data
    _parser.write((const char *)buf, pLen);
    // send EDM Message stop byte
    _parser.putc(EDM_STOP_BYTE);
    _smutex.unlock();
    int msWait = (pLen + 5+20)/20;
    wait_ms(msWait);
}

void ATCmdManager::return_response(bool download) {
    char * resp = (char *) resp_data->buffer;
    //_smutex.lock();
    printf("\n[ATCMD MAN] received response:\n");
    if(download == false) // not download must be ascii header
    {
        printf("%.*s\r\n", resp_data->dataLen, resp);
    }
    else // dump payload as hex
    {
        printBufferInHex((uint8_t *)resp, resp_data->dataLen);
    }
    //_parser.write(resp, resp_data->dataLen);
    outputEDMdata((const uint8_t *)resp, resp_data->dataLen, DATA_MSG_ID, 
                  EVENT_MSG_TYPE, WIFI_CHANNEL);
    //_smutex.unlock();
    _wiFi2ATDatamPool->free(resp_data);
    resp_data = NULL;
    //print_memory_info();
    _event_queue.call_in(10, &print_heap_and_isr_stack_info);
    //print_heap_and_isr_stack_info();   
}
