#include "ATCmdManager.h"
#include "common_config.h"
#include "common_types.h"


ATCmdManager::ATCmdManager(PinName tx, PinName rx, SMDevicePeripheral *blePeripheral, 
                           events::EventQueue &event_queue, WiFiManager *wifi, 
                           MemoryPool<wifi_cmd_message_t, 16> *aT2WiFimPool, 
                           Queue<wifi_cmd_message_t, 16>      *aT2WiFiCmdQueue, 
                           MemoryPool<at_resp_message_t, 16>  *wiFi2ATmPool, 
                           Queue<at_resp_message_t, 16>       *wiFi2ATCmdQueue, 
                           MemoryPool<wifi_data_msg_t, 4>     *aT2WiFiDatamPool, 
                           Queue<wifi_data_msg_t, 4>          *aT2WiFiDataQueue, 
                           MemoryPool<at_data_msg_t, 4>       *wiFi2ATDatamPool, 
                           Queue<at_data_msg_t, 4>            *wiFi2ATDataQueue, 
                           bool debug)
    :
    _serial(tx, rx, DEFAULT_BAUD_RATE),
    blePeripheral(blePeripheral),
    _event_queue(event_queue),
    wiFiManager(wiFiManager),
    _aT2WiFimPool(aT2WiFimPool),
    _aT2WiFiCmdQueue(aT2WiFiCmdQueue),
    
    _wiFi2ATmPool(wiFi2ATmPool),
    _wiFi2ATCmdQueue(wiFi2ATCmdQueue),
    
    _aT2WiFiDatamPool(aT2WiFiDatamPool),
    _aT2WiFiDataQueue(aT2WiFiDataQueue),
    
    _wiFi2ATDatamPool(wiFi2ATDatamPool),
    _wiFi2ATDataQueue(wiFi2ATDataQueue),
    
    _parser(&_serial)
    

{ 
// constructor
    _serial.set_baud(DEFAULT_BAUD_RATE);
    _parser.debug_on(debug);
    _parser.set_delimiter("\r\n");
    _parser.send("+STARTUP");
    _parser.oob("AT\n", callback(this, &ATCmdManager::_oob_ok_hdlr));
    _parser.oob("ATE0", callback(this, &ATCmdManager::_oob_echo_off));
    _parser.oob("ATE1", callback(this, &ATCmdManager::_oob_echo_on));
    _parser.oob("AT+UMRS", callback(this, &ATCmdManager::_oob_uart_setup));
    
    _parser.oob("ATO", callback(this, &ATCmdManager::_oob_data_mode));
    _parser.oob("AT+UMLA", callback(this, &ATCmdManager::_oob_get_mac_addr));
    _parser.oob("AT+UBTLE?", callback(this, &ATCmdManager::_oob_get_ble_role));
    _parser.oob("AT+UBTLE=2", callback(this, &ATCmdManager::_oob_ena_ble_peri));
    _parser.oob("AT+CPWROFF", callback(this, &ATCmdManager::_oob_reboot));
    _parser.oob("AT+CGMR", callback(this, &ATCmdManager::_oob_get_fw_ver));
    _parser.oob("AT+UWSCAN", callback(this, &ATCmdManager::_oob_scanWiFiNetworks));
    _parser.oob("AT+UWSCA=", callback(this, &ATCmdManager::_oob_connect2WiFiNetwork));
    _parser.oob("AT+UWSCD=", callback(this, &ATCmdManager::_oob_disconnectWiFiNetwork));
    _parser.oob("AT+UDDRP", callback(this, &ATCmdManager::_oob_setupInternetConnection));
    _parser.oob("AT+UWSC=0,2", callback(this, &ATCmdManager::_oob_setWiFiSSID));
    _parser.oob("AT+UWSC=0,8", callback(this, &ATCmdManager::_oob_setWiFiPWD));
    _parser.oob("AT+UWSC=0,5", callback(this, &ATCmdManager::_oob_setWiFiSecurity));
 
//AT+UWSC=0,2,<SSID>
//AT+UWSC=0,8,<PWD>
//AT+UWSC=0,5,2 (WPA)
    //_parser.oob("ATE0", callback(this, &ATCmdManager::_oob_startup_hdlr));
    printf("\n --- ATCmdManager constructor completed ---\n");
    at_resp = AT_RESP_NONE;
    dataMode = AT_CMD_DATA_MODE;
    //AT+UMRS=230400,2,8,1,1,1
    // AT+UBTLE
}


void ATCmdManager::runMain(){
    while(true){
        _process_oob(UBLOX_ODIN_W2_RECV_TIMEOUT, true);
        wait_ms(MAIN_LOOP_WAIT_TIME_MS); // allow BTLE/WiFi some time 
        processResponses();
    }
    
}

void ATCmdManager::processResponses(){
        dequeueATresponse();
        switch(at_resp){
            case AT_RESP_NONE:
                // IDLE response state
                break;
            case AT_SCAN_RESP:
                // AT_SCAN_RESP response state
                _smutex.lock();
                printf("\n [ATCMD MAN] WIFI SCAN RESPONSE RECEIVED!!\n");
                _parser.send("OK\n");
                _smutex.unlock();
                at_resp = AT_RESP_NONE;
                break;
            case AT_DETAILED_SCAN_RESP:
                // AT_DETAILED_SCAN_RESP response state
                _smutex.lock();
                printf("\n [ATCMD MAN] WIFI DETAILED SCAN RESPONSE RECEIVED!!\n");
                _parser.send("OK\n");
                _smutex.unlock();
                at_resp = AT_RESP_NONE;
                break;
            case AT_CONNECT_RESP:  
                // AT_CONNECT_RESP response state 
                _smutex.lock();
                printf("\n [ATCMD MAN] WIFI CONNECT RESPONSE RECEIVED!!\n");
                _parser.send("OK\n");
                _smutex.unlock();
                at_resp = AT_RESP_NONE;
                break;
            case AT_DISCONNECT_RESP:  
                // AT_DISCONNECT_RESP response state 
                _smutex.lock();
                printf("\n [ATCMD MAN] WIFI DISCONNECT RESPONSE RECEIVED!!\n");
                _parser.send("OK\n");
                _smutex.unlock();
                at_resp = AT_RESP_NONE;
                break;
            case AT_CONFIG_RESP:  
                // AT_CONFIG_RESP response state 
                _smutex.lock();
                printf("\n [ATCMD MAN] WIFI CONFIG RESPONSE RECEIVED!!\n");
                _parser.send("OK\n");
                _smutex.unlock();
                at_resp = AT_RESP_NONE;
                break;
            case AT_HTTP_RESP:  
                // AT_HTTP_RESP response state 
                _smutex.lock();
                printf("\n [ATCMD MAN] WIFI HTTP RESPONSE RECEIVED!!\n");
                _parser.send("OK\n");
                _smutex.unlock();
                at_resp = AT_RESP_NONE;
                break;
            default:
                // UNKNOWN response state
                _smutex.lock();
                printf("\n [ATCMD MAN] UNKNOWN RESPONSE RECEIVED!!\n");
                _parser.send("OK\n");
                _smutex.unlock();
                at_resp = AT_RESP_NONE;
                break;
        }
}




// OOB processing
void ATCmdManager::_process_oob(uint32_t timeout, bool all){
    set_timeout(timeout);
    // Poll for inbound packets
    while (_parser.process_oob() && all) {
    }
    set_timeout();
}


// OOB message handlers
void ATCmdManager::_oob_startup_hdlr(){
}

void ATCmdManager::_oob_ok_hdlr(){
    _smutex.lock();
    _parser.send("OK\n");
    _smutex.unlock();
}


void ATCmdManager::_oob_bleRole_hdlr(){
}


void ATCmdManager::_oob_wifiMode_err(){
}


void ATCmdManager::_oob_conn_already(){
}


void ATCmdManager::_oob_err(){
}

void ATCmdManager::_oob_get_fw_ver()
{
}


void ATCmdManager::_oob_uart_setup(){
    int uOpts[NUM_UART_OPTIONS];
    //if(_parser.recv("=%d,%d,%d,%d,%d,%d", &uOpts[0], &uOpts[1], &uOpts[2], &uOpts[3], &uOpts[4], &uOpts[5])) {
    if(_parser.scanf("=%d,%d,%d,%d,%d,%d", &uOpts[0], &uOpts[1], &uOpts[2], &uOpts[3], &uOpts[4], &uOpts[5]) >0) {
        printf("\nATCmdParser: Uart Options=%d,%d,%d,%d,%d,%d\n", uOpts[0], uOpts[1], uOpts[2], uOpts[3], uOpts[4], uOpts[5]);
       //AT+UMRS=230400,2,8,1,1,1
        printf("\n Changing Baud Rate to %d\n", uOpts[0]);

        _serial.set_baud(uOpts[0]);
        printf("\n Baud Rate now %d\n", uOpts[0]);

    } else {
        printf("\nATCmdParser: Retrieving Uart Options failed");
    }
}

void ATCmdManager::set_timeout(uint32_t timeout_ms)
{
    _parser.set_timeout(timeout_ms);
}


void ATCmdManager::_oob_echo_off()
{
    _smutex.lock();
    printf("\n Received ATEO OOB command!!\n");
    printf("\n turning echo OFF!!\n");
    _parser.debug_on(false);
    _parser.send("OK\n");
    _smutex.unlock();
}


void ATCmdManager::_oob_echo_on()
{
    _smutex.lock();
    printf("\n Received ATE1 OOB command!!\n");
    printf("\n turning echo ON!!\n");
    _parser.debug_on(true);
    _parser.send("OK\n");
    _smutex.unlock();
}


void ATCmdManager::_oob_data_mode(){
    int dataMode;
    _smutex.lock();
    printf("\n Received EDM mode command!!\n");
    if(_parser.scanf("%d", &dataMode) >0) {
        printf("\nATCmdParser: Data mode=%d\n", dataMode);
        switch(dataMode)
        {
            case 0:
                printf("\nATCmdParser: Command Mode request received");
                dataMode = AT_CMD_DATA_MODE;
                break;
            case 1:
                printf("\nATCmdParser: Data Mode request received");
                dataMode = AT_STD_DATA_MODE;
                break;
            case 2:
                printf("\nATCmdParser: Extended data Mode request received");
                dataMode = AT_EXT_DATA_MODE;
                break;
            default:
                printf("\nATCmdParser: ERROR - UNKNOWN DATA MODE RECEIVED!!! \n");
                break;
        }
    } else {
        printf("\nATCmdParser: Retrieving Uart Options failed");
    }
    _parser.send("OK\n");
    _smutex.unlock();
}

void ATCmdManager::_oob_get_mac_addr(){
    int bleOrWifi;
    _smutex.lock();
    if(_parser.scanf("=%d", &bleOrWifi) >0) {
        switch(bleOrWifi)
        {
            case 1:
                printf("\nATCmdParser: BLE MAC Address request received");
                break;
            case 2:
                printf("\nATCmdParser: WiFi MAC Address request received");
                break;
            default:
                printf("\nATCmdParser: ERROR - UNKNOWN MAC ADDRESS REQUEST RECEIVED!!! \n");
                break;
        }
    } else {
        printf("\nATCmdParser: Retrieving Uart Options failed");
    }
    _parser.send("OK\n");
    _smutex.unlock();
}

void ATCmdManager::_oob_get_ble_role(){
    _smutex.lock();
    printf("\n Received get BLE role command!!\n");
    _parser.send("OK\n");
    _smutex.unlock();
}

void ATCmdManager::_oob_ena_ble_peri(){
    _smutex.lock();
    printf("\n Received enable BLE Peripheral command!!\n");
    _parser.send("OK\n");
    _smutex.unlock();
}

void ATCmdManager::_oob_reboot(){
    _smutex.lock();
    printf("\n Received reboot command!!\n");
    _parser.send("OK\n");
    _parser.send("System Resetting....\n");
    system_reset();
    _smutex.unlock();
}

const char * ATCmdManager::sec2str(nsapi_security_t sec)
{
    switch (sec) {
        case NSAPI_SECURITY_NONE:
            return "None";
        case NSAPI_SECURITY_WEP:
            return "WEP";
        case NSAPI_SECURITY_WPA:
            return "WPA";
        case NSAPI_SECURITY_WPA2:
            return "WPA2";
        case NSAPI_SECURITY_WPA_WPA2:
            return "WPA/WPA2";
        case NSAPI_SECURITY_UNKNOWN:
        default:
            return "Unknown";
    }
}

bool ATCmdManager::setNextResponse(at_cmd_resp_t resp)
{
     if(at_resp == AT_RESP_NONE){
        at_resp = resp;
        return true; // success
    }
    return false; // wiFiManager busy
}


void ATCmdManager::_oob_scanWiFiNetworks(){
    _smutex.lock();
    printf("\n Received scanWiFiNetworks command!!\n");
    _parser.send("OK\n");
    _smutex.unlock();
    wifi_cmd_t cmd = WIFI_CMD_SCAN;
    // queue next command
    queueWiFiCommand(cmd);
    return;    
}

void ATCmdManager::_oob_connect2WiFiNetwork()
{
    wifi_cmd_t cmd = WIFI_CMD_CONNECT;
    // queue next command
    queueWiFiCommand(cmd);
    return;
}


void ATCmdManager::_oob_disconnectWiFiNetwork()
{
    wifi_cmd_t cmd = WIFI_CMD_DISCONNECT;
    // queue next command
    queueWiFiCommand(cmd);
    return;
}

void ATCmdManager::_oob_setupInternetConnection()
{
    char url[200];
    int n;
    internet_config_t internet_config;
    _smutex.lock();
    n = _parser.scanf("=%1d,%199[^,],%1d", &internet_config.peer_id, 
                                      url,
                                     &internet_config.connectionScheme);
    if(n>0) 
    {
        internet_config.url = url;
        printf("peer_id = %1d, url = %s, connScheme = %1d\n", internet_config.peer_id, 
                                                          internet_config.url.c_str(), 
                                                          internet_config.connectionScheme);
        // package and send on wifi data queue
        wifi_data_msg_t data_msg;
        data_msg.wifi_cmd = WIFI_CMD_INTERNET_CONFIG;
        data_msg.dataLen = sizeof(internet_config) + internet_config.url.length();
        memcpy(data_msg.buffer,&internet_config, data_msg.dataLen);
        // queue next data request
        queueWiFiDataRequest(data_msg);
        _parser.send("OK\n");
    } else {
        printf("\n[ATCMD MAN]: internet configuration failed %d fields parsed \n", n);
        _parser.send("NAK\n");
    }
    _smutex.unlock();
}

wifi_config_t ATCmdManager::init_wifi_config()
{ 
    wifi_config_t wifi_cfg;
    wifi_cfg.ssid[0]  = NULL;
    wifi_cfg.pass[0]  = NULL;
    wifi_cfg.security = NSAPI_SECURITY_UNKNOWN;
    return wifi_cfg;    
}

int ATCmdManager::readBytes(uint8_t *buf, int maxBytes)
{ 
    int c;
    int sptr = 0;
    for(int i=0;i<maxBytes;i++){
        c = _parser.getc();
        if(c==-1){
            buf[sptr] = '\0'; // null terminate if string
            return i;
        }
        if(c != ',' && c!= '"'){
            buf[sptr++] = (uint8_t) c;
        }
    }
}

void ATCmdManager::_oob_setWiFiSSID()
{
    int n;
    wifi_config_t wifi_cfg = init_wifi_config();
    _smutex.lock();
    n = readBytes((uint8_t *)wifi_cfg.ssid, 32);
    printf("[ATCMD MAN]: number of bytes read = %d\n", n);
    if(n>0) 
    {
        printf("[ATCMD MAN]: wifi_cfg.ssid = %s\n", wifi_cfg.ssid);
        // package and send on wifi data queue
        wifi_data_msg_t data_msg;
        data_msg.wifi_cmd = WIFI_CMD_CONFIG;
        data_msg.dataLen = sizeof(wifi_config_t);
        memcpy(data_msg.buffer,&wifi_cfg, data_msg.dataLen);
        queueWiFiDataRequest(data_msg);
        _parser.send("OK\n");
    } else {
        printf("\n[ATCMD MAN]: wifi configuration failed  \n");
        _parser.send("NAK\n");
    }
    _smutex.unlock();
}

void ATCmdManager::_oob_setWiFiPWD()
{
    int n;
    wifi_config_t wifi_cfg = init_wifi_config();
    _smutex.lock();
    //n = _parser.scanf("%31[^\r\n]", wifi_cfg.pass);
    n = readBytes((uint8_t *)wifi_cfg.pass, 32);
    if(n>0) 
    {
        printf("ATCMD MAN]: wifi_cfg.pass = %s\n", wifi_cfg.pass);
        // package and send on wifi data queue
        wifi_data_msg_t data_msg;
        data_msg.wifi_cmd = WIFI_CMD_CONFIG;
        data_msg.dataLen = sizeof(wifi_config_t);
        memcpy(data_msg.buffer,&wifi_cfg, data_msg.dataLen);
        queueWiFiDataRequest(data_msg);
        _parser.send("OK\n");
    } else {
        printf("\n[ATCMD MAN]: wifi configuration failed  \n");
        _parser.send("NAK\n");
    }
    _smutex.unlock();
}

void ATCmdManager::_oob_setWiFiSecurity()
{
    int n;
    wifi_config_t wifi_cfg = init_wifi_config();
    _smutex.lock();
    n = _parser.scanf(",%d", &wifi_cfg.security);
    if(n>0) 
    {
        printf("ATCMD MAN]: wifi_cfg.security = %s\n", sec2str(wifi_cfg.security));
        // package and send on wifi data queue
        wifi_data_msg_t data_msg;
        data_msg.wifi_cmd = WIFI_CMD_CONFIG;
        data_msg.dataLen = sizeof(wifi_config_t);
        memcpy(data_msg.buffer,&wifi_cfg, data_msg.dataLen);
        queueWiFiDataRequest(data_msg);
        _parser.send("OK\n");
    } else {
        printf("\n[ATCMD MAN]: wifi configuration failed  \n");
        _parser.send("NAK\n");
    }
    _smutex.unlock();
}

bool  ATCmdManager::queueWiFiCommand(wifi_cmd_t cmd){
    wifi_cmd_message_t *wifiCmd = _aT2WiFimPool->alloc();
    wifiCmd->wifi_cmd            = cmd;
    _aT2WiFiCmdQueue->put(wifiCmd);
    return true;
}

bool  ATCmdManager::dequeueATresponse(){
    if(at_resp != AT_RESP_NONE) return false; // busy
    osEvent evt = _wiFi2ATCmdQueue->get(0);
    if(evt.status == osEventMessage){
        at_resp_message_t *resp = (at_resp_message_t*)evt.value.p;
        setNextResponse(resp->at_resp);
        _wiFi2ATmPool->free(resp);
    }
    return true;
}

bool  ATCmdManager::queueWiFiDataRequest(wifi_data_msg_t data_req){
    wifi_data_msg_t *wifiData = _aT2WiFiDatamPool->alloc();
    wifiData->wifi_cmd        = data_req.wifi_cmd;
    wifiData->dataLen        = data_req.dataLen;
    memcpy(wifiData->buffer, data_req.buffer, data_req.dataLen);
    _aT2WiFiDataQueue->put(wifiData);
    printf("[ATCMD MAN] queued data size = %d : wifi_cmd = %d\n", data_req.dataLen, data_req.wifi_cmd);
    return true;
}

bool  ATCmdManager::dequeueATdataResponse(){
    if(at_resp != AT_RESP_NONE) return false; // busy
    osEvent evt = _wiFi2ATDataQueue->get(0);
    if(evt.status == osEventMessage){
        resp_data = (at_data_msg_t*)evt.value.p;
        setNextResponse(resp_data->at_resp);
        //_wiFi2ATDatamPool->free(resp_data);
    }
    return true;
}



