/*------------------------------------------------------------------------------------*/
/*  Ethernet UDP Server (to be used with Ethernet_UDP_client)                         */
/*------------------------------------------------------------------------------------*/

/*--COMPANY-----AUTHOR------DATE------------REVISION----NOTES-------------------------*/
/*  NXP         mareikeFSL  2015.12.23      rev 1.0     initial                       */
/*                                                                                    */
/*------------------------------------------------------------------------------------*/
/*  This "Hello World" program is used in conjunction with the Ethernet_UDP_client    */
/*  program. It communicates between two FRDM-K64F boards via the Ethernet protocol.  */
/*  To use this program, you need to do the following:                                */
/*      - Connect an Ethernet cable between two FRDM-K64F boards (a crossover cable   */
/*        is not required).                                                           */
/*      - Flash one board with Ethernet_UDP_client and the other with                 */
/*        Ethernet_UDP_server                                                         */
/*      - [optional] If you would like to see the "Hello World" output on your        */
/*        monitor, install and open a terminal. Tera Term is used in the Wiki for     */
/*        this program.                                                               */
/*------------------------------------------------------------------------------------*/


/*--INCLUDES----------------------------------------------------------------------------*/
#include "mbed.h"
#include "EthernetInterface.h"
#include "LM75B.h"
#include "C12832.h"
/*--DEFINES-----------------------------------------------------------------------------*/



/*--CONSTANTS---------------------------------------------------------------------------*/
const int PORT = 7;                             //arbitrary port

static const char* SERVER_IP = "192.168.1.101"; //IP of server board
static const char* MASK = "255.255.255.0";      //mask
static const char* GATEWAY = "192.168.1.1";     //gateway
static const char* BOILER_IP = "192.168.1.102"; //Boiler device
static const char* MASTER_IP = "192.168.1.110"; //Master device


/*--INITS-------------------------------------------------------------------------------*/
Serial pc(USBTX, USBRX);        //create PC interface
EthernetInterface eth;          //create ethernet
UDPSocket server;               //create server
Endpoint client;                //create endpoint

DigitalOut red(LED_RED);        //debug led
DigitalOut green(LED_GREEN);    //debug led

LM75B sensor(D14,D15);          //temp sensor
C12832 lcd(D11, D13, D12, D7, D10);//LCD screen arduino notation

/*--VARIABLES---------------------------------------------------------------------------*/
int n;                  //size of received message
int idealtemp;     //ideal room temp
char counter[1] = {0};  //sample receive/send buffer


/*--FUNCTION DECLARATIONS---------------------------------------------------------------*/
void init_usb(void);    //initializes pc.printf if required
void init_eth(void);    //initializes Ethernet
void receive(void);     //receives packets
int main(void);         //main


/*--FUNCTION DEFINITIONS----------------------------------------------------------------*/

/*****************************************************************************INIT_USB***/
void init_usb(void)
{
    pc.baud(9600);    //baud
    
}   //end init_usb()

/*****************************************************************************INIT_ETH***/
void init_eth(void)
{
    eth.init(SERVER_IP, MASK, GATEWAY);                                         //set up IP
    eth.connect();                                                              //connect ethernet
    pc.printf("\nSERVER - Server IP Address is %s\r\n", eth.getIPAddress());    //get server IP address;
    
    server.bind(PORT);                                                          //bind server
        
}   //end init_eth()

/******************************************************************************RECEIVE***/
void receive(void)
{
    pc.printf("\nSERVER - Waiting for UDP packet...\r\n");                                      //wait for packet
    n = server.receiveFrom(client, counter, sizeof(counter));                                   //receive message from client
    counter[n] = '\0';                                                                          //add \0 to end of message

    if(client.get_address()==BOILER_IP)                                                         //if address is boiler device
    {
        if(sensor.read() < idealtemp)
        {
            counter[0] = 0;                                                                   //reset counter if temp is below ideal
            counter[1] = '/0';
        }                                                                                      
    }
    if(client.get_address()==MASTER_IP)                                                         //if address is boiler device
    {
        idealtemp = counter[0];                                                                 //new ideal temp is from master device
        counter[0] = (int)sensor.read();                                                        //send sensor temp to master                            
    }

    
    pc.printf("SERVER - Received '%i' from client %s\r\n", counter[0], client.get_address());   //print message and client
    pc.printf("SERVER - Sending '%i' back to client %s\r\n", counter[0], client.get_address()); //print sending back
    server.sendTo(client, counter, n);                                                          //send message

}   //end receive()

/*********************************************************************************MAIN***/
int main(void)
{
    int j=0;
    lcd.cls();
    lcd.locate(0,3);
    lcd.printf("mbed application shield!");
    
    red = 1;
    green = 0;      //server
    
    init_usb();     //initialize the PC interface
    init_eth();     //initialize the Ethernet connection

    while (true)    //repeat forever
    {
        lcd.locate(0,15);
        lcd.printf("Counting : %d",j);
        j++;
        wait(1.0);
        receive();  //wait for message
    }

}   //end main()
