/*
* Author: Nikolai Zimfer, 2020
*
* HARDWARE:
* Board FRDM-K64F                        - https://os.mbed.com/platforms/FRDM-K64F/
* Display 4.3" 480 x 272 (RVT43ULFNWC03) - https://riverdi.com/product/rvt43ulfnwc0x/
* Break Out Board 20                     - https://riverdi.com/product/break-out-board-20/
* Cable FFC, 0.5mm pitch, 20 pin, 150 mm - https://riverdi.com/product/ffc0520150/
*
* WIRING:
* -----------------------------------
* FRDM-K64F           Break Out Board
* -----------------------------------
* +3.3V           --- Pin 1 VDD, Pin 17 BLVDD
* GND             --- Pin 2 GND      
* D13 (PTD1) SCLK --- Pin 3 SPI SCLK 
* D12 (PTD3) MISO --- Pin 4 MISO     
* D11 (PTD2) MOSI --- PIN 5 MOSI     
* D10 (PTD0)      --- Pin 6 CS       
* D9  (PTC4)      --- Pin 7 INT      
* D8  (PTC12)     --- Pin 8 PD       
* -----------------------------------
*
* IMAGE:
* https://os.mbed.com/media/uploads/nz/demo-clock.jpg
*/

#include "mbed.h"
#include "FT800.h"

/*
FT800 TFT(MOSI,MISO, SCK, CS, INT, PD);   */
FT800 TFT( D11, D12, D13, D10, D9, D8);   // FRDM-K64F

int16_t ox = 240, oy = 135; // center and size of clock
int16_t i, x, y;
uint16_t hurs, mins, secs;
float rads;
const float pi = 3.141592;
char  textbuff[32];
char  timebuff[32];

int main()
{   
    //printf("Mbed OS %d.%d.%d.\n", MBED_MAJOR_VERSION, MBED_MINOR_VERSION, MBED_PATCH_VERSION);
    sprintf(textbuff, "Mbed OS %d.%d.%d.", MBED_MAJOR_VERSION, MBED_MINOR_VERSION, MBED_PATCH_VERSION);
    
    TFT.Init();
        
    set_time(1577869200);   // Set RTC time to 9:00:00, date to 01.01.2020
           
    while(1)
    {
        time_t seconds = time(NULL);        
        
        strftime(timebuff, 3,"%H", localtime(&seconds));
        hurs = atoi(timebuff);
        strftime(timebuff, 3,"%M", localtime(&seconds));
        mins = atoi(timebuff);
        strftime(timebuff, 3,"%S", localtime(&seconds));
        secs = atoi(timebuff); 
        
        TFT.DLstart(WHITE,100);     // Start Display List
        
        // Draw the Minute ticks, angle difference is 6 degree
        for (i = 0; i < 60; i++)
        {        
            rads = (pi * i*6) / 180;      // radians
            x = (int16_t)(ox + (129 * cos(rads)));
            y = (int16_t)(oy + (129 * sin(rads)));
            TFT.drawLine(ox, oy, x, y, BLACK, 1);
        }
        TFT.drawPoint(ox, oy, WHITE, 124);
        
        // Draw the 5 min ticks, 5*6=30 degree
        for (i = 0; i < 12; i++)
        {
            rads = (pi * i*30) / 180;     // radians
            x = (int16_t)(ox + (131 * cos(rads)));
            y = (int16_t)(oy + (131 * sin(rads)));
            TFT.drawLine(ox, oy, x, y, BLACK, 2);
        }
        TFT.drawPoint(ox, oy, WHITE, 121);
        
        TFT.drawNumber(ox + 103, oy, 31, BLACK, OPT_CENTER, 3);
        TFT.drawNumber(ox - 103, oy, 31, BLACK, OPT_CENTER, 9);
        TFT.drawNumber(ox, oy + 100, 31, BLACK, OPT_CENTER, 6);
        TFT.drawNumber(ox, oy - 100, 31, BLACK, OPT_CENTER, 12);
        
        // Draw hour hand   // часовая стрелка
        rads = (pi * (hurs*(-30) - mins*0.5)) / 180;
        x = (int16_t)(ox + (84 * sin(rads) * (-1)));
        y = (int16_t)(oy + (84 * cos(rads) * (-1)));
        TFT.drawLine(ox, oy, x, y, BLACK, 3);
        x = (int16_t)(ox + (20 * sin(rads)));
        y = (int16_t)(oy + (20 * cos(rads)));
        TFT.drawLine(ox, oy, x, y, BLACK, 3);
        
        // Draw minute hand // минутная стрелка
        rads = (pi * mins*(-6)) / 180;
        x = (int16_t)(ox + (114 * sin(rads) * (-1)));
        y = (int16_t)(oy + (115 * cos(rads) * (-1)));
        TFT.drawLine(ox, oy, x, y, BLACK, 3);
        x = (int16_t)(ox + (20 * sin(rads)));
        y = (int16_t)(oy + (20 * cos(rads)));
        TFT.drawLine(ox, oy, x, y, BLACK, 3);
        
        // Draw second hand // секундная стрелка 
        rads = (pi * secs*(-6)) / 180;
        x = (int16_t)(ox + (110 * sin(rads) * (-1)));
        y = (int16_t)(oy + (110 * cos(rads) * (-1)));
        TFT.drawLine(ox, oy, x, y, RED, 1);
        x = (int16_t)(ox + (20 * sin(rads)));
        y = (int16_t)(oy + (20 * cos(rads)));
        TFT.drawLine(ox, oy, x, y, RED, 1);
        TFT.drawPoint(ox, oy, BLACK, 6);
        TFT.drawPoint(ox, oy, WHITE, 4);
        
        // Draw Time & Date // Время и дата
        strftime(timebuff, 32, "%X", localtime(&seconds));      // Time
        TFT.drawText(240,170, 29, DARKBLUE, OPT_CENTERX, timebuff);
        
        strftime(timebuff, 32, "%A", localtime(&seconds));      // Weekday
        TFT.drawText(420,25, 27, BLACK, OPT_CENTERX, timebuff);
        
        strftime(timebuff, 32, "%d.%m.%Y", localtime(&seconds));// Date
        TFT.drawText(420,50, 27, BLACK, OPT_CENTERX, timebuff);
        
        TFT.drawText(10, 25, 27, BLACK, OPT_FLAT, textbuff);    // Mbed Version
        
        TFT.DLend();
        
        thread_sleep_for(1000);         // 1000 ms
    }
}