#ifndef MEM_H
#define MEM_H

#include "mbed.h"

#define MEM_START_ADDR      0
#define MEM_PAGE_OFFSET     (1 << 8)
#define MEM_SECTOR_OFFSET   (1 << 12)

// Chip Manufacturer Info
typedef struct {
    int manuf_id;
    int mem_type;
    int mem_size;
} FLASH_MEM_INFO;

// Supported Commands
struct FLASH_MEM_CMD {
    static const int WREN         = 0x06;  // Set Write Enable Latch
    static const int WRDI         = 0x04;  // Reset Write Enable Latch
    static const int RDSR         = 0x05;  // Read Status Register
    static const int WRSR         = 0x01;  // Write Status Register
    static const int READ         = 0x03;  // Read Data from Memory Array
    static const int FAST_READ    = 0x0B;  // Read Data from Memory Array (with dummy cycles)
    static const int PROGRAM      = 0x02;  // Program Data Into Memory Array
    static const int SECTOR_ERASE = 0x20;  // Erase 1 4KB Sector in Memory
    static const int BLOCK_ERASE  = 0x52;  // Erase 1 64KB Block in Memory
    static const int CHIP_ERASE   = 0x60;  // Erase Entire Memory Array
    static const int RDID         = 0x9F;  // Read Manufacturer and Product ID
};

// Memory Operations Status
enum MEM_STATUS {
    FAIL    = 0,
    SUCCESS,
    BUSY,
    READY,
};


class MEM_DEVICE
{
    SPI *dev;

public:
     /**
     * Constructor
     *
     * @param mosi MOSI pin for SPI
     * @param miso MISO pin for SPI
     * @param sck  clock pin for SPI
     * @param cs Chip Select pin
     */
     MEM_DEVICE(PinName mosi, PinName miso, PinName sck, PinName cs);
     
     
     /**
     * Destructor
     */
     ~MEM_DEVICE();
     
     /**
     * Set mode and number of bits to use
     *
     * @param databits number data bits
     * @param mode SPI mode to use (0, 1, 2, 3)
     */
     void format(int databits, int mode);
    
     /**
     * Read multiple consecutive bytes from memory device
     *
     * @param startAddr address where to start reading from
     * @param numOfBytes number of consecutive bytes to read
     * @param destBuffer buffer where to copy bytes to
     * @return status of the read operation (FAIL or SUCCESS)
     */
     MEM_STATUS read_bytes(int startAddr, int numOfBytes, char *destBuffer);

     /**
     * Write multiple consecutive bytes to memory device
     *
     * @param startAddr address where to start writing to
     * @param numOfBytes number of consecutive bytes to write
     * @param srcBuffer buffer where to write from
     * @return status of the write operation (FAIL or SUCCESS)
     */
     MEM_STATUS write_bytes(int startAddr, int numOfBytes, char *srcBuffer);
     
     /**
     * Clears the entire flash memory
     *
     * @return status of the clear operation (FAIL or SUCCESS)
     */
     MEM_STATUS erase();
    
    /**
     * Reads the status of the memory device
     *
     * @return status of the operation (FAIL, BUSY, READY)
     */
     MEM_STATUS read_status();
};

#endif